package gov.va.cpss.job;

import java.text.SimpleDateFormat;
import java.util.Date;

import org.apache.log4j.Logger;
import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;

/**
 * StepExecutionListener that logs the start time, end time and elapsed time in
 * milliseconds of a step.
 * 
 * @author Brad Pickle
 *
 */
public class LogExecutionTimeStepExecutionListener implements StepExecutionListener {

	public static final int MILLIS_IN_HOUR = 3600000;

	public static final int MILLIS_IN_MINUTE = 60000;

	public static final int MILLIS_IN_SECOND = 1000;

	protected static final Logger logger = Logger
			.getLogger(LogExecutionTimeStepExecutionListener.class.getCanonicalName());

	private long startTimeMillis;
	private long endTimeMillis;

	@Override
	public void beforeStep(StepExecution stepExecution) {
		captureStartTime(stepExecution);
	}

	@Override
	public ExitStatus afterStep(StepExecution stepExecution) {
		captureEndTime(stepExecution);
		logElapsedTime(stepExecution);
		return null;
	}

	/**
	 * Capture the start time and log the start time message for the step
	 * 
	 * @param stepExecution
	 */
	private void captureStartTime(StepExecution stepExecution) {
		startTimeMillis = System.currentTimeMillis();

		logTime(getMessage(stepExecution, "Start"), startTimeMillis);
	}

	/**
	 * Capture the end time and log the end time message for the step
	 * 
	 * @param stepExecution
	 */
	private void captureEndTime(StepExecution stepExecution) {
		endTimeMillis = System.currentTimeMillis();

		logTime(getMessage(stepExecution, "End"), endTimeMillis);
	}

	/**
	 * Log the elapsed time of the step
	 */
	private void logElapsedTime(StepExecution stepExecution) {
		logger.info(getMessage(stepExecution, "Elapsed Time") + ":" + getElapsedTimeString(startTimeMillis, endTimeMillis));
	}

	/**
	 * Build the log message for the given step and execution point (Start/End)
	 * 
	 * @param stepExecution
	 * @param executionPoint
	 * @return String log message
	 */
	private static String getMessage(StepExecution stepExecution, String executionPoint) {
		final StringBuilder s = new StringBuilder();

		s.append("Job:").append(stepExecution.getJobExecution().getJobInstance().getJobName());
		s.append(", Step:").append(stepExecution.getStepName());
//		s.append("  Summary:").append(stepExecution.getSummary());
		s.append("  ").append(executionPoint);

		return s.toString();
	}

	/**
	 * Log the given message and formatted time of the given time.
	 * 
	 * @param message
	 * @param timeMillis
	 */
	private static void logTime(String message, long timeMillis) {
		Date date = new Date(timeMillis);
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");

		logger.info(message + ":" + sdf.format(date));
	}

	/**
	 * Build a formatted String of the elapsed time in milliseconds between the
	 * given start and end times.
	 * 
	 * @param startTimeMillis
	 * @param endTimeMillis
	 * @return String of elapsed time in milliseconds
	 */
	private static String getElapsedTimeString(long startTimeMillis, long endTimeMillis) {
		long remainingMillis = (endTimeMillis > startTimeMillis) ? (endTimeMillis - startTimeMillis) : 0L;

		final long elapsedHours = (remainingMillis >= MILLIS_IN_HOUR) ? remainingMillis / MILLIS_IN_HOUR : 0L;
		remainingMillis -= elapsedHours * MILLIS_IN_HOUR;

		final long elapsedMinutes = (remainingMillis >= MILLIS_IN_MINUTE) ? remainingMillis / MILLIS_IN_MINUTE : 0L;
		remainingMillis -= elapsedMinutes * MILLIS_IN_MINUTE;

		final long elapsedSeconds = (remainingMillis >= MILLIS_IN_SECOND) ? remainingMillis / MILLIS_IN_SECOND : 0L;
		remainingMillis -= elapsedSeconds * MILLIS_IN_SECOND;

		String delim = "";
		final StringBuilder s = new StringBuilder();

		if (elapsedHours > 0) {
			s.append(delim).append(elapsedHours).append((elapsedHours > 1) ? " hours" : " hour");
			delim = ", ";
		}

		if (elapsedMinutes > 0) {
			s.append(delim).append(elapsedMinutes).append((elapsedMinutes > 1) ? " minutes" : " minute");
			delim = ", ";
		}

		if (elapsedSeconds > 0) {
			s.append(delim).append(elapsedSeconds).append((elapsedSeconds > 1) ? " seconds" : " second");
			delim = ", ";
		}

		s.append(delim).append(remainingMillis).append(" milliseconds");

		return s.toString();
	}

}
