package gov.va.cpss.job;

import static gov.va.cpss.job.CbssJobProcessingConstants.BATCH_RUN_ID_KEY;
import static gov.va.cpss.job.fps.FpsProcessingConstants.INPUT_DIRECTORY_KEY;
import static gov.va.cpss.job.fps.FpsProcessingConstants.INPUT_RESOURCE_KEY;
import static gov.va.cpss.job.fps.FpsProcessingConstants.RECEIVED_ID_KEY;

import java.io.File;
import java.util.List;

import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.JobParameters;
import org.springframework.batch.core.JobParametersBuilder;
import org.springframework.batch.core.job.flow.FlowJob;
import org.springframework.beans.factory.annotation.Autowired;

import gov.va.cpss.job.loadbill.LoadBillProcessingConstants;
import gov.va.cpss.model.bal.SiteBalRec;
import gov.va.cpss.service.LoadBillService;

public class LoadBillDataJob extends AbstractProcessFileListJob {

	@Autowired
	private FlowJob LoadBillDataJobBatch;
	
	@Autowired
	protected LoadBillService loadBillService;
	
	private String detailErrorMessage;

	@Autowired
	private String loadBillServerTargetDirectory;
	
	@Autowired
	private String loadBillServerArchiveTargetDirectory;
	
	@Autowired
	private String loadBillServerErrorTargetDirectory;
	
	@Override
	protected FlowJob getJob() {
		return LoadBillDataJobBatch;
	}
	
	@Override
	protected boolean processList(final int batchRunId, FlowJob job, List<String> fileL) {
		// This job should only process one file at a time.
		List<String> subList = fileL;
		if(fileL != null && fileL.size() > 1) {
			subList = fileL.subList(0, 1);
		}
		final boolean successful = super.processList(batchRunId, job, subList);
		
		setInfoMessage();
		
		return successful;
	}
	
	private void setInfoMessage() {
		StringBuffer strBuff = new StringBuffer();
		strBuff.append("\nTotal Files Received: ");
		strBuff.append(this.getFileCount()); // Read the total files received
		strBuff.append("\nTotal Files Processed: ");
		strBuff.append(this.getSuccessfulFileCount()); // Read the total files processed
		strBuff.append("\n");
		strBuff.append("\nUnprocessed Files: ");
		strBuff.append(getFileCount() - getSuccessfulFileCount()); // Read unprocessed files
		if(getFailedFileList() != null) {
			for(String fileName: getFailedFileList()) {
				strBuff.append("\n" + fileName);
			}
		}
		if(detailErrorMessage != null && !detailErrorMessage.isEmpty()) {
			strBuff.append("\n\nPatient Detail Errors: \n");
			strBuff.append(detailErrorMessage);
		}
		appendInfoMessage(strBuff.toString());
	}

	@Override
	public String getDataDirectory() {
		return loadBillServerTargetDirectory;
	}
	
	@Override
	public String getArchiveSubDirectory() {
		return loadBillServerArchiveTargetDirectory;
	}
	@Override
	public String getErrorSubDirectory(){
		return loadBillServerErrorTargetDirectory;
	}

	@Override
	public String getProcessListErrorMessage() {
		return "Unable to connect to the SFTP server to retrieve the bill data files.\n";
	}

	@Override
	protected boolean processFile(String file, int batchRunId, FlowJob job) {
		SiteBalRec siteBalRec = initializeJob(batchRunId);

		JobParameters parameters = getParameters(siteBalRec);

		JobExecution execution = executeJob(job, parameters);
		if(execution.getExecutionContext().containsKey(LoadBillProcessingConstants.DETECTED_DETAIL_ERRORS_KEY)) {
			String errorMessage = execution.getExecutionContext().getString(
					LoadBillProcessingConstants.DETECTED_DETAIL_ERRORS_KEY);
			if(detailErrorMessage == null || detailErrorMessage.isEmpty()) {
				detailErrorMessage = errorMessage;
			}
			else{
				detailErrorMessage += errorMessage;
			}
		}
		
		return loadBillService.endProcessLoadBillDataJob(execution, siteBalRec);
	}
	
	protected SiteBalRec initializeJob(final int batchRunId) {

		// Use the job service to start the job.
		return loadBillService.startProcessLoadBillDataJob(batchRunId, (new File(filename)).getName());
	}
	
	/**
	 * Build parameters for the batch job based on the SiteBalRec object.
	 * 
	 * @param siteBalRec
	 *            The SiteBalRec object representing the database entry.
	 * @return JobParameters object for this batch job run.
	 */
	protected JobParameters getParameters(SiteBalRec siteBalRec) {

		// Build job parameters for the input filename, received primary key id,
		// and the created or modified by string.
		return new JobParametersBuilder().addString(INPUT_RESOURCE_KEY, filename).addString(INPUT_DIRECTORY_KEY, getDataDirectory())
				.addLong(RECEIVED_ID_KEY, new Long(siteBalRec.getId())).addLong(BATCH_RUN_ID_KEY, new Long(siteBalRec.getBatchRunId())).toJobParameters();
	}
}