package gov.va.cpss.job;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.concurrent.BlockingQueue;

public class CbssStreamToQueueThreadReadLineImpl extends CbssStreamToQueueThread {

	/**
	 * Max value of bytes that represent a line feed.
	 */
	private static final int MAX_LINE_FEED_BYTE_COUNT = 2;

	/**
	 * Constructor for thread to process the provided input stream in to an
	 * output queue for each record as indicated by a line feed. The expected
	 * bytes is provided for verification check the data was successfully
	 * processed.
	 * 
	 * @param expectedByteCount
	 *            The expected amount of bytes to read.
	 * @param in
	 *            The input stream.
	 * @param outputQueue
	 *            The output queue.
	 * @throws UnsupportedEncodingException
	 */
	public CbssStreamToQueueThreadReadLineImpl(long expectedByteCount, InputStream in,
			BlockingQueue<String> outputQueue) throws UnsupportedEncodingException {
		super(expectedByteCount, in, outputQueue);
	}

	@Override
	public boolean populateQueue() {

		boolean successful = false;

		long lineCounter = 0;

		try {
			String line = bufferedReader.readLine();

			// Loop until thread is interrupted or no more lines available.
			while (!isInterrupted() && (line != null)) {
				++lineCounter;
				if (!line.isEmpty()) {
					outputQueue.put(line);
				}
				readByteCount += line.length();
				line = bufferedReader.readLine();
			}

			// Adjust the expected byte count to account for the number of line
			// feeds processed.
			for (int i = 1; i <= MAX_LINE_FEED_BYTE_COUNT; ++i) {
				final long adjustedReadByteCount = getAdjustedReadByteCount(i, lineCounter);

				// Adjustment to allow tolerance of an extra line feed.
				if (Math.abs(adjustedReadByteCount - expectedByteCount) <= i) {
					logger.info("Successfully processed file with line feed size: " + i);
					successful = true;
					break;
				}
			}

			if (!successful) {
				logger.error("Did not read expected number of bytes: " + expectedByteCount);
			}

		} catch (IOException e) {
			logger.info("The stream queue processing thread may have been interrupted");
		} catch (InterruptedException e) {
			logger.info("The stream queue processing thread was interrupted");
		}

		return successful;
	}

	/**
	 * Adjust the bytes read by accounting for the line feed characters which
	 * are not included when using BufferedReader getLine method.
	 * 
	 * @param lineCount
	 *            The number of lines read.
	 * @return The adjusted size for the read byte count.
	 */
	private long getAdjustedReadByteCount(final int lineFeedAdjustment, final long lineCount) {
		return readByteCount + (lineCount * lineFeedAdjustment);
	}
}
