package gov.va.cpss.job;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.atomic.AtomicBoolean;

import org.apache.log4j.Logger;

public abstract class CbssStreamToQueueThread extends Thread {

	protected static final Logger logger = Logger.getLogger(CbssStreamToQueueThread.class.getCanonicalName());

	/**
	 * The default encoding.
	 */
	private static final String DEFAULT_ENCODING = "utf-8";

	/**
	 * The byte count expected to read from the stream.
	 */
	protected long expectedByteCount = 0;

	/**
	 * The buffered reader to read from the provided input stream.
	 */
	protected BufferedReader bufferedReader;

	/**
	 * The output queue populated by reading data from the input stream.
	 */
	protected BlockingQueue<String> outputQueue;

	/**
	 * The byte count read from the stream.
	 */
	protected long readByteCount = 0;

	/**
	 * The result of processing.
	 */
	private AtomicBoolean successful = new AtomicBoolean(false);

	/**
	 * Constructor for thread to process the provided input stream in to an
	 * output queue. The expected bytes is provided for verification check the
	 * data was successfully processed. The calling thread must join this reader
	 * thread and the calling thread must close the input stream resource.
	 * 
	 * @param expectedByteCount
	 *            The expected amount of bytes to read.
	 * @param in
	 *            The input stream.
	 * @param outputQueue
	 *            The output queue.
	 * @throws UnsupportedEncodingException
	 */
	public CbssStreamToQueueThread(final long expectedByteCount, InputStream in, BlockingQueue<String> outputQueue)
			throws UnsupportedEncodingException {
		this(expectedByteCount, in, outputQueue, DEFAULT_ENCODING);
	}

	/**
	 * Constructor for thread to process the provided input stream in to an
	 * output queue. The expected bytes is provided for verification check the
	 * data was successfully processed.
	 * 
	 * @param expectedByteCount
	 *            The expected amount of bytes to read.
	 * @param in
	 *            The input stream.
	 * @param outputQueue
	 *            The output queue.
	 * @param encoding
	 *            The encoding to use for the input stream reader.
	 * @throws UnsupportedEncodingException
	 */
	public CbssStreamToQueueThread(final long expectedByteCount, InputStream in, BlockingQueue<String> outputQueue,
			final String encoding) throws UnsupportedEncodingException {
		this.expectedByteCount = expectedByteCount;
		this.bufferedReader = new BufferedReader(new InputStreamReader(in, encoding));
		this.outputQueue = outputQueue;
	}

	/**
	 * Thread run method that populates the queue.
	 */
	public void run() {
		successful.set(populateQueue());
	}

	/**
	 * Subclasses should provide an implementation to process the input stream
	 * into a queue.
	 */
	protected abstract boolean populateQueue();

	/**
	 * Check if processing into queue was successful.
	 * 
	 * @return Boolean value indicating if successful or not.
	 */
	public boolean isSuccessful() {
		return successful.get();
	}

}
