package gov.va.cpss.job;

import java.io.File;
import java.util.List;

import org.springframework.batch.core.scope.context.ChunkContext;

import gov.va.cpss.service.SftpService;

/**
 * Remove any partially transferred files on the sftp server that have the
 * temporary name. Log errors but continue processing.
 * 
 * @author DNS 
 */
public class CbssProcessFileCleanupTasklet extends CbssJobBaseTransactionTasklet {

	private String tempFilenamePostfix;

	private String serverTargetDirectory;

	private SftpService sftpService;

	private String stagingDirectory;

	public String getTempFilenamePostfix() {
		return tempFilenamePostfix;
	}

	public void setTempFilenamePostfix(String tempFilenamePostfix) {
		this.tempFilenamePostfix = tempFilenamePostfix;
	}

	public String getServerTargetDirectory() {
		return serverTargetDirectory;
	}

	public void setServerTargetDirectory(String serverTargetDirectory) {
		this.serverTargetDirectory = serverTargetDirectory;
	}

	public SftpService getSftpService() {
		return sftpService;
	}

	public void setSftpService(SftpService sftpService) {
		this.sftpService = sftpService;
	}

	public String getStagingDirectory() {
		return stagingDirectory;
	}

	public void setStagingDirectory(String stagingDirectory) {
		this.stagingDirectory = stagingDirectory;
	}

	@Override
	protected boolean executeLogic(ChunkContext chunkContext) {

		boolean successful = true;

		taskletLogger.info("File Cleanup Tasklet Running!");

		// Delete any files on the target sftp server that might be left over
		// from a previously failed batch run.
		List<String> serverFileL = sftpService.ftpGetFileListInDirectory(serverTargetDirectory);

		if ((serverFileL != null) && !serverFileL.isEmpty()) {

			// Loop through and delete any old files.
			for (String tempFile : serverFileL) {
				if (tempFile.toLowerCase().endsWith(tempFilenamePostfix)) {

					if (sftpService.ftpRemoveFileFromDirectory(tempFile, serverTargetDirectory)) {
						taskletLogger.warn("Cleaning up old temporary file on server: " + tempFile);
					} else {
						successful = false;
						taskletLogger.error("Error cleaning up old temporary file on server: " + tempFile);
					}
				}
			}
		}

		// Delete any files on the local that might be left over from a
		// previously failed batch run.
		File stagingFolder = new File(stagingDirectory);
		File[] tempFileL = stagingFolder.listFiles();
		if (tempFileL == null) {
			taskletLogger.error("Staging Directory does not exist or is not a folder:" + stagingDirectory);
			successful = false;
		} else {
			for (File tempFile : tempFileL) {
				if (tempFile.isFile()) {
					if (tempFile.delete()) {
						taskletLogger.warn("Cleaning up old temporary file on local: " + tempFile.getAbsolutePath());
					} else {
						successful = false;
						taskletLogger.error("Error cleaning up old temporary file on local: " + tempFile.getAbsolutePath());
					}
				}
			}
		}

		// Log a message but continue processing if unsuccessful.
		if (successful) {
			taskletLogger.info("File Cleanup was Successful");
		} else {
			taskletLogger.error("File Cleanup Tasklet Detected an Error but Processing will Continue");
		}

		return successful;
	}

}
