package gov.va.cpss.job;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;

import org.apache.log4j.Logger;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepContribution;
import org.springframework.batch.core.scope.context.ChunkContext;
import org.springframework.batch.core.step.tasklet.Tasklet;
import org.springframework.batch.repeat.RepeatStatus;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.TransactionCallbackWithoutResult;
import org.springframework.transaction.support.TransactionTemplate;

/**
 * Abstract base class that implements logic for a tasklet to do logic in a
 * transaction.
 * 
 * @author DNS   
 */
public abstract class CbssJobBaseTransactionTasklet implements Tasklet {

	protected static final Logger taskletLogger = Logger
			.getLogger(CbssJobBaseTransactionTasklet.class.getCanonicalName());

	private TransactionTemplate transactionTemplate;

	public TransactionTemplate getTransactionTemplate() {
		return transactionTemplate;
	}

	public void setTransactionTemplate(TransactionTemplate transactionTemplate) {
		this.transactionTemplate = transactionTemplate;
	}

	@Override
	public RepeatStatus execute(StepContribution contribution, ChunkContext chunkContext) throws Exception {

		transactionTemplate.execute(new TransactionCallbackWithoutResult() {
			@Override
			protected void doInTransactionWithoutResult(TransactionStatus transactionStatus) {

				if (!executeLogic(chunkContext)) {
					taskletLogger.error("Tasklet detected error and performed rollback");
					transactionStatus.setRollbackOnly();
				}
			}

		});

		return RepeatStatus.FINISHED;
	}

	/**
	 * Custom logic provided by implementing class to perform custom logic,
	 * customizing the logic for the implementing class.
	 * 
	 * @return Boolean flag indicating successful or not.
	 */
	abstract protected boolean executeLogic(ChunkContext chunkContext);

	/**
	 * Set the failure in the job execution context.
	 * 
	 * @param status
	 *            The failure status.
	 */
	protected void setFailureStatus(JobExecution jobExecution, final String status, final String message) {

		// Log job failure status.
		taskletLogger.error("Job completion failed with status: " + status);

		// Set job failure.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_KEY, status);

		// Log job failure message.
		taskletLogger.error("Job completion failure message: " + message);

		// Set job failure message.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_MESSAGE_KEY, message);
	}

}
