package gov.va.cpss.job;

/**
 * Implementation of CbssSftpReadLineItemReader that handles reading of special
 * formatted multi-record messages from AITC-CCPC.
 * 
 * Messages are delimited by '~' at the beginning of new line. Records are
 * delimited by carriage return and x85 (NEL). The NEL gets decoded as \uFFFD by
 * the BufferedReader of the CbssStreamToQueueThread.
 * 
 * An optional masterRecordCode may be set which will also be used to identify
 * the end of the current message. This allows processing files with messages
 * all on one line and no message delimiter line.
 * 
 * @author Brad Pickle
 *
 * @param <T>
 */
public class CCPCMessageSftpItemReader<T> extends CbssSftpReadLineItemReader<T> {
	public static final char NEL = '\uFFFD';
	public static final char MESSAGE_DELIMITER = '~';

	private String masterRecordCode;

	private StringBuffer lookAheadNextLine;

	/**
	 * The master record code is the sequence of characters identifying the
	 * record that starts a message. For example PA for print acks. Messages in
	 * print ack are composed of one PA record followed by many PD records. When
	 * processing a message of PA/PD records, when the next PA record is
	 * encountered this signifies the end of the preceding message.
	 * 
	 * @return String masterRecordCode if set
	 */
	public String getMasterRecordCode() {
		return masterRecordCode;
	}

	/**
	 * Set the masterRecordCode. When this code is encountered in the stream,
	 * readLine will recognize this as the end of a message and return
	 * everything up to but not including the next masterRecordCode.
	 * 
	 * @param masterRecordCode
	 */
	public void setMasterRecordCode(String masterRecordCode) {
		this.masterRecordCode = masterRecordCode;
	}

	@Override
	protected String readLine() throws Exception {
		final StringBuffer nextMessage = new StringBuffer();
		boolean foundMessage = false;

		StringBuffer nextLineBuffer = readLineWithLookAhead();

		while (nextLineBuffer != null) {

			if (isMessageDelimiterLine(nextLineBuffer)) {
				if (foundMessage) {
					nextMessage.append(MESSAGE_DELIMITER);
					break;
				}
				// Else read next line - Ignoring empty messages
			} else {
				foundMessage = true;
				nextMessage.append(nextLineBuffer);
				if (isMasterRecordCodeNext())
					break;
			}

			nextLineBuffer = readLineWithLookAhead();
		}

		return foundMessage ? nextMessage.toString() : null;
	}

	/**
	 * Read the next line from the queue and return as a StringBuffer. If the
	 * next line has already been read and saved during a look ahead, then the
	 * look ahead line will be returned instead then forgotten.
	 * 
	 * @return The look ahead line if already read from the queue, or the next
	 *         line in the queue. Null if the queue is empty.
	 * @throws Exception
	 */
	private StringBuffer readLineWithLookAhead() throws Exception {
		StringBuffer nextLineBuffer = null;

		if (lookAheadNextLine == null) {
			final String nextLine = super.readLine();
			if (nextLine != null) {
				nextLineBuffer = new StringBuffer(nextLine);
				removeNELAtStart(nextLineBuffer);
			}
		} else {
			nextLineBuffer = lookAheadNextLine;
			lookAheadNextLine = null;
		}
		return nextLineBuffer;
	}

	private StringBuffer lookAhead() throws Exception {
		lookAheadNextLine = readLineWithLookAhead();
		return lookAheadNextLine;
	}

	/**
	 * Deteremine if the first character in the given line is the message
	 * delimiter.
	 * 
	 * @param nextLineBuffer
	 * @return
	 */
	private boolean isMessageDelimiterLine(final StringBuffer nextLineBuffer) {
		return (nextLineBuffer.length() > 0) && (nextLineBuffer.charAt(0) == MESSAGE_DELIMITER);
	}

	/**
	 * Removes the first character from the given line if it is NEL.
	 * 
	 * @param nextLineBuffer
	 */
	private void removeNELAtStart(final StringBuffer nextLineBuffer) {
		if ((nextLineBuffer.length() > 0) && (nextLineBuffer.charAt(0) == NEL))
			nextLineBuffer.deleteCharAt(0);
	}

	/**
	 * Returns true if the master record code has been set.
	 * 
	 * @return
	 */
	private boolean masterRecordCodeDefined() {
		return (masterRecordCode != null) && (masterRecordCode.length() > 0);
	}

	/**
	 * Looks ahead in the queue to determine if the next line starts with the
	 * master record code, signifying the end of the current message. The line
	 * read during the look ahead will be saved and returned during the next
	 * readLineWithLookAhead.
	 * 
	 * @return
	 * @throws Exception
	 */
	private boolean isMasterRecordCodeNext() throws Exception {
		if (masterRecordCodeDefined()) {
			final StringBuffer nextLineBuffer = lookAhead();

			final boolean returnVal = ((nextLineBuffer != null)
					&& (nextLineBuffer.length() >= masterRecordCode.length())
					&& nextLineBuffer.substring(0, masterRecordCode.length()).equals(masterRecordCode));

			return returnVal;
		}

		return false;
	}
}
