package gov.va.cpss.job;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.RESULTS_SIZE_KEY;

import java.util.ArrayList;
import java.util.List;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.StepExecution;

/**
 * 
 * This is an abstract paging item processor that will processes items in a page.
 * 
 * Copyright HPE / VA
 * February 3, 2017
 * 
 * @author Yiping Yao
 * @version 1.0.0
 *
 */
@SuppressWarnings("nls")
public abstract class CBSSPagingItemProcessor<I, O> extends CBSSBaseItemProcessor<I, O>
{
    private List<I> inputItemsList;
    private int itemCount;
    private boolean isProcessing;

    /**
     * Before processing capture the job execution and the received id for data
     * processing.
     * 
     * @param stepExecution
     */
    @Override
    public void beforeStep(StepExecution stepExecution)
    {
        super.beforeStep(stepExecution);

        this.itemCount = 0;
        this.isProcessing = false;

        if (this.inputItemsList == null)
        {
            this.inputItemsList = new ArrayList<>();
        }
        else
        {
            this.inputItemsList.clear();
        }
    }

    /**
     * After processing check for unexpected conditions that suggest an error.
     * 
     * @param stepExecution
     * @return 
     */
    @Override
    public ExitStatus afterStep(StepExecution stepExecution)
    {
        this.inputItemsList = null;

        return super.afterStep(stepExecution);
    }

    @Override
    public void beforeProcess(I item)
    {
        super.beforeProcess(item);

        this.itemCount++;

        if (!this.isForceStop)
        {
            // Accumulate the items until it reaches the page size, then process at once.
            this.inputItemsList.add(item);

            if (this.itemCount == this.jobExecution.getExecutionContext().getInt(RESULTS_SIZE_KEY))
            {
                this.isProcessing = true;
            }
            else
            {
                this.isProcessing = false;
            }
        }
    }

    @Override
    public O process(final I item) throws Exception
    {
        // If job has already failed then just return null.
        if (this.jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY))
        {
            return null;
        }

        // If the accumulated items have not reached page size, don't process.
        if (!this.isProcessing)
        {
            return null;
        }

        this.logger.debug("Processing item count: " + this.itemCount);

        return doProcess(this.inputItemsList);
    }

    @Override
    public void afterProcess(final I item, final O result)
    {
        super.afterProcess(item, result);

        if (this.isProcessing)
        {
            this.itemCount = 0;
            this.inputItemsList.clear();
        }
    }


   // Abstract method that does the bulk processing.
    abstract protected O doProcess(List<I> inInputItemsList) throws Exception;
}
