package gov.va.cpss.job;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.DATA_ERROR_STATUS;

import java.io.File;
import java.io.IOException;

import java.nio.file.Files;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepContribution;
import org.springframework.batch.core.scope.context.ChunkContext;
import org.springframework.batch.core.step.tasklet.Tasklet;
import org.springframework.batch.repeat.RepeatStatus;

import gov.va.cpss.service.SftpService;

import gov.va.cpss.time.TimeUtils;

/**
 * 
 * CBSS / APPS Base File Transfer Tasklet.
 * 
 * Copyright DXC / VA
 * April 21, 2017
 * 
 * @author Yiping Yao
 * @version 1.0.0
 * 
 */
@SuppressWarnings("nls")
public abstract class CBSSBaseFileTransferTasklet implements Tasklet
{
    protected Log logger = LogFactory.getLog(getClass());

    protected final static String TIMESTAMP_FORMAT = "HH_mm_ss";

	// Injected properties
	protected String tempFilenamePostfix;
	protected String stagingDirectory;
	protected SftpService sftpService;

    public String getTempFilenamePostfix()
	{
		return this.tempFilenamePostfix;
	}

	public void setTempFilenamePostfix(String inTempFilenamePostfix)
	{
		this.tempFilenamePostfix = inTempFilenamePostfix;
	}

    /**
     * @return the stagingDirectory
     */
    public String getStagingDirectory()
    {
        return this.stagingDirectory;
    }

    /**
     * @param inStagingDirectory the stagingDirectory to set
     */
    public void setStagingDirectory(String inStagingDirectory)
    {
        this.stagingDirectory = inStagingDirectory;
    }

	public SftpService getSftpService()
	{
		return this.sftpService;
	}

	public void setSftpService(SftpService inSftpService)
	{
		this.sftpService = inSftpService;
	}

    @Override
    abstract public RepeatStatus execute(StepContribution contribution, ChunkContext chunkContext) throws Exception;

    protected String getJobParameter(final JobExecution jobExecution, final String jobParameterKey)
    {
        final String jobParameter = jobExecution.getJobParameters().getString(jobParameterKey);

        // If an error getting the jobParameter then set the failure status and trigger roll back.
        if ((jobParameter != null) && !jobParameter.isEmpty())
        {
            this.logger.debug(jobParameterKey + ": " + jobParameter);

        }
        else
        {
            // This is an unexpected and unrecoverable error.
            final String error = "Could not obtain the job parameter, " + jobParameterKey;
            setFailureStatus(jobExecution, DATA_ERROR_STATUS, error);
        }
        
        return jobParameter;
    }
    
    /**
     * Build a temporary filename for use on the target sftp server. The format
     * of the temporary filename: Given filename: 'APPS-06022016.txt' would
     * return temporary filename: 'APPS-06022016.txt.tempFilenamePostfix<timestamp>.'
     * 
     * @param filename
     *            The base filename.
     * @return The temporary filename.
     */
    protected String getTemporaryFilename(final String filename)
    {
        if ((filename != null) && !filename.isEmpty())
        {
            StringBuilder tempFilename = new StringBuilder(filename);
            tempFilename.append(".");
            tempFilename.append(TimeUtils.getTimestamp(TIMESTAMP_FORMAT));
            tempFilename.append(".");
            tempFilename.append(this.tempFilenamePostfix);

            return tempFilename.toString();
        }

        return null;
    }

    /**
     * @param tempFilePath
     *          The path to the file that needs to be deleted.
     * @return
     *          True if successful, false if not.
     */
    protected boolean deleteResource(String tempFilePath)
    {
        File file = new File(tempFilePath);

        try
        {
            return Files.deleteIfExists(file.toPath());
        }
        catch (IOException e)
        {
            final StringBuilder error = new StringBuilder();

            error.append("An error occurred while attempting to delete the local file, ");
            error.append(tempFilePath);
            error.append(", error: ");
            error.append(e.getClass().getSimpleName());
            error.append("\nMessage: ");
            error.append(e.getMessage());

            this.logger.error(error.toString());

            return false;
        }
    }

    /**
     * 
     * Set the failure in the job execution context.
     * 
     * @param jobExecution
     * @param status - The failure status
     * @param message - The message
     * 
     */
    protected void setFailureStatus(JobExecution jobExecution, final String status, final String message)
    {
        // Log job failure status.
        this.logger.error("Job completion failed with status: " + status);

        // Set job failure.
        jobExecution.getExecutionContext().putString(JOB_FAILURE_KEY, status);

        // Log job failure message.
        this.logger.error("Job completion failure message: " + message);

        // Set job failure message.
        jobExecution.getExecutionContext().putString(JOB_FAILURE_MESSAGE_KEY, message);
    }
}
