package gov.va.cpss.dao.impl;

import java.sql.Connection;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Iterator;
import java.util.List;

import javax.sql.DataSource;

import org.apache.log4j.Logger;
import org.springframework.jdbc.core.BatchPreparedStatementSetter;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.PreparedStatementCreator;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.support.GeneratedKeyHolder;
import org.springframework.jdbc.support.KeyHolder;

import gov.va.cpss.cobol.Money;
import gov.va.cpss.dao.CBSSAuditDatesDAO;
import gov.va.cpss.dao.CBSSiteTransDAO;
import gov.va.cpss.model.cbs.CBSSiteStmt;
import gov.va.cpss.model.cbs.CBSSiteTrans;
import gov.va.cpss.model.cbs.CBSStmt;

/**
 * An implementation of the CBSStmtDAO interface.
 * 
 * @author Brad Pickle
 */
public class CBSSiteTransDAOImpl implements CBSSiteTransDAO {

	@SuppressWarnings("unused")
	private final Logger daoLogger;

	private JdbcTemplate jdbcTemplate;

	private CBSSAuditDatesDAO cbssAuditDatesDAO;

	public CBSSiteTransDAOImpl() {
		daoLogger = Logger.getLogger(this.getClass().getCanonicalName());
	}

	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}

	public CBSSAuditDatesDAO getCbssAuditDatesDAO() {
		return cbssAuditDatesDAO;
	}

	public void setCbssAuditDatesDAO(CBSSAuditDatesDAO cbssAuditDatesDAO) {
		this.cbssAuditDatesDAO = cbssAuditDatesDAO;
	}

	@Override
	public List<CBSSiteTrans> getMessageStatementTransForSite(final long cbsMessageId, final String stationNum) {
		final String sql = "SELECT trans.id AS id, trans.siteStmtId AS siteStmtId, trans.datePosted AS datePosted, trans.transDesc AS transDesc, "
				+ "		trans.transAmntRaw AS transAmntRaw, trans.referenceNum AS referenceNum, trans.orderNum AS orderNum, trans.createdBy AS createdBy, "
				+ "		trans.createdDate AS createdDate, trans.modifiedBy AS modifiedBy, trans.modifiedDate AS modifiedDate "
				+ "FROM CBSStmt " + "  INNER JOIN CBSSiteStmt primarySite " + "    ON CBSStmt.id = primarySite.stmtId "
				+ "  INNER JOIN CBSSiteStmt site " + "    ON CBSStmt.id = site.stmtId "
				+ "  INNER JOIN CBSSiteTrans trans " + "    ON site.id = trans.siteStmtId " + "WHERE CBSStmt.messageId="
				+ cbsMessageId + " AND primarySite.isPrimary='Y' AND primarySite.stationNum='" + stationNum + "' "
				+ "ORDER BY site.stmtId, site.stationNum, trans.orderNum";

		final List<CBSSiteTrans> results = jdbcTemplate.query(sql, new CBSSiteTransRowMapper());

		return results;
	}

	@Override
	public List<CBSSiteTrans> getAllByCBSSiteStmtID(long stmtId) {

		// Get all CBSSiteTrans ordered by ID that are in all CBSSiteStmt
		// associated with the specified CBSStmt.
		String sql = "SELECT * FROM (SELECT * FROM CBSSiteTrans order by id) WHERE siteStmtId IN (SELECT id FROM (SELECT * FROM CBSSiteStmt order by id) WHERE stmtId = "
				+ stmtId + ")";

		List<CBSSiteTrans> cbsSiteTransL = jdbcTemplate.query(sql, new CBSSiteTransRowMapper());

		return cbsSiteTransL;
	}

	@Override
	public CBSSiteTrans get(long id) {
		final String sql = "SELECT * FROM CBSSiteTrans WHERE id = " + id;

		final List<CBSSiteTrans> cbsSiteTransL = jdbcTemplate.query(sql, new CBSSiteTransRowMapper());

		return (cbsSiteTransL.size() > 0) ? cbsSiteTransL.get(0) : null;
	}

	/**
	 * Maps a ResultSet to a new CBSSiteTrans object.
	 * 
	 * @author Brad Pickle
	 *
	 */
	private static class CBSSiteTransRowMapper implements RowMapper<CBSSiteTrans> {

		@Override
		public CBSSiteTrans mapRow(ResultSet rs, int rowNum) throws SQLException {

			CBSSiteTrans cbsSiteTrans = new CBSSiteTrans();

			cbsSiteTrans.setId(rs.getLong("id"));
			cbsSiteTrans.setSiteStmtId(rs.getLong("siteStmtId"));
			cbsSiteTrans.setDatePosted(rs.getDate("datePosted"));
			cbsSiteTrans.setTransDesc(rs.getString("transDesc"));
			cbsSiteTrans.setTransAmount(new Money(rs.getString("transAmntRaw")));
			cbsSiteTrans.setReferenceNum(rs.getString("referenceNum"));
			cbsSiteTrans.setOrderNum(rs.getInt("orderNum"));
			cbsSiteTrans.setCreatedBy(rs.getString("createdBy"));
			cbsSiteTrans.setCreatedDate(rs.getDate("createdDate"));
			cbsSiteTrans.setModifiedBy(rs.getString("modifiedBy"));
			cbsSiteTrans.setModifiedDate(rs.getDate("modifiedDate"));

			return cbsSiteTrans;
		}

	}

	@Override
	public long save(final CBSSiteTrans cbsSiteTrans) {
		// insert
		final String sql = "INSERT INTO CBSSiteTrans (siteStmtId, datePosted, transDesc, transAmnt, transAmntRaw, referenceNum, orderNum) VALUES (?, ?, ?, ?, ?, ?, ?)";
		KeyHolder keyHolder = new GeneratedKeyHolder();

		jdbcTemplate.update(new PreparedStatementCreator() {
			@Override
			public PreparedStatement createPreparedStatement(Connection connection) throws SQLException {
				PreparedStatement ps = connection.prepareStatement(sql, new String[] { "id" });
				ps.setLong(1, cbsSiteTrans.getSiteStmtId());
				ps.setDate(2, (cbsSiteTrans.getDatePosted() == null) ? null
						: new Date(cbsSiteTrans.getDatePosted().getTime()));
				ps.setString(3, cbsSiteTrans.getTransDesc());
				ps.setDouble(4, cbsSiteTrans.getTransAmount().getDouble());
				ps.setString(5, cbsSiteTrans.getTransAmount().getCobol());
				ps.setString(6, cbsSiteTrans.getReferenceNum());
				ps.setInt(7, cbsSiteTrans.getOrderNum());
				return ps;
			}
		}, keyHolder);
		cbsSiteTrans.setId(keyHolder.getKey().longValue());

		// Load the new audit fields and set on cbsSiteTrans
		cbssAuditDatesDAO.getAuditDates("CBSSiteTrans", cbsSiteTrans.getId(), cbsSiteTrans);

		return cbsSiteTrans.getId();
	}

	@Override
	public void saveBatch(List<CBSStmt> statements) {
		final String sql = "INSERT INTO CBSSiteTrans (siteStmtId, datePosted, transDesc, transAmnt, "
				+ "transAmntRaw, referenceNum, orderNum) VALUES (?, ?, ?, ?, ?, ?, ?)";

		jdbcTemplate.batchUpdate(sql, new BatchPreparedStatementSetter() {

			private Iterator<CBSStmt> cbsStmtIter;
			private Iterator<CBSSiteStmt> cbsSiteStmtIter;
			private Iterator<CBSSiteTrans> cbsSiteTransIter;
			private CBSStmt currentStatement;
			private CBSSiteStmt currentSiteStatement;

			@Override
			public void setValues(PreparedStatement ps, int i) throws SQLException {
				CBSSiteTrans cbsSiteTrans = getNextCBSSiteTrans();

				ps.setLong(1, currentSiteStatement.getId());
				ps.setDate(2, (cbsSiteTrans.getDatePosted() == null) ? null
						: new Date(cbsSiteTrans.getDatePosted().getTime()));
				ps.setString(3, cbsSiteTrans.getTransDesc());
				ps.setDouble(4, cbsSiteTrans.getTransAmount().getDouble());
				ps.setString(5, cbsSiteTrans.getTransAmount().getCobol());
				ps.setString(6, cbsSiteTrans.getReferenceNum());
				ps.setInt(7, cbsSiteTrans.getOrderNum());

			}

			@Override
			public int getBatchSize() {
				// Return total number of CBSSiteTrans in list of CBSStmts.
				return statements.stream().filter(stmt -> stmt.getSiteStmtL() != null)
						.flatMap(stmt -> stmt.getSiteStmtL().stream())
						.filter(siteStmt -> siteStmt.getSiteTransL() != null)
						.mapToInt(siteStmt -> siteStmt.getSiteTransL().size()).sum();
			}

			private CBSSiteTrans getNextCBSSiteTrans() {
				if ((cbsSiteTransIter == null) || (!cbsSiteTransIter.hasNext())) {
					List<CBSSiteTrans> cbsSiteTransList = getNextCBSSiteTransList();
					if (cbsSiteTransList != null) {
						cbsSiteTransIter = cbsSiteTransList.iterator();
					}
				}

				CBSSiteTrans nextCBSSiteTrans = null;
				if ((cbsSiteTransIter != null) && (cbsSiteTransIter.hasNext())) {
					nextCBSSiteTrans = cbsSiteTransIter.next();
				}

				return nextCBSSiteTrans;
			}

			private List<CBSSiteTrans> getNextCBSSiteTransList() {
				List<CBSSiteTrans> cbsSiteTransList = null;
				do {
					currentSiteStatement = getNextCBSSiteStmt();
					if (currentSiteStatement != null) {
						cbsSiteTransList = currentSiteStatement.getSiteTransL();
					} else {
						cbsSiteTransList = null;
						break;
					}
				} while ((cbsSiteTransList == null) || (cbsSiteTransList.size() == 0));

				return cbsSiteTransList;
			}
			
			private CBSSiteStmt getNextCBSSiteStmt() {
				if ((cbsSiteStmtIter == null) || (!cbsSiteStmtIter.hasNext())) {
					List<CBSSiteStmt> cbsSiteStmtList = getNextCBSSiteStmtList();
					if (cbsSiteStmtList != null) {
						cbsSiteStmtIter = cbsSiteStmtList.iterator();
					}
				}

				CBSSiteStmt nextCBSSiteStmt = null;
				if ((cbsSiteStmtIter != null) && (cbsSiteStmtIter.hasNext())) {
					nextCBSSiteStmt = cbsSiteStmtIter.next();
				}

				return nextCBSSiteStmt;
			}

			private List<CBSSiteStmt> getNextCBSSiteStmtList() {
				if (cbsStmtIter == null)
					cbsStmtIter = statements.iterator();

				List<CBSSiteStmt> cbsSiteStmtList = null;
				do {
					if (cbsStmtIter.hasNext()) {
						currentStatement = cbsStmtIter.next();
						cbsSiteStmtList = currentStatement.getSiteStmtL();
					} else {
						currentStatement = null;
						cbsSiteStmtList = null;
						break;
					}
				} while ((cbsSiteStmtList == null) || (cbsSiteStmtList.size() == 0));

				return cbsSiteStmtList;
			}
		});

	}

	public static String getInsertCBSSiteTransSql() {
		return "INSERT INTO CBSSiteTrans(siteStmtId, datePosted, transDesc, transAmnt, transAmntRaw, referenceNum, orderNum) "
				+ "values(?, ?, ?, ?, ?, ?, ?)";
	}
}
