package gov.va.cpss.dao.impl;

import java.sql.Connection;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Iterator;
import java.util.List;

import javax.sql.DataSource;

import org.springframework.jdbc.core.BatchPreparedStatementSetter;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.PreparedStatementCreator;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.support.GeneratedKeyHolder;
import org.springframework.jdbc.support.KeyHolder;

import gov.va.cpss.cobol.Money;
import gov.va.cpss.dao.CBSSAuditDatesDAO;
import gov.va.cpss.dao.CBSSiteStmtDAO;
import gov.va.cpss.model.BooleanChar;
import gov.va.cpss.model.cbs.CBSSiteStmt;
import gov.va.cpss.model.cbs.CBSStmt;

/**
 * An implementation of the CBSSiteStmtDAO interface.
 */
public class CBSSiteStmtDAOImpl implements CBSSiteStmtDAO {

	private JdbcTemplate jdbcTemplate;

	private CBSSAuditDatesDAO cbssAuditDatesDAO;

	public CBSSiteStmtDAOImpl() {
	}

	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}

	public CBSSAuditDatesDAO getCbssAuditDatesDAO() {
		return cbssAuditDatesDAO;
	}

	public void setCbssAuditDatesDAO(CBSSAuditDatesDAO cbssAuditDatesDAO) {
		this.cbssAuditDatesDAO = cbssAuditDatesDAO;
	}

	@Override
	public long save(final CBSSiteStmt siteStatement, final boolean updateAuditDates) {
		// insert
		final String sql = "INSERT INTO CBSSiteStmt (stmtId, stationNum, stationPhoneNum, processDate, statementDate, amntDue, preBalance, totalCharges, totalCredits, newBalance, specialNote, noparacdes, largeFontInd, ARAddressFlag, lastBillPrepDate, isPrimary, isPrimaryAddress) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
		KeyHolder keyHolder = new GeneratedKeyHolder();

		jdbcTemplate.update(new PreparedStatementCreator() {
			@Override
			public PreparedStatement createPreparedStatement(Connection connection) throws SQLException {
				PreparedStatement ps = connection.prepareStatement(sql, new String[] { "id" });
				ps.setLong(1, siteStatement.getStmtId());
				ps.setString(2, siteStatement.getStationNum());
				ps.setString(3, siteStatement.getStationPhoneNum());
				ps.setDate(4, new Date(siteStatement.getProcessDate().getTime()));
				ps.setDate(5, new Date(siteStatement.getStatementDate().getTime()));
				ps.setDouble(6, siteStatement.getAmountDue().getDouble());
				ps.setDouble(7, siteStatement.getPrevBalance().getDouble());
				ps.setDouble(8, siteStatement.getTotalCharges().getDouble());
				ps.setDouble(9, siteStatement.getTotalCredits().getDouble());
				ps.setDouble(10, siteStatement.getNewBalance().getDouble());
				ps.setString(11, siteStatement.getSpecialNotes());
				ps.setString(12, siteStatement.getNoParaCdes());
				ps.setString(13, siteStatement.getLargeFontInd().toString());
				ps.setString(14, siteStatement.getArAddressFlag().toString());
				ps.setDate(15, new Date(siteStatement.getLastBillPrepDate().getTime()));
				ps.setString(16, siteStatement.getIsPrimary().toString());
				ps.setString(17, siteStatement.getIsPrimaryAddress().toString());
				return ps;
			}
		}, keyHolder);
		siteStatement.setId(keyHolder.getKey().longValue());

		if (updateAuditDates) {
			// Load the new audit fields and set on siteStatement
			cbssAuditDatesDAO.getAuditDates("CBSSiteStmt", siteStatement.getId(), siteStatement);
		}

		return siteStatement.getId();
	}

	@Override
	public long save(final CBSSiteStmt siteStatement, final long batchRunId, final long accountId) {
		// insert
		final String sql = "INSERT INTO CBSSiteStmt (stmtId, stationNum, stationPhoneNum, processDate, "
				+ "statementDate, amntDue, preBalance, totalCharges, totalCredits, newBalance, specialNote, "
				+ "noparacdes, largeFontInd, ARAddressFlag, lastBillPrepDate, isPrimary, isPrimaryAddress) "
				+ "SELECT id, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ? "
				+ "FROM CBSStmt WHERE batchRunId = ? AND accountId = ?";
		KeyHolder keyHolder = new GeneratedKeyHolder();

		jdbcTemplate.update(new PreparedStatementCreator() {
			@Override
			public PreparedStatement createPreparedStatement(Connection connection) throws SQLException {
				PreparedStatement ps = connection.prepareStatement(sql, new String[] { "id" });

				ps.setString(1, siteStatement.getStationNum());
				ps.setString(2, siteStatement.getStationPhoneNum());
				ps.setDate(3, new Date(siteStatement.getProcessDate().getTime()));
				ps.setDate(4, new Date(siteStatement.getStatementDate().getTime()));
				ps.setDouble(5, siteStatement.getAmountDue().getDouble());
				ps.setDouble(6, siteStatement.getPrevBalance().getDouble());
				ps.setDouble(7, siteStatement.getTotalCharges().getDouble());
				ps.setDouble(8, siteStatement.getTotalCredits().getDouble());
				ps.setDouble(9, siteStatement.getNewBalance().getDouble());
				ps.setString(10, siteStatement.getSpecialNotes());
				ps.setString(11, siteStatement.getNoParaCdes());
				ps.setString(12, siteStatement.getLargeFontInd().toString());
				ps.setString(13, siteStatement.getArAddressFlag().toString());
				ps.setDate(14, new Date(siteStatement.getLastBillPrepDate().getTime()));
				ps.setString(15, siteStatement.getIsPrimary().toString());
				ps.setString(16, siteStatement.getIsPrimaryAddress().toString());

				ps.setLong(17, batchRunId);
				ps.setLong(18, accountId);
			
				return ps;
			}
		}, keyHolder);
		siteStatement.setId(keyHolder.getKey().longValue());

		return siteStatement.getId();
	}

	@Override
	public void saveBatch(List<CBSStmt> statements) {
		final String sql = "INSERT INTO CBSSiteStmt (stmtId, stationNum, stationPhoneNum, processDate, "
				+ "statementDate, amntDue, preBalance, totalCharges, totalCredits, newBalance, specialNote, "
				+ "noparacdes, largeFontInd, ARAddressFlag, lastBillPrepDate, isPrimary, isPrimaryAddress) "
				+ "SELECT id, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ? "
				+ "FROM CBSStmt WHERE batchRunId = ? AND accountId = ?";

		jdbcTemplate.batchUpdate(sql, new BatchPreparedStatementSetter() {

			private Iterator<CBSStmt> cbsStmtIter;
			private Iterator<CBSSiteStmt> cbsSiteStmtIter;
			private CBSStmt currentStatement;

			@Override
			public void setValues(PreparedStatement ps, int i) throws SQLException {
				CBSSiteStmt siteStatement = getNextCBSSiteStmt();

				ps.setString(1, siteStatement.getStationNum());
				ps.setString(2, siteStatement.getStationPhoneNum());
				ps.setDate(3, new Date(siteStatement.getProcessDate().getTime()));
				ps.setDate(4, new Date(siteStatement.getStatementDate().getTime()));
				ps.setDouble(5, siteStatement.getAmountDue().getDouble());
				ps.setDouble(6, siteStatement.getPrevBalance().getDouble());
				ps.setDouble(7, siteStatement.getTotalCharges().getDouble());
				ps.setDouble(8, siteStatement.getTotalCredits().getDouble());
				ps.setDouble(9, siteStatement.getNewBalance().getDouble());
				ps.setString(10, siteStatement.getSpecialNotes());
				ps.setString(11, siteStatement.getNoParaCdes());
				ps.setString(12, siteStatement.getLargeFontInd().toString());
				ps.setString(13, siteStatement.getArAddressFlag().toString());
				ps.setDate(14, new Date(siteStatement.getLastBillPrepDate().getTime()));
				ps.setString(15, siteStatement.getIsPrimary().toString());
				ps.setString(16, siteStatement.getIsPrimaryAddress().toString());

				ps.setLong(17, currentStatement.getBatchRunId());
				ps.setLong(18, currentStatement.getAccountId());
			}

			@Override
			public int getBatchSize() {
				// Return total number of CBSSiteStmts in list of CBSStmts.
				return statements.stream().filter(stmt -> stmt.getSiteStmtL() != null)
						.mapToInt(stmt -> stmt.getSiteStmtL().size()).sum();
			}

			private CBSSiteStmt getNextCBSSiteStmt() {
				if ((cbsSiteStmtIter == null) || (!cbsSiteStmtIter.hasNext())) {
					List<CBSSiteStmt> cbsSiteStmtList = getNextCBSSiteStmtList();
					if (cbsSiteStmtList != null) {
						cbsSiteStmtIter = cbsSiteStmtList.iterator();
					}
				}

				CBSSiteStmt nextCBSSiteStmt = null;
				if ((cbsSiteStmtIter != null) && (cbsSiteStmtIter.hasNext())) {
					nextCBSSiteStmt = cbsSiteStmtIter.next();
				}

				return nextCBSSiteStmt;
			}

			private List<CBSSiteStmt> getNextCBSSiteStmtList() {
				if (cbsStmtIter == null)
					cbsStmtIter = statements.iterator();

				List<CBSSiteStmt> cbsSiteStmtList = null;
				do {
					if (cbsStmtIter.hasNext()) {
						currentStatement = cbsStmtIter.next();
						cbsSiteStmtList = currentStatement.getSiteStmtL();
					} else {
						currentStatement = null;
						cbsSiteStmtList = null;
						break;
					}
				} while ((cbsSiteStmtList == null) || (cbsSiteStmtList.size() == 0));

				return cbsSiteStmtList;
			}
		});

	}

	@Override
	public List<CBSSiteStmt> getMessageStatementSitesForSite(final long cbsMessageId, final String stationNum) {
		final String sql = "SELECT site.id AS site_id, site.stmtId AS site_stmtId, site.stationNum AS site_stationNum, "
				+ "		site.stationPhoneNum AS site_stationPhoneNum, site.processDate AS site_processDate, site.statementDate AS site_statementDate, "
				+ "		site.amntDue AS site_amntDue, site.preBalance AS site_preBalance, site.totalCharges AS site_totalCharges, site.totalCredits AS site_totalCredits, "
				+ "		site.newBalance AS site_newBalance, site.specialNote AS site_specialNote, site.noParaCdes AS site_noParaCdes, site.largeFontInd AS site_largeFontInd, "
				+ "		site.arAddressFlag AS site_arAddressFlag, site.lastBillPrepDate AS site_lastBillPrepDate, site.isPrimary AS site_isPrimary, "
				+ "		site.isPrimaryAddress AS site_isPrimaryAddress, site.createdBy AS site_createdBy, site.createdDate AS site_createdDate, "
				+ "		site.modifiedBy AS site_modifiedBy, site.modifiedDate AS site_modifiedDate, " +

				"		patient.id AS patient_id, patient.siteStmtId AS patient_siteStmtId, patient.icn AS patient_ICN, patient.dfn AS patient_DFN, patient.oldAcntNum AS patient_oldAcntNum, "
				+ "		patient.firstName AS patient_firstName, patient.lastName AS patient_lastName, patient.middleName AS patient_middleName, "
				+ "		patient.address1 AS patient_address1, patient.address2 AS patient_address2, patient.address3 AS patient_address3, "
				+ "		patient.city AS patient_city, patient.state AS patient_state, patient.zipCode AS patient_zipCode, "
				+ "		patient.country AS patient_country, patient.createdBy AS patient_createdBy, patient.createdDate AS patient_createdDate, "
				+ "		patient.modifiedBy AS patient_modifiedBy, patient.modifiedDate AS patient_modifiedDate "

				+ "FROM CBSStmt INNER JOIN CBSSiteStmt primarySite ON CBSStmt.id = primarySite.stmtId "
				+ "  INNER JOIN CBSSiteStmt site ON CBSStmt.id = site.stmtId "
				+ "  INNER JOIN CBSSitePatient patient ON site.id = patient.siteStmtId "

				+ "WHERE CBSStmt.messageId=" + cbsMessageId
				+ " AND primarySite.isPrimary='Y' AND primarySite.stationNum='" + stationNum + "' "
				
				+ "ORDER BY site.stmtId, site.isPrimary DESC, site.stationNum";

		final List<CBSSiteStmt> results = jdbcTemplate.query(sql, new CBSSiteStmtRowMapper(true));

		return results;
	}

	@Override
	public List<CBSSiteStmt> getAllByCBSStmtID(final long stmtId, final boolean loadSitePatient) {

		final String sql = "SELECT site.id as site_id, site.stmtId as site_stmtId, site.stationNum as site_stationNum, site.stationPhoneNum as site_stationPhoneNum, site.processDate as site_processDate, "
				+ "		site.statementDate as site_statementDate, site.amntDue as site_amntDue, site.preBalance as site_preBalance, site.totalCharges as site_totalCharges, site.totalCredits as site_totalCredits, site.newBalance as site_newBalance, "
				+ "		site.specialNote as site_specialNote, site.noParaCdes as site_noParaCdes, site.largeFontInd as site_largeFontInd, site.arAddressFlag as site_arAddressFlag, site.lastBillPrepDate as site_lastBillPrepDate, "
				+ "		site.isPrimary as site_isPrimary, site.isPrimaryAddress as site_isPrimaryAddress, site.createdBy as site_createdBy, site.createdDate as site_createdDate, site.modifiedBy as site_modifiedBy, site.modifiedDate as site_modifiedDate "
				+

				(loadSitePatient
						? "," + "		patient.id as patient_id, patient.siteStmtId as patient_siteStmtId, patient.icn as patient_icn, patient.dfn as patient_dfn, patient.oldAcntNum as patient_oldAcntNum, patient.firstName as patient_firstName, patient.lastName as patient_lastName, patient.middleName as patient_middleName, "
								+ "		patient.address1 as patient_address1, patient.address2 as patient_address2, patient.address3 as patient_address3, patient.city as patient_city, patient.state as patient_state, patient.zipCode as patient_zipCode, "
								+ "		patient.country as patient_country, patient.createdBy as patient_createdBy, patient.createdDate as patient_createdDate, patient.modifiedBy as patient_modifiedBy, patient.modifiedDate as patient_modifiedDate "
						: "")
				+

				"FROM CBSSiteStmt site "

				+

				(loadSitePatient ? "  INNER JOIN CBSSitePatient patient ON site.id = patient.siteStmtId " : " ")

				+

				"WHERE site.stmtId = " + stmtId;

		final List<CBSSiteStmt> results = jdbcTemplate.query(sql, new CBSSiteStmtRowMapper(loadSitePatient));

		return results;
	}
	

	/**
	 * Maps a ResultSet to a new CBSSiteStmt object. A loadPatient flag passed
	 * to the constructor indicates whether or not to also map the ResultSet to
	 * a new CBSSitePatient object for the CBSSiteStmt object's sitePatient
	 * property.
	 * 
	 * ResultSet fields for the CBSSiteStmt object are expected to be prepended
	 * with "site_" and fields for the CBSSitePatient object are expected to be
	 * prepended with "patient_".
	 * 
	 * @author Brad Pickle
	 *
	 */
	private static class CBSSiteStmtRowMapper implements RowMapper<CBSSiteStmt> {

		private final boolean loadPatient;

		private final CBSSitePatientDAOImpl.CBSSitePatientRowMapper patientRowMapper;

		CBSSiteStmtRowMapper(boolean loadPatient) {
			this.loadPatient = loadPatient;
			this.patientRowMapper = loadPatient ? new CBSSitePatientDAOImpl.CBSSitePatientRowMapper("patient_") : null;
		}

		@Override
		public CBSSiteStmt mapRow(ResultSet rs, int row) throws SQLException {
			CBSSiteStmt cbsSiteStmt = new CBSSiteStmt();

			cbsSiteStmt.setId(rs.getLong("site_id"));
			cbsSiteStmt.setStmtId(rs.getLong("site_stmtId"));
			cbsSiteStmt.setStationNum(rs.getString("site_stationNum"));
			cbsSiteStmt.setStationPhoneNum(rs.getString("site_stationPhoneNum"));
			cbsSiteStmt.setProcessDate(rs.getDate("site_processDate"));
			cbsSiteStmt.setStatementDate(rs.getDate("site_statementDate"));
			cbsSiteStmt.setAmountDue(new Money(rs.getDouble("site_amntDue"), 9));
			cbsSiteStmt.setPrevBalance(new Money(rs.getDouble("site_preBalance"), 9));
			cbsSiteStmt.setTotalCharges(new Money(rs.getDouble("site_totalCharges"), 9));
			cbsSiteStmt.setTotalCredits(new Money(rs.getDouble("site_totalCredits"), 9));
			cbsSiteStmt.setNewBalance(new Money(rs.getDouble("site_newBalance"), 9));
			cbsSiteStmt.setSpecialNotes(rs.getString("site_specialNote"));
			cbsSiteStmt.setNoParaCdes(rs.getString("site_noParaCdes"));
			cbsSiteStmt.setLargeFontInd(BooleanChar.from(rs.getString("site_largeFontInd")));
			cbsSiteStmt.setArAddressFlag(BooleanChar.from(rs.getString("site_arAddressFlag")));
			cbsSiteStmt.setLastBillPrepDate(rs.getDate("site_lastBillPrepDate"));
			cbsSiteStmt.setIsPrimary(BooleanChar.from(rs.getString("site_isPrimary")));
			cbsSiteStmt.setIsPrimaryAddress(BooleanChar.from(rs.getString("site_isPrimaryAddress")));
			cbsSiteStmt.setCreatedBy(rs.getString("site_createdBy"));
			cbsSiteStmt.setCreatedDate(rs.getDate("site_createdDate"));
			cbsSiteStmt.setModifiedBy(rs.getString("site_modifiedBy"));
			cbsSiteStmt.setModifiedDate(rs.getDate("site_modifiedDate"));

			if (loadPatient) {
				cbsSiteStmt.setSitePatient(patientRowMapper.mapRow(rs, row));
			}

			return cbsSiteStmt;
		}

	}
}
