package gov.va.cpss.dao.impl;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Iterator;
import java.util.List;

import javax.sql.DataSource;

import org.springframework.jdbc.core.BatchPreparedStatementSetter;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.PreparedStatementCreator;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.support.GeneratedKeyHolder;
import org.springframework.jdbc.support.KeyHolder;

import gov.va.cpss.dao.CBSSAuditDatesDAO;
import gov.va.cpss.dao.CBSSitePatientDAO;
import gov.va.cpss.model.cbs.CBSSitePatient;
import gov.va.cpss.model.cbs.CBSSiteStmt;
import gov.va.cpss.model.cbs.CBSStmt;

/**
 * An implementation of the CBSSitePatientDAO interface.
 */
public class CBSSitePatientDAOImpl implements CBSSitePatientDAO {

	private JdbcTemplate jdbcTemplate;

	private CBSSAuditDatesDAO cbssAuditDatesDAO;

	public CBSSitePatientDAOImpl() {
	}

	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}

	public CBSSAuditDatesDAO getCbssAuditDatesDAO() {
		return cbssAuditDatesDAO;
	}

	public void setCbssAuditDatesDAO(CBSSAuditDatesDAO cbssAuditDatesDAO) {
		this.cbssAuditDatesDAO = cbssAuditDatesDAO;
	}

	@Override
	public long save(final CBSSitePatient patient) {
		// insert
		final String sql = "INSERT INTO CBSSitePatient (siteStmtId, ICN, DFN, oldAcntNum, firstName, lastName, middleName, address1, address2, address3, city, state, zipCode, country) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
		KeyHolder keyHolder = new GeneratedKeyHolder();

		jdbcTemplate.update(new PreparedStatementCreator() {
			@Override
			public PreparedStatement createPreparedStatement(Connection connection) throws SQLException {
				PreparedStatement ps = connection.prepareStatement(sql, new String[] { "id" });
				ps.setLong(1, patient.getSiteStmtId());
				ps.setString(2, patient.getIcn());
				ps.setLong(3, patient.getDfn());
				ps.setString(4, patient.getOldAcntNum());
				ps.setString(5, patient.getFirstName());
				ps.setString(6, patient.getLastName());
				ps.setString(7, patient.getMiddleName());
				ps.setString(8, patient.getAddress1());
				ps.setString(9, patient.getAddress2());
				ps.setString(10, patient.getAddress3());
				ps.setString(11, patient.getCity());
				ps.setString(12, patient.getState());
				ps.setString(13, patient.getZipCode());
				ps.setString(14, patient.getCountry());
				return ps;
			}
		}, keyHolder);
		patient.setId(keyHolder.getKey().longValue());

		// Load the new audit fields and set on patient
		cbssAuditDatesDAO.getAuditDates("CBSSitePatient", patient.getId(), patient);

		return patient.getId();
	}

	@Override
	public void saveBatch(List<CBSStmt> statements) {
		final String sql = "INSERT INTO CBSSitePatient (siteStmtId, ICN, DFN, oldAcntNum, firstName, "
				+ "lastName, middleName, address1, address2, address3, city, state, zipCode, country) "
				+ "VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

		jdbcTemplate.batchUpdate(sql, new BatchPreparedStatementSetter() {

			private Iterator<CBSStmt> cbsStmtIter;
			private Iterator<CBSSiteStmt> cbsSiteStmtIter;
			private CBSStmt currentStatement;

			@Override
			public void setValues(PreparedStatement ps, int i) throws SQLException {
				CBSSiteStmt siteStatement = getNextCBSSiteStmt();
				CBSSitePatient patient = siteStatement.getSitePatient();

				ps.setLong(1, siteStatement.getId());
				ps.setString(2, patient.getIcn());
				ps.setLong(3, patient.getDfn());
				ps.setString(4, patient.getOldAcntNum());
				ps.setString(5, patient.getFirstName());
				ps.setString(6, patient.getLastName());
				ps.setString(7, patient.getMiddleName());
				ps.setString(8, patient.getAddress1());
				ps.setString(9, patient.getAddress2());
				ps.setString(10, patient.getAddress3());
				ps.setString(11, patient.getCity());
				ps.setString(12, patient.getState());
				ps.setString(13, patient.getZipCode());
				ps.setString(14, patient.getCountry());

			}

			@Override
			public int getBatchSize() {
				// Return total number of CBSSiteStmts in list of CBSStmts.
				return statements.stream().filter(stmt -> stmt.getSiteStmtL() != null)
						.mapToInt(stmt -> stmt.getSiteStmtL().size()).sum();
			}

			private CBSSiteStmt getNextCBSSiteStmt() {
				if ((cbsSiteStmtIter == null) || (!cbsSiteStmtIter.hasNext())) {
					List<CBSSiteStmt> cbsSiteStmtList = getNextCBSSiteStmtList();
					if (cbsSiteStmtList != null) {
						cbsSiteStmtIter = cbsSiteStmtList.iterator();
					}
				}

				CBSSiteStmt nextCBSSiteStmt = null;
				if ((cbsSiteStmtIter != null) && (cbsSiteStmtIter.hasNext())) {
					nextCBSSiteStmt = cbsSiteStmtIter.next();
				}

				return nextCBSSiteStmt;
			}

			private List<CBSSiteStmt> getNextCBSSiteStmtList() {
				if (cbsStmtIter == null)
					cbsStmtIter = statements.iterator();

				List<CBSSiteStmt> cbsSiteStmtList = null;
				while ((cbsSiteStmtList == null) || (cbsSiteStmtList.size() == 0)) {
					if (cbsStmtIter.hasNext()) {
						currentStatement = cbsStmtIter.next();
						cbsSiteStmtList = currentStatement.getSiteStmtL();
					} else {
						currentStatement = null;
						cbsSiteStmtList = null;
						break;
					}
				}

				return cbsSiteStmtList;
			}
		});

	}

	@Override
	public CBSSitePatient get(long id) {
		final String sql = "SELECT * FROM CBSSitePatient WHERE id = " + id;

		final List<CBSSitePatient> cbsSitePatients = jdbcTemplate.query(sql, new CBSSitePatientRowMapper(""));

		return (cbsSitePatients.size() > 0) ? cbsSitePatients.get(0) : null;
	}

	/**
	 * Maps a ResultSet to a new CBSSitePatient object. A tablePrefix is
	 * specified in the constructor which will be prepended to all of the
	 * ResultSet field names.
	 * 
	 * @author Brad Pickle
	 *
	 */
	static class CBSSitePatientRowMapper implements RowMapper<CBSSitePatient> {

		private String tablePrefix;

		CBSSitePatientRowMapper(String tablePrefix) {
			this.tablePrefix = (tablePrefix == null) ? "" : tablePrefix;
		}

		@Override
		public CBSSitePatient mapRow(ResultSet rs, int rowNum) throws SQLException {

			CBSSitePatient cbsSitePatient = new CBSSitePatient(rs.getLong(tablePrefix + "id"));

			cbsSitePatient.setSiteStmtId(rs.getLong(tablePrefix + "siteStmtId"));
			cbsSitePatient.setIcn(rs.getString(tablePrefix + "ICN"));
			cbsSitePatient.setDfn(rs.getLong(tablePrefix + "DFN"));
			cbsSitePatient.setOldAcntNum(rs.getString(tablePrefix + "oldAcntNum"));
			cbsSitePatient.setFirstName(rs.getString(tablePrefix + "firstName"));
			cbsSitePatient.setLastName(rs.getString(tablePrefix + "lastName"));
			cbsSitePatient.setMiddleName(rs.getString(tablePrefix + "middleName"));
			cbsSitePatient.setAddress1(rs.getString(tablePrefix + "address1"));
			cbsSitePatient.setAddress2(rs.getString(tablePrefix + "address2"));
			cbsSitePatient.setAddress3(rs.getString(tablePrefix + "address3"));
			cbsSitePatient.setCity(rs.getString(tablePrefix + "city"));
			cbsSitePatient.setState(rs.getString(tablePrefix + "state"));
			cbsSitePatient.setZipCode(rs.getString(tablePrefix + "zipCode"));
			cbsSitePatient.setCountry(rs.getString(tablePrefix + "country"));
			cbsSitePatient.setCreatedBy(rs.getString(tablePrefix + "createdBy"));
			cbsSitePatient.setCreatedDate(rs.getDate(tablePrefix + "createdDate"));
			cbsSitePatient.setModifiedBy(rs.getString(tablePrefix + "modifiedBy"));
			cbsSitePatient.setModifiedDate(rs.getDate(tablePrefix + "modifiedDate"));

			return cbsSitePatient;
		}

	}

	@SuppressWarnings("nls")
	@Override
	public int[] batchUpdate(final String[] inNewICNs, final String[] inOldICNs) {
		String sql = "UPDATE CBSSitePatient SET icn = ? WHERE icn = ?";

		return this.jdbcTemplate.batchUpdate(sql, new BatchPreparedStatementSetter() {
			@Override
			public void setValues(PreparedStatement ps, int i) throws SQLException {
				ps.setString(1, inNewICNs[i]);
				ps.setString(2, inOldICNs[i]);
			}

			@Override
			public int getBatchSize() {
				return inOldICNs.length;
			}
		});
	}

}
