package gov.va.cpss.dao.impl;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.stream.Stream;

import javax.sql.DataSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.springframework.dao.DataAccessException;

import org.springframework.jdbc.core.BatchPreparedStatementSetter;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.PreparedStatementCreator;
import org.springframework.jdbc.core.PreparedStatementSetter;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.SqlTypeValue;
import org.springframework.jdbc.core.StatementCreatorUtils;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;

import org.springframework.jdbc.support.GeneratedKeyHolder;
import org.springframework.jdbc.support.KeyHolder;

import gov.va.cpss.dao.CBSSAuditDatesDAO;
import gov.va.cpss.dao.ProcessStatusDAO;

import gov.va.cpss.model.CBSSAbstractModel;
import gov.va.cpss.model.ProcessStatus.Status;

/**
 * 
 * An abstract super class for CBSS DAO implementation classes, which contains some common properties and methods.
 * 
 * (Sprint 2 - Story 325577 Update ICNs / Task 325578 Business service implementation to update ICN across CBSS.)
 * 
 * Copyright HPE / VA
 * June 14, 2016
 * 
 * @author Yiping Yao
 * @version 1.0.0
 *
 *
 * Sprint 5 - Epic 349068, Tasks 370262 and 370266
 *            DAO Implementation, and 
 *            Implement ICN Update Service - Update based on new design.
 *
 * Updated with new methods.
 *
 * Copyright HPE / VA
 * August 29, 2016
 * 
 * @author Yiping Yao
 * @version 2.0.0
 *
 */
@SuppressWarnings({"nls", "static-method"})
public abstract class CBSSBaseDAOImpl
{
    protected Log logger = LogFactory.getLog(getClass());

    //
    // Constants
    //
    // Primary Key / ID column
    public static final String ID = "ID";

    // No ID
    public static final long NO_ID = -1L;

    // Currently, there is a limitation with the Oracle DB / driver
    // of only supporting up to 1000 parameters in a prepared statement.
    public static final int PARAMS_LIMIT = 1000;

    // Optimization hint for Oracle DB
    protected static final String PARALLEL_HINT = "/*+ PARALLEL */";
    protected static final String APPEND_HINT = "/*+ APPEND */";
    protected static final String APPEND_VALUES_HINT = "/*+ APPEND_VALUES */";
    protected static final String FIRST_ROWS_HINT = "/*+ FIRST_ROWS */";
    protected static final String FIRST_ROWS_PARALLEL_HINT = "/*+ FIRST_ROWS PARALLEL */";

    // SQL Clause
    protected static final String ORDER_BY_ID_CLAUSE = " ORDER BY " + ID;

    // Spring JDBC Template
    protected JdbcTemplate jdbcTemplate;
    protected NamedParameterJdbcTemplate namedParameterJdbcTemplate;

    // Injected properties - need to be setup in the context.xml to be able to use them.
    protected CBSSAuditDatesDAO cbssAuditDatesDAO;
    protected ProcessStatusDAO processStatusDAO;

    /**
     * @return the cbssAuditDatesDAO
     */
    public CBSSAuditDatesDAO getCbssAuditDatesDAO()
    {
        return this.cbssAuditDatesDAO;
    }

    /**
     * @param inCbssAuditDatesDAO the cbssAuditDatesDAO to set
     */
    public void setCbssAuditDatesDAO(CBSSAuditDatesDAO inCbssAuditDatesDAO)
    {
        this.cbssAuditDatesDAO = inCbssAuditDatesDAO;
    }

    /**
     * @return the processStatusDAO
     */
    public ProcessStatusDAO getProcessStatusDAO()
    {
        return this.processStatusDAO;
    }

    /**
     * @param inProcessStatusDAO the processStatusDAO to set
     */
    public void setProcessStatusDAO(ProcessStatusDAO inProcessStatusDAO)
    {
        this.processStatusDAO = inProcessStatusDAO;
    }

    /**
     * @param dataSource
     */
    public void setDataSource(DataSource dataSource)
    {
        this.jdbcTemplate = new JdbcTemplate(dataSource);
        this.namedParameterJdbcTemplate = new NamedParameterJdbcTemplate(dataSource);
    }

    /**
     * @param model
     */
    public void setAuditFields(final CBSSAbstractModel model)
    {
        if (model == null)
        {
            return;
        }

        final String sql = "SELECT createdBy, createdDate, modifiedBy, modifiedDate FROM " + getTableName() + " WHERE " + ID + " = " + model.getId();
        
        this.jdbcTemplate.query(sql, new RowMapper<CBSSAbstractModel>()
        {
            @Override
            public CBSSAbstractModel mapRow(ResultSet rs, int rowNum) throws SQLException
            {
                mapAuditFields(rs, model);

                return model;
            }
        });

        return;
    }


    //
    // DAO methods
    //
    /**
     * Return the last index in the underlying table.
     * Implement the abstract method in CBSSBaseDAO Interface.
     * 
     * @return Long - Last index
     * 
     */
    public Long getLastIndex()
    {
        // query
        final String sql = "SELECT * FROM ( SELECT * FROM " + getTableName() + ORDER_BY_ID_CLAUSE + " DESC ) WHERE rownum <= 1";

        List<Long> results = this.jdbcTemplate.query(sql, new RowMapper<Long>()
        {
            @Override
            public Long mapRow(ResultSet rs, int row) throws SQLException
            {
                return new Long(rs.getLong(1));
            }
        });

        if ((results != null) && !results.isEmpty())
        {
            return results.get(0);
        }

        return null;
    }

    /**
     * Save CBSSAbstractModel and return the generated key.
     * Implement the abstract method in CBSSBaseDAO Interface.
     * 
     * @param month
     * @return long - the generated key
     * 
     */
    public long save(final CBSSAbstractModel model)
    {
        if (model == null)
        {
            return NO_ID;
        }

        // insert
        final String sql = getInsertSQL();
        KeyHolder keyHolder = new GeneratedKeyHolder();

        this.jdbcTemplate.update(new PreparedStatementCreator()
        {
            @Override
            public PreparedStatement createPreparedStatement(Connection connection) throws SQLException
            {
                PreparedStatement ps = connection.prepareStatement(sql, new String[]{ID});

                mapRows(ps, model);

                return ps;
            }
        }, keyHolder);

        model.setId(keyHolder.getKey().longValue());

        // Load the new audit fields and set on Patient
        setAuditFields(model);

        return model.getId();
    }

    /**
     * Batch insert / save and return the generated keys.
     * Implement the abstract method in CBSSBaseDAO Interface.
     * 
     * @param models
     * @return array of long - the generated keys
     * 
     */
    public long[] batchSave(final List<? extends CBSSAbstractModel> models)
    {
        return batchInsertWithGeneratedKeys(models);
    }

    /**
     * Batch insert / save and return the generated keys.
     * Implement the abstract method in CBSSBaseDAO Interface.
     * 
     * @param models
     * @return array of long - the generated keys
     * 
     */
    public long[] batchInsertWithGeneratedKeys(final List<? extends CBSSAbstractModel> models)
    {
        if (models == null || models.isEmpty())
        {
            return null;
        }

        KeyHolder keyHolder = new GeneratedKeyHolder();
        long[] keys = new long[models.size()];
        String[] keyColumns = new String[]{ID};

        int i = 0;

        for (CBSSAbstractModel model : models)
        {
            this.jdbcTemplate.update(new PreparedStatementCreator()
            {
                @Override
                public PreparedStatement createPreparedStatement(Connection connection) throws SQLException
                {
                    PreparedStatement ps = connection.prepareStatement(getInsertSQL(), keyColumns);

                    mapRows(ps, model);

                    return ps;
                }
            }, keyHolder);

            if (keyHolder.getKey() != null)
            {
                model.setId(keyHolder.getKey().longValue());
            }
            else
            {
                model.setId(NO_ID);
            }

            keys[i] = model.getId();

            // Load the new audit fields and set on model
            setAuditFields(model);

            i++;
        }

        return keys;
    }

    /**
     * Batch insert / save / update without returning the generated keys.
     *
     * @param sql
     * @param models
     * @return int[] - an array of the number of rows affected by each statement
     * 
     */
    public int[] batchUpdate(final String sql, final List<? extends CBSSAbstractModel> models)
    {
        if (models == null || models.isEmpty())
        {
            return null;
        }

        return this.jdbcTemplate.batchUpdate(sql, new BatchPreparedStatementSetter()
               {
                    @Override
                    public void setValues(PreparedStatement ps, int i) throws SQLException
                    {
                        CBSSAbstractModel model = models.get(i);
                        mapRows(ps, model);
                    }

                    @Override
                    public int getBatchSize()
                    {
                        return models.size();
                    }
               });
    }

    /**
     * Batch insert without returning the generated key.
     * 
     * @param models
     * @return
     */
    public int[] batchInsert(final List<? extends CBSSAbstractModel> models)
    {
        return batchUpdate(getInsertSQL(), models);
    }

    /**
     * Update the status of the model.
     * 
     * @param model
     * @return the number of updated models.
     */
    public int updateStatus(final CBSSAbstractModel model)
    {
        return this.jdbcTemplate.update(getUpdateStatusSQL(), Integer.valueOf(model.getStatusId()), Long.valueOf(model.getId()));
    }

    /**
     * Update the status based on the model key / primary ID.
     * 
     * @param id
     * @param status
     * @return the number of updated models.
     */
    public int update(final long id, final Status status)
    {
        return this.jdbcTemplate.update(getUpdateStatusSQL(), Integer.valueOf(getStatusID(status)), Long.valueOf(id));
    }

    /**
     * Batch update the status of list of model keys / primary ID's.
     * 
     * @param ids
     * @param status
     * @return the numbers of updated models for each ID.
     */
    public int[] update(final long[] ids, final Status status)
    {
        int statusId = getStatusID(status);

        return this.jdbcTemplate.batchUpdate(getUpdateStatusSQL(), new BatchPreparedStatementSetter()
               {
                    @Override
                    public void setValues(PreparedStatement ps, int i) throws SQLException
                    {
                        ps.setInt(1, statusId);
                        ps.setLong(2, ids[i]);
                    }

                    @Override
                    public int getBatchSize()
                    {
                        return ids.length;
                    }
               });
    }

    /**
     * Delete one row of data by its primary ID / key.
     * 
     * @param id
     * 
     */
    public int deleteById(final long id)
    {
        return this.jdbcTemplate.update(getDeleteSQL(), Long.valueOf(id));
    }

    /**
     * Select data rows.
     * 
     * @param whereClause
     * @param orderByClause
     * @return list of models
     * 
     */
    public List<? extends CBSSAbstractModel> select(final String whereClause, final String orderByClause)
    {
        String sql = buildSQL(getSelectSQL(), whereClause, orderByClause);

        this.logger.debug("Select SQL: " + sql);

        List<? extends CBSSAbstractModel> models = this.jdbcTemplate.query(sql, new RowMapper<CBSSAbstractModel>()
        {
            @Override
            public CBSSAbstractModel mapRow(ResultSet rs, int rowNum) throws SQLException
            {
                return mapResult(rs);
            }
        });

        return models;
    }

    /**
     * Select data rows with parameterized where clause.
     * 
     * @param whereClause
     * @param orderByClause
     * @param params
     * @return list of models
     * 
     */
    public List<? extends CBSSAbstractModel> select(final String whereClause, final String orderByClause, final Object[] params)
    {
        String sql = buildSQL(getSelectSQL(), whereClause, orderByClause);

        this.logger.debug("Select SQL: " + sql);

        List<? extends CBSSAbstractModel> models = this.jdbcTemplate.query(sql, params, new RowMapper<CBSSAbstractModel>()
        {
            @Override
            public CBSSAbstractModel mapRow(ResultSet rs, int rowNum) throws SQLException
            {
                return mapResult(rs);
            }
        });

        return models;
    }

    /**
     * Build a complete SQL from different parts.
     * 
     * @param partialSQL
     * @param whereClause
     * @param orderByClause
     * @return
     */
    protected String buildSQL(final String partialSQL, final String whereClause, final String orderByClause)
    {
        StringBuilder sql = new StringBuilder();

        sql.append(partialSQL);

        if (whereClause != null && !whereClause.isEmpty())
        {
            sql.append(" ");
            sql.append(whereClause);
        }

        if (orderByClause != null && !orderByClause.isEmpty())
        {
            sql.append(" ");
            sql.append(orderByClause);
        }

        return sql.toString();
    }

    /**
     * Select data rows with pagination.
     * 
     * @param pageNumber
     * @param pageSize
     * @return list of models
     * 
     */
    public List<? extends CBSSAbstractModel> selectWithPaging(final int pageNumber, final int pageSize)
    {
        return selectWithPaging(pageNumber, pageSize, getSelectSQL() + ORDER_BY_ID_CLAUSE);
    }

    /**
     * Select data rows with pagination.
     * 
     * @param pageNumber
     * @param pageSize
     * @param selectWithPagingSQL
     * @return list of models
     * 
     */
    protected List<? extends CBSSAbstractModel> selectWithPaging(final int pageNumber, final int pageSize, final String selectWithPagingSQL)
    {
        String sql = "SELECT * FROM " +
                     "( " +
                         "SELECT a.*, rownum rn " +
                             "FROM " +
                             "( " + selectWithPagingSQL + " ) a " +
                         "WHERE rownum < " + ( (pageNumber * pageSize) + 1 ) +
                      ") " +
                      "WHERE rn >= " + ( ((pageNumber - 1) * pageSize) + 1 );

        this.logger.debug("Select with paging SQL: " + sql);

        List<? extends CBSSAbstractModel> models = this.jdbcTemplate.query(sql, new RowMapper<CBSSAbstractModel>()
        {
            @Override
            public CBSSAbstractModel mapRow(ResultSet rs, int rowNum) throws SQLException
            {
                return mapResult(rs);
            }
        });

        return models;
    }

    /**
     * Select data rows with pagination.
     * 
     * @param pageNumber
     * @param pageSize
     * @param selectWithPagingSQLwithArgs
     * @param params
     * @return list of models
     * 
     */
    protected List<? extends CBSSAbstractModel> selectWithPaging(final int pageNumber, final int pageSize, final String selectWithPagingSQLwithParams, final Object[] params)
    {
        String sql = "SELECT * FROM " +
                     "( " +
                         "SELECT a.*, rownum rn " +
                             "FROM " +
                             "( " + selectWithPagingSQLwithParams + " ) a " +
                         "WHERE rownum < " + ( (pageNumber * pageSize) + 1 ) +
                      ") " +
                      "WHERE rn >= " + ( ((pageNumber - 1) * pageSize) + 1 );

        this.logger.debug("Select with paging SQL: " + sql);

        List<? extends CBSSAbstractModel> models = this.jdbcTemplate.query(sql, params, new RowMapper<CBSSAbstractModel>()
        {
            @Override
            public CBSSAbstractModel mapRow(ResultSet rs, int rowNum) throws SQLException
            {
                return mapResult(rs);
            }
        });

        return models;
    }

    /**
     * Batch select data based on the input parameters and using the IN clause.
     * 
     * Note that Oracle DB has a parameter limit of 1,000 in the IN clause,
     * therefore, the query needs to be broken into sub queries. 
     * 
     * @param openingSQL
     * @param closingSQL
     * @param openingParams
     * @param inClauseParams
     * @param closingParams
     * @return list of models
     * 
     */
    public List<? extends CBSSAbstractModel> batchSelectWithINClause(final String openingSQL, final String closingSQL,
                                                                     final Object[] openingParams, final Object[] inClauseParams, final Object[] closingParams)
    {
        StringBuilder sqlBuilder = new StringBuilder();
        List<CBSSAbstractModel> allModels = new ArrayList<>();
        List<? extends CBSSAbstractModel> models;

        int size = inClauseParams.length;
        int pages = ( size / PARAMS_LIMIT ) + (size % PARAMS_LIMIT == 0 ? 0 : 1);

        for (int i = 0; i < pages; i++)
        {
            // Starting index
            int from = i * PARAMS_LIMIT;

            // There are three cases for the end index "to":
            //  1. It is equal to the size;
            //  2. It is equal to the page end, if there is more than one page;
            //  3. It is equal to the partial page end, if it is the last page.
            int to = ( size <= ((i + 1) * PARAMS_LIMIT) ) ? size : (i + 1) * PARAMS_LIMIT;

            // Clear the string builder
            sqlBuilder.setLength(0);

            sqlBuilder.append(openingSQL);
            sqlBuilder.append(buildINClause(to - from));

            if (closingSQL != null)
            {
                sqlBuilder.append(closingSQL);
            }

            this.logger.debug(sqlBuilder.toString());

            Object[] subINClauseParams = Arrays.copyOfRange(inClauseParams, from, to);

            Object[] allParams = subINClauseParams;

            if (openingParams != null && openingParams.length > 0)
            {
                allParams = Stream.concat(Arrays.stream(openingParams), Arrays.stream(allParams)).toArray(Object[]::new);
            }

            if (closingParams != null && closingParams.length > 0)
            {
                allParams = Stream.concat(Arrays.stream(allParams), Arrays.stream(closingParams)).toArray(Object[]::new);
            }

            models = batchSelect(sqlBuilder.toString(), allParams);
            // if there is no matching icn in the DB, query may return empty Result set.
            //models may be null and it may throw null pointer exception.
            // defect 574224. 
            if(models != null){
            allModels.addAll(models);
            }
        }

        return allModels;
    }

    /**
     * Batch query database based on the array of single parameter.
     * 
     * @param sql
     * @param params
     * @return list of models
     * 
     */
    public List<? extends CBSSAbstractModel> batchSelect(final String sql, final Object[] params)
    {
        return this.jdbcTemplate.query(sql, params, new ResultSetExtractor<List<? extends CBSSAbstractModel>>()
        {
            @Override
            public List<? extends CBSSAbstractModel> extractData(ResultSet rs) throws SQLException, DataAccessException
            {
                List<CBSSAbstractModel> models = null;

                if (rs.isBeforeFirst())
                {
                    models = new ArrayList<>();

                    while (rs.next())
                    {
                        CBSSAbstractModel model = mapResult(rs);

                        models.add(model);
                    }
                }

                return models;
            }
        });
    }

    /**
     * Batch query database based on the arrays of multiple parameters.
     * 
     * @param sql
     * @param params
     * @return list of models
     * 
     */
    public List<? extends CBSSAbstractModel> batchSelect(final String sql, final List<List<?>> paramsList)
    {
        List<CBSSAbstractModel> allModels = new ArrayList<>();

        for (List<?> params : paramsList)
        {
            List<CBSSAbstractModel> models = this.jdbcTemplate.query(sql,
                new PreparedStatementSetter()
                {
                    @Override
                    public void setValues(PreparedStatement preparedStatement) throws SQLException
                    {
                        for (int i = 0; i < params.size(); i++)
                        {
                            StatementCreatorUtils.setParameterValue(preparedStatement, i + 1, SqlTypeValue.TYPE_UNKNOWN, params.get(i));
                        }
                    }
                },
                new ResultSetExtractor<List<CBSSAbstractModel>>()
                {
                    @Override
                    public List<CBSSAbstractModel> extractData(ResultSet rs) throws SQLException
                    {
                        List<CBSSAbstractModel> modelsFromRS = null;

                        if (rs.isBeforeFirst())
                        {
                            modelsFromRS = new ArrayList<>();

                            while (rs.next())
                            {
                                CBSSAbstractModel modelFromRS = mapResult(rs);

                                modelsFromRS.add(modelFromRS);
                            }
                        }

                        return modelsFromRS;
                    }
                });

            if (models != null)
            {
                allModels.addAll(models);
            }
        }

        return allModels;
    }


    /**
     * Default method that returns the Select SQL string.
     * This is a generic method and it is just for backward compatibility.
     * The sub-class may need to overwrite it with the specific select SQL.
     * 
     * @return select SQL string
     * 
     */
    public String getSelectSQL()
    {
        return "SELECT * FROM " + getTableName();
    }

    /**
     * Default method that returns the Delete SQL string.
     * This is a generic method and it is just for backward compatibility.
     * The sub-class may need to overwrite it with the specific delete SQL.
     * 
     * @return delete SQL string
     * 
     */
    public String getDeleteSQL()
    {
        return "DELETE FROM " + getTableName() + " WHERE " + ID + " = ?";
    }

    /**
     * This method returns the Update status SQL string.
     * 
     * @return update status SQL string
     * 
     */
    public String getUpdateStatusSQL()
    {
        return "UPDATE " + getTableName() + " SET statusId = ? WHERE " + ID + " = ?";
    }


    /**
     * Build parameterized IN clause: " IN (?, ?, ?, ...) ".
     * 
     * @param size
     * @return INClause
     * 
     */
    protected String buildINClause(final int size)
    {
        StringBuilder inClauseBuilder = new StringBuilder();

        inClauseBuilder.append(" IN ( ");

        for (int i = 0; i < size; i++)
        {
            inClauseBuilder.append("?");

            if (i + 1 < size)
            {
                inClauseBuilder.append(", ");
            }
        }

        inClauseBuilder.append(" ) ");

        return inClauseBuilder.toString();
    }

    /**
     * Map the common audit fields from the ResultSet to the model.
     * 
     * @param rs
     * @param model
     * @throws SQLException
     * 
     */
    protected void mapAuditFields(ResultSet rs, CBSSAbstractModel model) throws SQLException
    {
        model.setCreatedBy(rs.getString("createdBy"));
        model.setCreatedDate(new Date(rs.getDate("createdDate").getTime()));
        model.setModifiedBy(rs.getString("modifiedBy"));
        model.setModifiedDate(new Date(rs.getDate("modifiedDate").getTime()));
    }


    //
    // Utility methods
    //
    /**
     * Return Status ID from Status ENUM.
     * (This method could / should be in Process Status model / DAO.)
     * 
     * @param inStatus
     * @return status id
     * 
     */
    public int getStatusID(Status inStatus)
    {
        if (this.processStatusDAO == null)
        {
            this.logger.error("The Process Status DAO is not set.");

            return -1;
        }

        Integer statusID = this.processStatusDAO.getStatusFromEnum(inStatus);

        if (statusID == null)
        {
            this.logger.error("Unable to obtain status mapping for: " + inStatus);

            return -1;
        }

        return statusID.intValue();
    }

    /**
     * Return Status ENUM from Status ID.
     * (This method could / should be in ProcStatus model.)
     * 
     * @param statusID
     * @return status
     * 
     */
    public Status getStatus(int statusID)
    {
        if (this.processStatusDAO == null)
        {
            this.logger.error("The Process Status DAO is not set.");

            return null;
        }

        return this.processStatusDAO.getStatusType(statusID).getStatus();
    }


    /**
     * Static method to get 4 characters year from Date.
     * 
     * @param date
     * @return year
     * 
     */
    public static int getYear(Date date)
    {
        if (date == null)
        {
            return 0;
        }

        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);

        return calendar.get(Calendar.YEAR);
    }

    /**
     * Static method to get Date from year, assuming
     * first day and first month of the year.
     * 
     * @param year
     * @return Date
     * 
     */
    public static Date getDate(int year)
    {
        if (year == 0)
        {
            return null;
        }

        Calendar calendar = Calendar.getInstance();

        calendar.set(Calendar.YEAR, year);
        calendar.set(Calendar.MONTH, 1);
        calendar.set(Calendar.DATE, 1);

        return calendar.getTime();
    }


    //
    // Abstract methods
    //
    /**
     * Abstract method that returns the table name and subclass must implement.
     */
    abstract public String getTableName();

    /**
     * Abstract method that returns the Insert SQL string.
     */
    abstract public String getInsertSQL();

    /**
     * Abstract method that returns the model mapped from the ResultSet.
     */
    abstract protected CBSSAbstractModel mapResult(ResultSet rs) throws SQLException;

    /**
     * Abstract method that maps rows in Prepared Statement with the corresponding model.
     */
    abstract protected void mapRows(PreparedStatement ps, CBSSAbstractModel model) throws SQLException;
}
