package gov.va.cpss.dao.impl;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.sql.DataSource;

import org.apache.log4j.Logger;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.PreparedStatementCreator;
import org.springframework.jdbc.core.PreparedStatementSetter;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.support.GeneratedKeyHolder;
import org.springframework.jdbc.support.KeyHolder;

import gov.va.cpss.dao.CBSAccountDAO;
import gov.va.cpss.dao.CBSSAuditDatesDAO;
import gov.va.cpss.dao.util.SQLValuesList;
import gov.va.cpss.model.cbs.CBSAccount;

/**
 * An implementation of the CBSAccountDAO interface.
 */
@SuppressWarnings("nls")
public class CBSAccountDAOImpl implements CBSAccountDAO {

	public final static int BATCH_SELECT_MAX_SIZE = 100;

	private final static String BATCH_SELECT_SQL_MAX = buildBatchSelectSQL(BATCH_SELECT_MAX_SIZE);

	@SuppressWarnings("unused")
	private final Logger daoLogger;

	private JdbcTemplate jdbcTemplate;

	private CBSSAuditDatesDAO cbssAuditDatesDAO;

	public CBSAccountDAOImpl() {
		daoLogger = Logger.getLogger(this.getClass().getCanonicalName());
	}

	/**
	 * @return the cbssAuditDatesDAO
	 */
	public CBSSAuditDatesDAO getCbssAuditDatesDAO() {
		return this.cbssAuditDatesDAO;
	}

	/**
	 * @param inCbssAuditDatesDAO
	 *            the cbssAuditDatesDAO to set
	 */
	public void setCbssAuditDatesDAO(CBSSAuditDatesDAO inCbssAuditDatesDAO) {
		this.cbssAuditDatesDAO = inCbssAuditDatesDAO;
	}

	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}

	@Override
	public Long save(final String icn) {
		// insert
		final String sql = "INSERT INTO CBSAccount (icn) VALUES (?)";
		KeyHolder keyHolder = new GeneratedKeyHolder();

		jdbcTemplate.update(new PreparedStatementCreator() {
			@Override
			public PreparedStatement createPreparedStatement(Connection connection) throws SQLException {
				PreparedStatement ps = connection.prepareStatement(sql, new String[] { "id" });
				ps.setString(1, icn);
				return ps;
			}
		}, keyHolder);

		return keyHolder.getKey().longValue();
	}

	@Override
	public Long selectByICN(final String icn) {

		String sql = "SELECT id FROM CBSAccount where icn = '" + icn + "'";
		return jdbcTemplate.query(sql, new ResultSetExtractor<Long>() {

			@Override
			public Long extractData(ResultSet rs) throws SQLException, DataAccessException {
				if (rs.next()) {
					return rs.getLong("id");
				}

				return null;
			}

		});
	}

	@Override
	public String selectICNById(final long id) {

		String sql = "SELECT ICN FROM CBSAccount where id = '" + id + "'";
		return jdbcTemplate.query(sql, new ResultSetExtractor<String>() {

			@Override
			public String extractData(ResultSet rs) throws SQLException, DataAccessException {
				if (rs.next()) {
					return rs.getString("ICN");
				}

				return null;
			}

		});
	}

	@Override
	public Long getByICN(final String icn) {

		Long accountNumber = selectByICN(icn);

		if (accountNumber == null) {
			accountNumber = save(icn);
		}

		return accountNumber;
	}

	@Override
	public List<CBSAccount> batchSelect(String[] inICNs) {
		final ICNPreparedStatementSetter pss = new ICNPreparedStatementSetter(inICNs);

		final List<CBSAccount> allCbsAccounts = new ArrayList<>(inICNs.length);

		while (pss.getRemainingICNCount() > 0) {
			allCbsAccounts.addAll(this.jdbcTemplate.query(pss.getNextBatchSelectSQL(), pss,
					new ResultSetExtractor<List<CBSAccount>>() {

						@Override
						public List<CBSAccount> extractData(ResultSet rs) throws SQLException, DataAccessException {
							List<CBSAccount> cbsAccounts = null;

							cbsAccounts = new ArrayList<>();

							while (rs.next()) {
								CBSAccount cbsAccount = new CBSAccount();

								cbsAccount.setId(rs.getLong("ID"));
								cbsAccount.setIcn(rs.getString("ICN"));

								cbsAccounts.add(cbsAccount);
							}

							return cbsAccounts;
						}

					}));
		}

		return allCbsAccounts;
	}

	/**
	 * PreparedStatementSetter implementation for populating ICN values in
	 * SELECT queries. Allows the array of ICNs to be batched across multiple,
	 * potentially different-sized queries. Assumes all parameters in the
	 * PreparedStatement are for ICNs.
	 * 
	 * @author Brad Pickle
	 *
	 */
	private class ICNPreparedStatementSetter implements PreparedStatementSetter {

		private String[] icns;

		private int nextIndex = 0;

		/**
		 * Create a PreparedStatementSetter for the given array of ICNs.
		 * 
		 * @param inICNs
		 */
		ICNPreparedStatementSetter(String[] inICNs) {
			icns = (inICNs == null) ? new String[0] : inICNs;
		}

		@Override
		public void setValues(PreparedStatement ps) throws SQLException {
			final int nextBatchSize = getNextBatchSize();
			for (int i = 1; i <= nextBatchSize; i++) {
				final String nextICN = (nextIndex < icns.length) ? icns[nextIndex++] : (String) null;
				ps.setString(i, nextICN);
			}
		}

		/**
		 * @return The remaining number of ICNs that have not been injected into
		 *         a statement yet.
		 */
		public int getRemainingICNCount() {
			final int remainingICNCount = (nextIndex < icns.length) ? (icns.length - nextIndex) : 0;
			return remainingICNCount;
		}
		
		/**
		 * Returns the parameterized query String for the next batch of ICNs.
		 * 
		 * @param remainingCount
		 *            The remaining number of ICNs to query.
		 * @return
		 */
		private String getNextBatchSelectSQL() {
			final int nextBatchSize = getNextBatchSize();
			
			String nextBatchSelectSQL;
			if (nextBatchSize >= BATCH_SELECT_MAX_SIZE) {
				nextBatchSelectSQL = BATCH_SELECT_SQL_MAX;
			} else {
				nextBatchSelectSQL = buildBatchSelectSQL(nextBatchSize);
			}
			
			return nextBatchSelectSQL;
		}

		public int getNextBatchSize() {
			final int nextBatchSize = Math.min(getRemainingICNCount(), BATCH_SELECT_MAX_SIZE);
			return nextBatchSize;
		}

	}

	@Override
	public List<CBSAccount> batchInsertAndReturnCBSAccounts(Set<String> inICNs) {
		final List<CBSAccount> cbsAccounts = new ArrayList<>(inICNs.size());
		
		final KeyHolder keyHolder = new GeneratedKeyHolder();
		final String[] keyColumn = new String[] { "ID" };

		for (String icn : inICNs) {
			this.jdbcTemplate.update(new PreparedStatementCreator() {
				@Override
				public PreparedStatement createPreparedStatement(Connection connection) throws SQLException {
					PreparedStatement ps = connection.prepareStatement(getInsertSQL(), keyColumn);
					ps.setString(1, icn);
					return ps;
				}
			}, keyHolder);

			final CBSAccount cbsAccount = new CBSAccount();
			
			if (keyHolder.getKey() != null) {
				cbsAccount.setId(keyHolder.getKey().longValue());
			} else {
				cbsAccount.setId(-1L);
			}
			
			cbsAccount.setIcn(icn);
			cbsAccounts.add(cbsAccount);
		}

		return cbsAccounts;
	}

	// Build batch select SQL for the given number of ICNs. Length should not
	// exceed 1000 due to the oracle jdbc driver having a limit of 1000
	// parameters in a prepared statement.
	private static String buildBatchSelectSQL(int length) {
		StringBuilder sql = new StringBuilder(getSelectClause());

		sql.append(" WHERE ICN IN ").append((new SQLValuesList(length)).toString());

		return sql.toString();
	}

	// Return insert SQL.
	static String getInsertSQL() {
		return "INSERT INTO CBSAccount (ICN) VALUES (?)";
	}

	// Return common select clause for CBSAccount table queries.
	private static String getSelectClause() {
		return "SELECT * FROM CBSAccount";
	}
}
