package gov.va.cpss.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.sql.DataSource;

import org.apache.log4j.Logger;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;

import gov.va.cpss.dao.BatchStatusDAO;
import gov.va.cpss.model.BatchStatus;
import static gov.va.cpss.ESAPI.EsapiValidator.validateStringInput;
import static gov.va.cpss.ESAPI.EsapiValidationType.LOG_FORGING;
/**
 * An implementation of the BatchStatusDAO interface.
 */
public class BatchStatusDAOImpl implements BatchStatusDAO {

    private final Logger daoLogger;

    private JdbcTemplate jdbcTemplate;

    private Map<Integer, BatchStatus> idToTypeM;
    private Map<BatchStatus.JobStatus, Integer> enumToIdM;

    public BatchStatusDAOImpl() {

        daoLogger = Logger.getLogger(this.getClass().getCanonicalName());

        idToTypeM = new HashMap<>();
        enumToIdM = new HashMap<>();
    }

    public void setDataSource(DataSource dataSource) {
        this.jdbcTemplate = new JdbcTemplate(dataSource);
    }

    @Override
    public void initTypeMaps() {

        List<BatchStatus> bSL = statusTypeList();

        for (BatchStatus bS : bSL) {
            BatchStatus.JobStatus enumValue = BatchStatus.JobStatus.lookupByName(bS.getName());
            if (enumValue == null) {
                daoLogger.error("Unrecognized batch status in database: " + validateStringInput(bS.getName(), LOG_FORGING));
            } else {
                bS.setJobStatus(enumValue);
            }
            idToTypeM.put(bS.getId(), bS);
            enumToIdM.put(enumValue, bS.getId());
        }
    }

    @Override
    public List<BatchStatus> statusTypeList() {
        String sql = "SELECT * FROM batchstatus";
        List<BatchStatus> statusL = jdbcTemplate.query(sql, new RowMapper<BatchStatus>() {

            @Override
            public BatchStatus mapRow(ResultSet rs, int rowNum) throws SQLException {
                BatchStatus statusType = new BatchStatus();

                statusType.setId(rs.getInt("id"));
                statusType.setName(rs.getString("name"));
                statusType.setDescription(rs.getString("description"));

                return statusType;
            }

        });

        return statusL;
    }

    @Override
    public BatchStatus getStatusType(int id) {
        return idToTypeM.get(id);
    }

    @Override
    public Integer getStatusFromEnum(BatchStatus.JobStatus js) {
        return enumToIdM.get(js);
    }

}
