package gov.va.cpss.dao.impl;

import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.time.Month;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
//import java.util.Objects;
import java.util.stream.Collectors;

//import org.apache.commons.lang.StringUtils;

import org.springframework.jdbc.core.BatchPreparedStatementSetter;
import org.springframework.jdbc.core.PreparedStatementSetter;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.jdbc.core.RowMapper;

import gov.va.cpss.dao.APSStmtDAO;
import gov.va.cpss.model.AITCDollar;
import gov.va.cpss.model.CBSSAbstractModel;
import gov.va.cpss.model.ProcessStatus.Status;
import gov.va.cpss.model.apps.APSPayment;
import gov.va.cpss.model.apps.APSSitePatient;
import gov.va.cpss.model.apps.APSSiteStmt;
import gov.va.cpss.model.apps.APSStmt;
import gov.va.cpss.model.appsprintack.APPSADFileRecord;

/**
 * An implementation of the APSStmtDAO interface.
 * 
 * Copyright HPE / VA
 * January 17, 2017
 * 
 * @author Yiping Yao
 * @version 1.0.0
 *
 */
@SuppressWarnings("nls")
public class APSStmtDAOImpl extends CBSSBaseDAOImpl implements APSStmtDAO
{	
	public static final String TABLE_NAME = "APSStmt";

	@Override
	public Long getGenCountWithStatus(final long genProcessId, final Status status)
	{
		final String sql = "SELECT COUNT(*) FROM " + TABLE_NAME + " WHERE batchRunProcessIdGen = " + genProcessId + " AND statusId = " + getStatusID(status);

		final List<Long> results = this.jdbcTemplate.query(sql, new RowMapper<Long>()
		{
			@Override
			public Long mapRow(ResultSet rs, int row) throws SQLException
			{
				return new Long(rs.getLong(1));
			}
		});

		if ((results != null) && !results.isEmpty())
		{
			return results.get(0);
		}

		return null;
	}

	@Override
	@SuppressWarnings("unchecked")
	public List<APSStmt> getExistingStatements(final List<APSStmt> inStmts)
	{
		final String sql = "SELECT * FROM " + TABLE_NAME + " WHERE accountId = ? AND statusId = ?";

		List<List<?>> paramsList = new ArrayList<>();

		for (int i = 0; i < inStmts.size(); i++)
		{
			APSStmt stmt = inStmts.get(i);
			List<Object> params = new ArrayList<>();

			params.add(Long.valueOf(stmt.getAccountId()));
			params.add(Integer.valueOf(stmt.getStatusId()));

			paramsList.add(params);
		}

		return (List<APSStmt>) batchSelect(sql, paramsList);
	}

	@Override
	public int[] batchUpdate(List<APSStmt> inStmts)
	{
		final String sql = "UPDATE " + getTableName() + " SET statementDate = ?, totalAmntRcv = ?, processDate = ?, statusId = ? WHERE id = ?";

		return this.jdbcTemplate.batchUpdate(sql, new BatchPreparedStatementSetter()
		{
			@Override
			public void setValues(PreparedStatement ps, int i) throws SQLException
			{
				if (inStmts.get(i).getStatementDate() != null)
				{
					ps.setDate(1, new Date(inStmts.get(i).getStatementDate().getTime()));
				}
				else
				{
					ps.setDate(1, new Date(0));
				}

				ps.setDouble(2, inStmts.get(i).getTotalAmountReceived().getDouble());

				if (inStmts.get(i).getProcessDate() != null)
				{
					ps.setDate(3, new Date(inStmts.get(i).getProcessDate().getTime()));
				}
				else
				{
					ps.setDate(3, new Date(0));
				}

				ps.setInt(4, inStmts.get(i).getStatusId());
				ps.setLong(5, inStmts.get(i).getId());
			}

			@Override
			public int getBatchSize()
			{
				return inStmts.size();
			}
		});
	}

	@Override
	public void deleteByGenProcessId(final long processId)
	{
		final String sql = "DELETE FROM " + TABLE_NAME + " where batchRunProcessIdGen = ?";

		this.jdbcTemplate.update(sql, Long.valueOf(processId));
	}

	@Override
	public int updateStatusByGenProcessId(long id, Status status)
	{
		final String sql = "UPDATE " + TABLE_NAME + " SET statusId = ? where batchRunProcessIdGen = ?";

		return this.jdbcTemplate.update(sql, Integer.valueOf(getStatusID(status)), Long.valueOf(id));
	}

    @Override
    public List<Long> getStatementIdsForPrintAck(final List<APPSADFileRecord> printAckDetailL, final int statementYear,
            final int sentStatusId, final int ackStatusId, final java.util.Date printDate, final long batchRunProcessId)
    {
        final String openingSQL = "SELECT * " +
                                  "FROM APSStmt " + 
                                  "WHERE stmtYear = ? " +
                                  "AND statusId IN (?, ?) " + 
                                  "AND accountId in ( SELECT DISTINCT ID FROM CBSAccount WHERE icn";

        final String closingSQL = ") " +
                                  "AND (printDate IS NULL OR printDate != ?) " +
                                  "AND (batchRunProcessIdAck IS NULL OR batchRunProcessIdAck != ?)";

        Object[] openingParams = new Object[]{Integer.valueOf(statementYear), Long.valueOf(sentStatusId), Long.valueOf(ackStatusId)};
        Object[] closingParams = new Object[]{new java.sql.Date(printDate.getTime()), Long.valueOf(batchRunProcessId)};

        Object[] inClauseParams = printAckDetailL.stream().map(APPSADFileRecord::getPatientAccount).collect(Collectors.toList()).toArray();

        List<? extends CBSSAbstractModel> models = batchSelectWithINClause(openingSQL, closingSQL, openingParams, inClauseParams, closingParams);

        List<Long> statementIDs = null;

        if (models != null && !models.isEmpty())
        {
            statementIDs = models.stream().map(CBSSAbstractModel::getId).collect(Collectors.toList());

            this.logger.debug("StatementIDs.size(): " + statementIDs.size());            
        }
 
        return statementIDs;
    }

	@Override
	public APSStmt get(long id) {
		final String sql = "SELECT * FROM APSStmt WHERE id = " + id;

		final List<APSStmt> APSStmtL = this.jdbcTemplate.query(sql, new APSStmtRowMapper());

		return (APSStmtL.size() > 0) ? APSStmtL.get(0) : null;
	}

	@Override
	public void updatePrintAckFields(final List<Long> apsStmtIdL, final long batchRunProcessIdAck,
			final java.util.Date printDate, final int statusId) {

		final String sql = "UPDATE " + TABLE_NAME
				+ " SET batchRunProcessIdAck = ?, printDate = ?, statusId = ?"
				+ " WHERE id = ? AND (batchRunProcessIdAck = null OR printDate = null OR statusId! = ?"
				+ " OR id = (SELECT stmtId FROM " + APSSiteStmtDAOImpl.TABLE_NAME + " WHERE stmtId = APSStmt.id ))";
		this.logger.debug(sql);

		this.jdbcTemplate.batchUpdate(sql, new BatchPreparedStatementSetter() {

			@Override
			public void setValues(PreparedStatement ps, int i) throws SQLException {
				ps.setLong(1, batchRunProcessIdAck);
				ps.setDate(2, new java.sql.Date(printDate.getTime()));
				ps.setLong(3, statusId);
				ps.setLong(4, apsStmtIdL.get(i).longValue());
				ps.setLong(5, statusId);
			}

			@Override
			public int getBatchSize() {
				return apsStmtIdL.size();
			}
		});
	}

	@Override
	public int[] undoPrintAcknowledgements(final List<Long> ackBatchRunProcessIdList, final int sentStatusId) {
		if ((ackBatchRunProcessIdList == null) || (ackBatchRunProcessIdList.isEmpty()))
			return new int[0];

		final String sql = "UPDATE APSStmt SET statusId=" + sentStatusId
				+ ", printDate=NULL, batchRunProcessIdAck=NULL WHERE batchRunProcessIdAck=? ";

		final int[] rowCounts = new int[ackBatchRunProcessIdList.size()];
		for (int i = 0; i < ackBatchRunProcessIdList.size(); i++) {
			rowCounts[i] = this.jdbcTemplate.update(sql, ackBatchRunProcessIdList.get(i));
		}

		return rowCounts;
	}

	@Override
	public List<APSStmt> getAllStatementsForStatus(final Status status) {
		final String sql = "SELECT * FROM APSStmt WHERE statusId=" + getStatusID(status);

		final List<APSStmt> apsStmtL = this.jdbcTemplate.query(sql, new APSStmtRowMapper());

		return (!apsStmtL.isEmpty()) ? apsStmtL : null;
	}

	@Override
	public int[] updateStatusToSent(List<APSStmt> apsStmtL, long batchRunProcessID) {
		final String sql = "UPDATE " + getTableName() + " SET batchRunProcessIdSend=?, statusId=? WHERE id=?";

		return this.jdbcTemplate.batchUpdate(sql, new BatchPreparedStatementSetter() {

			@Override
			public void setValues(PreparedStatement ps, int i) throws SQLException {
				ps.setLong(1, batchRunProcessID);
				ps.setLong(2, getStatusID(Status.SENT));
				ps.setLong(3, apsStmtL.get(i).getId());
			}

			@Override
			public int getBatchSize() {
				return apsStmtL.size();
			}
		});
	}


	@Override
	public String getTableName()
	{
		return TABLE_NAME;
	}

	@Override
	public String getInsertSQL()
	{
		return  "INSERT INTO " + TABLE_NAME +
				"(accountId, stmtYear, statementDate, totalAmntRcv, processDate, batchRunProcessIdGen, batchRunProcessIdSend, batchRunProcessIdAck," +
				" acntNumDisp,  replacedStmtId, printDate, statusId)" +
				" VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
	}

	@Override
	protected void mapRows(PreparedStatement ps, CBSSAbstractModel model) throws SQLException
	{
		APSStmt statement = (APSStmt) model;

		ps.setLong(1, statement.getAccountId());
		ps.setInt(2, getYear(statement.getStatementDate()));

		if (statement.getStatementDate() != null)
		{
			ps.setDate(3, new Date(statement.getStatementDate().getTime()));
		}
		else
		{
			ps.setDate(3, new Date(0));
		}

		if (statement.getTotalAmountReceived() != null)
		{
			ps.setDouble(4, statement.getTotalAmountReceived().getDouble());
		}
		else
		{
			ps.setDouble(4, 0.0);
		}

		if (statement.getProcessDate() != null)
		{
			ps.setDate(5, new Date(statement.getProcessDate().getTime()));
		}
		else
		{
			ps.setDate(5, new Date(0));
		}

		if (statement.getGenBatchRunId() > 0)
		{
			ps.setLong(6, statement.getGenBatchRunId());
		}
		else
		{
			ps.setNull(6, Types.NUMERIC);
		}

		if (statement.getSendBatchRunId() > 0)
		{
			ps.setLong(7, statement.getSendBatchRunId());
		}
		else
		{
			ps.setNull(7, Types.NUMERIC);
		}

		if (statement.getAckBatchRunId() > 0)
		{
			ps.setLong(8, statement.getAckBatchRunId());
		}
		else
		{
			ps.setNull(8, Types.NUMERIC);
		}

		ps.setString(9,  statement.getAccountNumDisp());
		ps.setLong(10, statement.getReplacedStmtId());

		if (statement.getPrintDate() != null)
		{
			ps.setDate(11, new Date(statement.getPrintDate().getTime()));
		}
		else
		{
			ps.setDate(11, null);
		}

		ps.setInt(12, statement.getStatusId());
	}

	@Override
	protected APSStmt mapResult(ResultSet rs) throws SQLException
	{
		APSStmt statement = new APSStmt();

		statement.setId(rs.getLong(ID));
		statement.setAccountId(rs.getLong("accountId"));
		statement.setStatementDate(rs.getDate("statementDate"));
		statement.setTotalAmountReceived(new AITCDollar(rs.getDouble("totalAmntRcv")));
		statement.setProcessDate(rs.getDate("processDate"));
		statement.setGenBatchRunId(rs.getLong("batchRunProcessIdGen"));
		statement.setSendBatchRunId(rs.getLong("batchRunProcessIdSend"));
		statement.setAckBatchRunId(rs.getLong("batchRunProcessIdAck"));
		statement.setAccountNumDisp(rs.getString("acntNumDisp"));
		statement.setReplacedStmtId(rs.getLong("replacedStmtId"));
		statement.setPrintDate(rs.getDate("printDate"));
		statement.setStatusId(rs.getInt("statusId"));

		// Audit Fields
		mapAuditFields(rs, statement);

		return statement;
	}

	/**
	 * Maps a ResultSet to a new APSStmt object.
	 * 
	 * @author Brad Pickle
	 *
	 */
	public static class APSStmtRowMapper implements RowMapper<APSStmt> {

		@Override
		public APSStmt mapRow(ResultSet rs, int row) throws SQLException {

			APSStmt statement = new APSStmt();

			statement.setId(rs.getLong(ID));
			statement.setAccountId(rs.getLong("accountId"));
			statement.setStatementDate(rs.getDate("statementDate"));
			statement.setTotalAmountReceived(new AITCDollar(rs.getDouble("totalAmntRcv")));
			statement.setProcessDate(rs.getDate("processDate"));
			statement.setGenBatchRunId(rs.getLong("batchRunProcessIdGen"));
			statement.setSendBatchRunId(rs.getLong("batchRunProcessIdSend"));
			statement.setAckBatchRunId(rs.getLong("batchRunProcessIdAck"));
			statement.setAccountNumDisp(rs.getString("acntNumDisp"));
			statement.setReplacedStmtId(rs.getLong("replacedStmtId"));
			statement.setPrintDate(rs.getDate("printDate"));
			statement.setStatusId(rs.getInt("statusId"));

			statement.setCreatedBy(rs.getString("createdBy"));
			statement.setCreatedDate(rs.getDate("createdDate"));
			statement.setModifiedBy(rs.getString("modifiedBy"));
			statement.setModifiedDate(rs.getDate("modifiedDate"));

			return statement;
		}

	}

	@Override
    public List<APSStmt> getApsStmtsWithPaging(final int page, final int pageSize, final int statementYear, final Status status) {

		final StringBuilder sql = new StringBuilder();
		sql.append("SELECT \n");
		sql.append("APSStmt.ID as APSStmt_ID, \n");
		sql.append("APSStmt.ACCOUNTID as APSStmt_ACCOUNTID, \n");
		sql.append("APSStmt.STMTYEAR as APSStmt_STMTYEAR, \n");
		sql.append("APSStmt.STATEMENTDATE as APSStmt_STATEMENTDATE, \n");
		sql.append("APSStmt.TOTALAMNTRCV as APSStmt_TOTALAMNTRCV, \n");
		sql.append("APSStmt.PROCESSDATE as APSStmt_PROCESSDATE, \n");
		sql.append("APSStmt.BATCHRUNPROCESSIDGEN as APSStmt_BATCHRUNPROCESSIDGEN, \n");
		sql.append("APSStmt.BATCHRUNPROCESSIDSEND as APSStmt_BATCHRUNPROCESSIDSEND, \n");
		sql.append("APSStmt.BATCHRUNPROCESSIDACK as APSStmt_BATCHRUNPROCESSIDACK, \n");
		sql.append("APSStmt.ACNTNUMDISP as APSStmt_ACNTNUMDISP, \n");
		sql.append("APSStmt.REPLACEDSTMTID as APSStmt_REPLACEDSTMTID, \n");
		sql.append("APSStmt.PRINTDATE as APSStmt_PRINTDATE, \n");
		sql.append("APSStmt.STATUSID as APSStmt_STATUSID, \n");
		sql.append(" \n");
		sql.append("APSSiteStmt.ID as APSSiteStmt_ID, \n");
		sql.append("APSSiteStmt.STMTID as APSSiteStmt_STMTID, \n");
		sql.append("APSSiteStmt.STATIONNUM as APSSiteStmt_STATIONNUM, \n");
		sql.append("APSSiteStmt.STATIONPHONENUM as APSSiteStmt_STATIONPHONENUM, \n");
		sql.append("APSSiteStmt.STMTYEAR as APSSiteStmt_STMTYEAR, \n");
		sql.append("APSSiteStmt.PROCESSDATE as APSSiteStmt_PROCESSDATE, \n");
		sql.append("APSSiteStmt.TOTALAMNTRCV as APSSiteStmt_TOTALAMNTRCV, \n");
		sql.append("APSSiteStmt.ARADDRESSFLAG as APSSiteStmt_ARADDRESSFLAG, \n");
		sql.append("APSSiteStmt.LASTBILLPREPDATE as APSSiteStmt_LASTBILLPREPDATE, \n");
		sql.append("APSSiteStmt.ISPRIMARY as APSSiteStmt_ISPRIMARY, \n");
		sql.append("APSSiteStmt.ISPRIMARYADDRESS as APSSiteStmt_ISPRIMARYADDRESS, \n");
		sql.append(" \n");
		sql.append("APSSitePatient.ID as APSSitePatient_ID, \n");
		sql.append("APSSitePatient.SITESTMTID as APSSitePatient_SITESTMTID, \n");
		sql.append("APSSitePatient.ICN as APSSitePatient_ICN, \n");
		sql.append("APSSitePatient.DFN as APSSitePatient_DFN, \n");
		sql.append("APSSitePatient.OLDACNTNUM as APSSitePatient_OLDACNTNUM, \n");
		sql.append("APSSitePatient.FIRSTNAME as APSSitePatient_FIRSTNAME, \n");
		sql.append("APSSitePatient.LASTNAME as APSSitePatient_LASTNAME, \n");
		sql.append("APSSitePatient.MIDDLENAME as APSSitePatient_MIDDLENAME, \n");
		sql.append("APSSitePatient.ADDRESS1 as APSSitePatient_ADDRESS1, \n");
		sql.append("APSSitePatient.ADDRESS2 as APSSitePatient_ADDRESS2, \n");
		sql.append("APSSitePatient.ADDRESS3 as APSSitePatient_ADDRESS3, \n");
		sql.append("APSSitePatient.CITY as APSSitePatient_CITY, \n");
		sql.append("APSSitePatient.STATE as APSSitePatient_STATE, \n");
		sql.append("APSSitePatient.ZIPCODE as APSSitePatient_ZIPCODE, \n");
		sql.append("APSSitePatient.COUNTRY as APSSitePatient_COUNTRY, \n");
		sql.append(" \n");
		sql.append("APSPayment.ID as APSPayment_ID, \n");
		sql.append("APSPayment.SITESTMTID as APSPayment_SITESTMTID, \n");
		sql.append("APSPayment.MONTH as APSPayment_MONTH, \n");
		sql.append("APSPayment.DATEPOSTED as APSPayment_DATEPOSTED, \n");
		sql.append("APSPayment.TRANSAMNT as APSPayment_TRANSAMNT, \n");
		sql.append("APSPayment.TRANSAMNTRAW as APSPayment_TRANSAMNTRAW \n");
		sql.append(" \n");
		// Paging occurs only on the APSStmt table to ensure only complete records are obtained.
		sql.append("FROM (SELECT * FROM (SELECT ROWNUM rnum, a.* FROM APSStmt a WHERE stmtYear = ? AND statusId = ? \n");
		sql.append("AND ROWNUM < ?) WHERE rnum >= ?) apsstmt \n");
		sql.append("LEFT JOIN APSSiteStmt ON APSStmt.id = APSSiteStmt.stmtId   \n");
		sql.append("LEFT JOIN APSSitePatient ON APSSiteStmt.id = APSSitePatient.siteStmtId  \n");
		sql.append("LEFT JOIN APSPayment ON APSSiteStmt.id = APSPayment.siteStmtId \n");
		sql.append("  \n");
		sql.append("ORDER BY apsStmt.id, apsSiteStmt.id, apsPayment.id ");
		
		this.logger.debug(sql);

		final Map<Long, APSStmt> stmtMap = new HashMap<>();

		this.jdbcTemplate.query(sql.toString(), new PreparedStatementSetter() {

			@Override
			public void setValues(PreparedStatement ps) throws SQLException {
				APSStmtDAOImpl.this.logger.debug(String.format("Statement year: %d; Status Id: %d (%s)", Integer.valueOf(statementYear), Integer.valueOf(getStatusID(status)), status.toString()));
				
				ps.setLong(1, statementYear);
				ps.setLong(2, getStatusID(status));
				ps.setLong(3, (page * pageSize) + 1);
				ps.setLong(4, ((page - 1) * pageSize) + 1);
			}

		}, new RowCallbackHandler() {
			
			@Override
			public void processRow(final ResultSet rs) throws SQLException {

				// Maintain state info
				final long retrievedStmtId = rs.getLong("APSStmt_ID");
				final long retrievedSiteStmtId = rs.getLong("APSSiteStmt_ID");
				final long retrievedPaymentId = rs.getLong("APSPayment_ID");
				
				// Check if APSStmt exists already, create if not
				if(!stmtMap.containsKey(Long.valueOf(retrievedStmtId))) {
					// If APSStmtStorageBean already has this statement but we found a row that references it, the statement is incomplete
					// and needs to be updated.
					final APSStmt stmt = new APSStmt();
					stmt.setId(retrievedStmtId);
					stmt.setAccountId(rs.getLong("APSStmt_ACCOUNTID"));
					stmt.setStatementDate(rs.getDate("APSStmt_STATEMENTDATE"));
					stmt.setTotalAmountReceived(new AITCDollar(rs.getDouble("APSStmt_TOTALAMNTRCV")));
					stmt.setProcessDate(rs.getDate("APSStmt_PROCESSDATE"));
					stmt.setGenBatchRunId(rs.getLong("APSStmt_BATCHRUNPROCESSIDGEN"));
					stmt.setSendBatchRunId(rs.getLong("APSStmt_BATCHRUNPROCESSIDSEND"));
					stmt.setAckBatchRunId(rs.getLong("APSStmt_BATCHRUNPROCESSIDACK"));
					stmt.setAccountNumDisp(rs.getString("APSStmt_ACNTNUMDISP"));
					stmt.setReplacedStmtId(rs.getLong("APSStmt_REPLACEDSTMTID"));
					stmt.setPrintDate(rs.getDate("APSStmt_PRINTDATE"));
					stmt.setStatusId(rs.getInt("APSStmt_STATUSID"));
					stmt.setSiteStmts(new ArrayList<APSSiteStmt>());
					stmtMap.put(Long.valueOf(retrievedStmtId), stmt);
				} 

				// Store current APS statement for use later
				final APSStmt currentApsStmt = stmtMap.get(Long.valueOf(retrievedStmtId));

				// Create current site statement if it hasn't been initialized yet
				if(!currentApsStmt.getSiteStmts().stream().mapToLong(APSSiteStmt::getId).anyMatch(id -> id == retrievedSiteStmtId)) {
					final APSSiteStmt siteStmt = new APSSiteStmt();
					siteStmt.setId(retrievedSiteStmtId);
					siteStmt.setStatementId(rs.getLong("APSSiteStmt_STMTID"));
					siteStmt.setFacilityNum(rs.getString("APSSiteStmt_STATIONNUM"));
					siteStmt.setFacilityPhoneNum(rs.getString("APSSiteStmt_STATIONPHONENUM"));
					siteStmt.setStatementDate(rs.getDate("APSStmt_STATEMENTDATE"));
					siteStmt.setProcessDate(rs.getDate("APSSiteStmt_PROCESSDATE"));
					siteStmt.setTotalAmountReceived(new AITCDollar(rs.getDouble("APSSiteStmt_TOTALAMNTRCV")));
					siteStmt.setArAddress(rs.getBoolean("APSSiteStmt_ARADDRESSFLAG"));
					siteStmt.setLastBillPrepDate(rs.getDate("APSSiteStmt_LASTBILLPREPDATE"));
					siteStmt.setPrimary(rs.getBoolean("APSSiteStmt_ISPRIMARY"));
					siteStmt.setPrimaryAddress(rs.getBoolean("APSSiteStmt_ISPRIMARYADDRESS"));
					siteStmt.setPayments(new ArrayList<APSPayment>());

					final APSSitePatient sitePatient = new APSSitePatient();
					sitePatient.setId(rs.getLong("APSSitePatient_ID"));
					sitePatient.setSiteStmtId(rs.getLong("APSSitePatient_SITESTMTID"));
					sitePatient.setIcn(rs.getString("APSSitePatient_ICN"));
					sitePatient.setDfn(rs.getLong("APSSitePatient_DFN"));
					sitePatient.setAccountNumber(rs.getString("APSSitePatient_OLDACNTNUM"));
					sitePatient.setFirstName(rs.getString("APSSitePatient_FIRSTNAME"));
					sitePatient.setMiddleName(rs.getString("APSSitePatient_MIDDLENAME"));
					sitePatient.setLastName(rs.getString("APSSitePatient_LASTNAME"));
					sitePatient.setAddress1(rs.getString("APSSitePatient_ADDRESS1"));
					sitePatient.setAddress2(rs.getString("APSSitePatient_ADDRESS2"));
					sitePatient.setAddress3(rs.getString("APSSitePatient_ADDRESS3"));
					sitePatient.setCity(rs.getString("APSSitePatient_CITY"));
					sitePatient.setState(rs.getString("APSSitePatient_STATE"));
					sitePatient.setZipCode(rs.getString("APSSitePatient_ZIPCODE"));
					sitePatient.setCountry(rs.getString("APSSitePatient_COUNTRY"));

					siteStmt.setPatient(sitePatient);

					currentApsStmt.getSiteStmts().add(siteStmt);
				}

				// Store current site statement for later
				final APSSiteStmt currentSiteStatement = currentApsStmt.getSiteStmts().stream().filter(siteStmt -> siteStmt.getId() == retrievedSiteStmtId).findFirst().get();

				// Initialize payment. Each row should have a unique payment due to being the leaf of the tree structure, but validate just in case
				if(!currentSiteStatement.getPayments().stream().anyMatch(payment -> payment.getId() == retrievedPaymentId)) {
					final APSPayment payment = new APSPayment();
					payment.setId(retrievedPaymentId);
					payment.setSiteStmtId(rs.getLong("APSPayment_SITESTMTID"));
					payment.setMonth(Month.of(rs.getInt("APSPayment_MONTH")));
					payment.setDatePosted(rs.getDate("APSPayment_DATEPOSTED"));
					payment.setTransactionAmount(new AITCDollar(rs.getDouble("APSPayment_TRANSAMNT")));
					payment.setTransactionAmountStr(rs.getString("APSPayment_TRANSAMNTRAW"));
					
					currentSiteStatement.getPayments().add(payment);
				}
			}
		});
		
		this.logger.debug(String.format("Retrieved %d APSStmts", Integer.valueOf(stmtMap.values().size())));
		
		return new ArrayList<>(stmtMap.values());
	}
}
