package gov.va.cpss.dao.impl;

import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import org.springframework.jdbc.core.BatchPreparedStatementSetter;
import org.springframework.jdbc.core.RowMapper;

import gov.va.cpss.dao.APSSiteStmtDAO;

import gov.va.cpss.model.AITCDollar;
import gov.va.cpss.model.BooleanChar;
import gov.va.cpss.model.CBSSAbstractModel;
import gov.va.cpss.model.apps.APSSiteStmt;

/**
 * An implementation of the APSSiteStmtDAO interface.
 * 
 * Copyright HPE / VA
 * January 18, 2017
 * 
 * @author Yiping Yao
 * @version 1.0.0
 *
*/
@SuppressWarnings("nls")
public class APSSiteStmtDAOImpl extends CBSSBaseDAOImpl implements APSSiteStmtDAO
{
    public static final String TABLE_NAME = "APSSiteStmt";


    @Override
    @SuppressWarnings("unchecked")
    public List<APSSiteStmt> getExistingSiteStatements(final Long[] inIDs)
    {
        final String openingSQL = getSelectSQL() + " WHERE stmtId";

        return (List<APSSiteStmt>) batchSelectWithINClause(openingSQL, null, null, inIDs, null);
    }

    @Override
    public int[] batchUpdate(List<APSSiteStmt> inSiteStmts)
    {
        String sql = "UPDATE " + getTableName() + " SET isPrimary = ?, isPrimaryAddress = ?" + " WHERE id = ?";

        return this.jdbcTemplate.batchUpdate(sql, new BatchPreparedStatementSetter()
        {
            @Override
            public void setValues(PreparedStatement ps, int i) throws SQLException
            {
                ps.setString(1, BooleanChar.from(inSiteStmts.get(i).isPrimary()).toString());
                ps.setString(2, BooleanChar.from(inSiteStmts.get(i).isPrimaryAddress()).toString());
                ps.setLong(3, inSiteStmts.get(i).getId());
            }

            @Override
            public int getBatchSize()
            {
                return inSiteStmts.size();
            }
        });
    }
    
    private class APSSiteStmtRowMapper implements RowMapper<APSSiteStmt>
    {    
        private boolean loadPatient;
        private APSSitePatientDAOImpl.APSSitePatientRowMapper patientRowMapper;
        
        public APSSiteStmtRowMapper(boolean inLoadPatient) {
            this.loadPatient = inLoadPatient;
            this.patientRowMapper = inLoadPatient ? new APSSitePatientDAOImpl.APSSitePatientRowMapper() : null;
        }
   
        @Override
        public APSSiteStmt mapRow(ResultSet rs, int row) throws SQLException
        {
            final APSSiteStmt appsSiteStmt = (APSSiteStmt) mapResult(rs);
            
            if(this.loadPatient) {
                appsSiteStmt.setPatient(this.patientRowMapper.mapRow(rs, row));
            }
            
            return appsSiteStmt;
        }
    }

    @Override
    public List<APSSiteStmt> getStmtSitesForStmt(long apsStmtId) {
        final String sql = "SELECT * FROM APSSiteStmt WHERE stmtId=" + apsStmtId + " ORDER BY isPrimary DESC";

        final List<APSSiteStmt> results = this.jdbcTemplate.query(sql, new APSSiteStmtRowMapper(false));

        return results;
    }


    @Override
    public String getTableName()
    {
        return TABLE_NAME;
    }

    @Override
    public String getInsertSQL()
    {
        return  "INSERT INTO " + TABLE_NAME +
                " (stmtId, stationNum, stationPhoneNum, stmtYear, processDate, totalAmntRcv, arAddressFlag, lastBillPrepDate, isPrimary, isPrimaryAddress) " +
                " VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    }

    @Override
    protected void mapRows(PreparedStatement ps, CBSSAbstractModel model) throws SQLException
    {
        APSSiteStmt siteStatement = (APSSiteStmt) model;

        ps.setLong(1, siteStatement.getStatementId());
        ps.setString(2, siteStatement.getFacilityNum());
        ps.setString(3, siteStatement.getFacilityPhoneNum());
        ps.setInt(4, getYear(siteStatement.getStatementDate()));
        ps.setDate(5, new Date(siteStatement.getProcessDate().getTime()));
        ps.setDouble(6, siteStatement.getTotalAmountReceived().getDouble());
        ps.setString(7, BooleanChar.from(siteStatement.isArAddress()).toString());
        ps.setDate(8, new Date(siteStatement.getLastBillPrepDate().getTime()));
        ps.setString(9, BooleanChar.from(siteStatement.isPrimary()).toString());
        ps.setString(10, BooleanChar.from(siteStatement.isPrimaryAddress()).toString());
    }

    @Override
    protected CBSSAbstractModel mapResult(ResultSet rs) throws SQLException
    {
        APSSiteStmt siteStatement = new APSSiteStmt();

        siteStatement.setId(rs.getLong(ID));
        siteStatement.setStatementId(rs.getLong("stmtId"));
        siteStatement.setFacilityNum(rs.getString("stationNum"));
        siteStatement.setFacilityPhoneNum(rs.getString("stationPhoneNum"));
        siteStatement.setStatementDate(getDate(rs.getInt("stmtYear")));
        siteStatement.setProcessDate(rs.getDate("processDate"));
        siteStatement.setTotalAmountReceived(new AITCDollar(rs.getDouble("totalAmntRcv")));
        siteStatement.setArAddress(BooleanChar.from(rs.getString("arAddressFlag")).isTrue());
        siteStatement.setLastBillPrepDate(rs.getDate("lastBillPrepDate"));
        siteStatement.setPrimary(BooleanChar.from(rs.getString("isPrimary")).isTrue());
        siteStatement.setPrimaryAddress(BooleanChar.from(rs.getString("isPrimaryAddress")).isTrue());

        // Audit fields
        mapAuditFields(rs, siteStatement);

        return siteStatement;
    }

}
