package gov.va.cpss.dao.impl;

import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.stream.Collectors;

import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.BatchPreparedStatementSetter;
import org.springframework.jdbc.core.PreparedStatementSetter;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowMapper;

import gov.va.cpss.dao.APSSiteDAO;

import gov.va.cpss.model.CBSSAbstractModel;

import gov.va.cpss.model.apps.APSPatient;
import gov.va.cpss.model.apps.APSSite;

/**
 * 
 * An implementation of the APSSiteDAO interface.
 * 
 * Copyright HPE / VA
 * October 27, 2016
 * 
 * @author Yiping Yao
 * @version 1.0.0
 * 
 */
@SuppressWarnings("nls")
public class APSSiteDAOImpl extends CBSSBaseDAOImpl implements APSSiteDAO
{
    public static final String TABLE_NAME = "APSSite";

	@Override
	public void deleteByReceivedId(final long receivedId)
	{
		final String sql = "DELETE FROM " + TABLE_NAME + " WHERE apsReceivedId = ?";

		this.jdbcTemplate.update(sql, Long.valueOf(receivedId));
	}

	@Override
    public int[] deleteByFacilityNumbers(final List<String> facilityNumbers)
    {
	    final String deleteSQL = "DELETE FROM " + TABLE_NAME + " WHERE facilityNum = ?";

	    return this.jdbcTemplate.batchUpdate(deleteSQL, new BatchPreparedStatementSetter()
        {
             @Override
             public void setValues(PreparedStatement ps, int i) throws SQLException
             {
                 ps.setString(1, facilityNumbers.get(i));
             }

             @Override
             public int getBatchSize()
             {
                 return facilityNumbers.size();
             }
        });
    }

	@Override
	public APSSite getById(final long id)
	{
		// query
		final String sql = "SELECT * FROM " + TABLE_NAME + " WHERE id = " + id;

		return this.jdbcTemplate.query(sql, new ResultSetExtractor<APSSite>()
		{
			@Override
			public APSSite extractData(ResultSet rs) throws SQLException, DataAccessException
			{
				if (rs.next())
				{
					return mapResult(rs);
				}

				return null;
			}
		});
	}

	@Override
	public List<APSSite> getAllByAPSReceivedId(final long receivedId)
	{
	    final String sql = "SELECT * FROM " + TABLE_NAME + " WHERE apsReceivedId = " + receivedId + ORDER_BY_ID_CLAUSE;

		List<APSSite> apsSites = this.jdbcTemplate.query(sql, new RowMapper<APSSite>()
		{
			@Override
			public APSSite mapRow(ResultSet rs, int rowNum) throws SQLException
			{
				return mapResult(rs);
			}

		});

		return apsSites;
	}

	@Override
	public List<Long> getAllIdByAPSReceivedId(final long receivedId)
	{

		final String sql = "SELECT " + ID + " FROM " + TABLE_NAME + " WHERE apsReceivedId = ?" + ORDER_BY_ID_CLAUSE;

		List<Long> ids = this.jdbcTemplate.query(sql, new PreparedStatementSetter()
		{
			@Override
            public void setValues(PreparedStatement preparedStatement) throws SQLException
			{
				preparedStatement.setLong(1, receivedId);
			}
		},
		new RowMapper<Long>()
		{
			@Override
			public Long mapRow(ResultSet rs, int row) throws SQLException
			{
				return new Long(rs.getLong(1));
			}
		});

		return ids;
	}

    @Override
    public List<APSSite> getPaymentReceivedSites(final int statementYear, final Date processDate)
    {
        //String sql = "SELECT * FROM APSSite WHERE stmtYear = ? AND processDate >= ? ORDER BY facilityNum, id";
        String sql = "SELECT site.id, site.apsReceivedId, site.seqNum, site.totSeqNum, site.facilityNum, " +
                            "site.facilityPhoneNum, site.totPatient, site.stmtYear, site.processDate, " +
                            "site.createdBy, site.createdDate, site.modifiedBy, site.modifiedDate " +
                     "FROM ( SELECT id, apsReceivedId, seqNum, totSeqNum, facilityNum, " +
                             "facilityPhoneNum, totPatient, stmtYear, processDate, " +
                             "createdBy, createdDate, modifiedBy, modifiedDate, " +
                             "ROW_NUMBER() OVER(PARTITION BY facilityNum ORDER BY processDate, id) rowNumber " +
                             "FROM " + TABLE_NAME + " WHERE stmtYear = ? AND processDate >= ? " +
                          ") site WHERE rowNumber = 1";

        List<APSSite> apsSites = this.jdbcTemplate.query(sql, new Object[]{Integer.valueOf(statementYear), processDate}, new RowMapper<APSSite>()
        {
            @Override
            public APSSite mapRow(ResultSet rs, int rowNum) throws SQLException
            {
                return mapResult(rs);
            }

        });

        return apsSites;
    }

    @Override
    @SuppressWarnings("unchecked")
    public List<APSSite> getByPatients(List<APSPatient> patients)
    {
        if (patients == null || patients.size() == 0)
        {
            return null;
        }

        // Example SQL
        // SELECT * FROM APSSite s WHERE s.id IN (SELECT p.apsSiteId FROM APSPatient p WHERE p.id IN (85, 86, 87, 88, 89, 90, 91));
        //"SELECT * FROM " + TABLE_NAME + " WHERE " + KEY + " IN (SELECT p.apsSiteId FROM " + APSPatientDAOImpl.TABLE_NAME + " p WHERE p.id " + buildInClause(size) + ")"; 

        final String openingSQL = getSelectSQL() +
                                  " WHERE " + ID +
                                  " IN (SELECT p.apsSiteId FROM " +
                                  APSPatientDAOImpl.TABLE_NAME +
                                  " p WHERE p.id";

        // Extract patients ID's
        List<Long> ids = patients.stream().map(p -> Long.valueOf(p.getId())).collect(Collectors.toList());

        return (List<APSSite>) batchSelectWithINClause(openingSQL, ")", null, ids.toArray(), null);
    }

    @Override
    public String getTableName()
    {
        return TABLE_NAME;
    }

    // Insert SQL string
    @Override
    public final String getInsertSQL()
    {
        return "INSERT " + PARALLEL_HINT + " INTO " + TABLE_NAME +
               "(apsReceivedId, seqNum, totSeqNum, facilityNum, facilityPhoneNum, totPatient, stmtYear, processDate)" +
               " VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
    }

    @Override
    protected void mapRows(PreparedStatement ps, CBSSAbstractModel model) throws SQLException
    {
        APSSite site = (APSSite) model;

        ps.setLong(1, site.getApsReceivedId());
        ps.setLong(2, site.getSeqNum());
        ps.setLong(3, site.getTotSeqNum());
        ps.setString(4, site.getFacilityNum());
        ps.setString(5, site.getFacilityPhoneNum());
        ps.setInt(6, site.getTotalPatient());
        ps.setInt(7, getYear(site.getStatementDate()));
        ps.setTimestamp(8, site.getProcessDate());
    }

    @Override
    protected APSSite mapResult(ResultSet rs) throws SQLException
    {
        APSSite apsSite = new APSSite(rs.getLong(ID));

        apsSite.setApsReceivedId(rs.getLong("apsReceivedId"));
        apsSite.setSeqNum(rs.getLong("seqNum"));
        apsSite.setTotSeqNum(rs.getLong("totSeqNum"));
        apsSite.setFacilityNum(rs.getString("facilityNum"));
        apsSite.setFacilityPhoneNum(rs.getString("facilityPhoneNum"));
        apsSite.setTotalPatient(rs.getInt("totPatient"));
        apsSite.setStatementDate(getDate(rs.getInt("stmtYear")));
        apsSite.setProcessDate(rs.getTimestamp("processDate"));

        // Audit fields
        mapAuditFields(rs, apsSite);

        return apsSite;
    }

}
