package gov.va.cpss.dao.impl;

import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import java.util.List;
import java.util.Map.Entry;

import org.springframework.jdbc.core.BatchPreparedStatementSetter;
import org.springframework.jdbc.core.RowMapper;

import gov.va.cpss.dao.APSPatientDAO;

import gov.va.cpss.model.AITCDollar;
import gov.va.cpss.model.BooleanChar;
import gov.va.cpss.model.CBSSAbstractModel;
import gov.va.cpss.model.ProcessStatus.Status;

import gov.va.cpss.model.apps.APSPatient;
import gov.va.cpss.model.apps.APSSite;

/**
 * 
 * An implementation of the APSPatientDAO interface.
 * 
 * Copyright HPE / VA
 * October 27, 2016
 * 
 * @author Yiping Yao
 * @version 1.0.0
 * 
 */
@SuppressWarnings("nls")
public class APSPatientDAOImpl extends CBSSBaseDAOImpl implements APSPatientDAO
{
    public static final String TABLE_NAME = "APSPatient";

    @Override
	public List<APSPatient> getAllByAPSSite(final APSSite apsSite)
	{
		if (apsSite != null)
		{
			final String sql = "SELECT * FROM " + TABLE_NAME + " WHERE apsSiteId = " + apsSite.getId() + ORDER_BY_ID_CLAUSE;

			List<APSPatient> patientL = this.jdbcTemplate.query(sql, new RowMapper<APSPatient>()
			{
				@Override
				public APSPatient mapRow(ResultSet rs, int rowNum) throws SQLException
				{
			        APSPatient patient = (APSPatient) mapResult(rs);

			        patient.setApsSite(apsSite);

			        return patient;
				}
			});

			return patientL;
		}

		return null;
	}

    @Override
    @SuppressWarnings("unchecked")
    public List<APSPatient> getPatientsWithPaging(int page, int pageSize, int statementYear, Status status)
    {
        // SQL:
        // SELECT * FROM APSPatient WHERE apsSiteId IN (SELECT id FROM APSSite WHERE stmtYear = 2016 AND apsReceivedId IN (SELECT id FROM APSReceived WHERE statusId = 2)) ORDER BY id;;
/*
        final String selectSQL = " SELECT *  FROM " + TABLE_NAME + " WHERE apsSiteId IN " +
                                 "(SELECT id FROM " + APSSiteDAOImpl.TABLE_NAME + " WHERE stmtYear = " + statementYear + " AND apsReceivedId IN " +
                                 "(SELECT id FROM " + APSReceivedDAOImpl.TABLE_NAME + " WHERE statusId = " + getStatusID(status) + ")) " +
                                 ORDER_BY_ID_CLAUSE;

        return (List<APSPatient>) selectWithPaging(page, pageSize, selectSQL);
*/
        final String selectSQL = "  SELECT      *     FROM " +                    TABLE_NAME + " WHERE apsSiteId IN " +
                                 "( SELECT " + ID + " FROM " +     APSSiteDAOImpl.TABLE_NAME + " WHERE stmtYear = ? AND apsReceivedId IN " +
                                 "( SELECT " + ID + " FROM " + APSReceivedDAOImpl.TABLE_NAME + " WHERE statusId = ? ) )" +
                                 ORDER_BY_ID_CLAUSE;
/*
        final String selectSQL = " SELECT * FROM " + TABLE_NAME + // Table 1
                                 " INNER JOIN " + APSSiteDAOImpl.TABLE_NAME + // Table 2
                                 " ON " + TABLE_NAME + "." + "apsSiteId = " + APSSiteDAOImpl.TABLE_NAME + "." + ID + // Table 1 FK = Table 2 PK 
                                 " INNER JOIN " + APSReceivedDAOImpl.TABLE_NAME + // Table 3
                                 " ON " + APSSiteDAOImpl.TABLE_NAME + "." + "apsReceivedId = " + APSReceivedDAOImpl.TABLE_NAME + "." + ID + // Table 2 FK = Table 3 PK
                                 " WHERE " + APSSiteDAOImpl.TABLE_NAME + "." + "stmtYear = ? AND " + APSReceivedDAOImpl.TABLE_NAME + "." + "statusId = ?" +
                                 " ORDER BY " + TABLE_NAME + "." + ID;
*/
        return (List<APSPatient>) selectWithPaging(page, pageSize, selectSQL, new Object[]{Integer.valueOf(statementYear), Integer.valueOf(getStatusID(status))});
    }

    @Override
    public int[] batchInsertWithoutKeys(List<Entry<APSSite, APSPatient>> patients)
    {
        final String sql = "INSERT " + PARALLEL_HINT + " INTO " + TABLE_NAME +
                           "(apsSiteId, icn, dfn, oldAcntNum, firstName, lastName, middleName, address1, address2, address3," +
                           " city, state, zipCode, country, totAmntRcv, arAddressFlag, lastBillPrepDate, numOfPD) " + 
                           "SELECT " + FIRST_ROWS_HINT + " " + APSSiteDAOImpl.TABLE_NAME + "." + ID + 
                           ", ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ? " +
                           "FROM " + APSSiteDAOImpl.TABLE_NAME +
                           " WHERE apsReceivedId = ? AND seqNum = ? AND totSeqNum = ? AND facilityNum = ? AND totPatient = ? AND stmtYear = ? AND processDate = ?";
                           //ORDER_BY_ID_CLAUSE + " DESC";

        return this.jdbcTemplate.batchUpdate(sql, new BatchPreparedStatementSetter()
        {
             @Override
             public void setValues(PreparedStatement ps, int i) throws SQLException
             {
                 Entry<APSSite, APSPatient> sitePatient = patients.get(i);
                 mapRows(ps, sitePatient);
             }

             @Override
             public int getBatchSize()
             {
                 return patients.size();
             }
        });
    }

    /**
     * Map rows with site and patient.
     * 
     * @param ps
     * @param sitePatient
     * @throws SQLException
     */
    @SuppressWarnings("static-method")
    protected void mapRows(PreparedStatement ps, Entry<APSSite, APSPatient> sitePatient) throws SQLException
    {
        APSSite site = sitePatient.getKey();
        APSPatient patient = sitePatient.getValue();

        // Patient
        ps.setString(1, patient.getIcn());
        ps.setLong(2, patient.getDfn());
        ps.setString(3, patient.getAccountNumber());
        ps.setString(4, patient.getFirstName());
        ps.setString(5, patient.getLastName());
        ps.setString(6, patient.getMiddleName());
        ps.setString(7, patient.getAddress1());
        ps.setString(8, patient.getAddress2());
        ps.setString(9, patient.getAddress3());
        ps.setString(10, patient.getCity());
        ps.setString(11, patient.getState());
        ps.setString(12, patient.getZipCode());
        ps.setString(13, patient.getCountry());
        ps.setDouble(14, patient.getTotalAmountReceived().getDouble());
        ps.setString(15, (patient.isArAddress()? "Y" : "N"));
        ps.setDate(16, (patient.getLastBillPrepDate() == null)? null : new Date(patient.getLastBillPrepDate().getTime()));
        ps.setInt(17, patient.getNumOfPD());

        // Site
        ps.setLong(18, site.getApsReceivedId());
        ps.setLong(19, site.getSeqNum());
        ps.setLong(20, site.getTotSeqNum());
        ps.setString(21, site.getFacilityNum());
        ps.setInt(22, site.getTotalPatient());
        ps.setInt(23, getYear(site.getStatementDate()));
        ps.setTimestamp(24, site.getProcessDate());
    }

    // Insert SQL string
    @Override
    public final String getInsertSQL()
    {
        return "INSERT " + PARALLEL_HINT + " INTO " + TABLE_NAME +
               "(apsSiteId, icn, dfn, oldAcntNum, firstName, lastName, middleName, address1, address2, address3, city, state, zipCode, country, totAmntRcv, arAddressFlag, lastBillPrepDate, numOfPD) " + 
               "VALUES ( ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ? )";
    }

	@Override
    public final String getTableName()
    {
        return TABLE_NAME;
    }

    @Override
    protected void mapRows(PreparedStatement ps, CBSSAbstractModel model) throws SQLException
    {
        APSPatient patient = (APSPatient) model;

        ps.setLong(1, patient.getApsSite().getId());
        ps.setString(2, patient.getIcn());
        ps.setLong(3, patient.getDfn());
        ps.setString(4, patient.getAccountNumber());
        ps.setString(5, patient.getFirstName());
        ps.setString(6, patient.getLastName());
        ps.setString(7, patient.getMiddleName());
        ps.setString(8, patient.getAddress1());
        ps.setString(9, patient.getAddress2());
        ps.setString(10, patient.getAddress3());
        ps.setString(11, patient.getCity());
        ps.setString(12, patient.getState());
        ps.setString(13, patient.getZipCode());
        ps.setString(14, patient.getCountry());
        ps.setDouble(15, patient.getTotalAmountReceived().getDouble());
        ps.setString(16, (patient.isArAddress()? "Y" : "N"));
        ps.setDate(17, (patient.getLastBillPrepDate() == null)? null : new Date(patient.getLastBillPrepDate().getTime()));
        ps.setInt(18, patient.getNumOfPD());
    }

    @Override
    protected CBSSAbstractModel mapResult(ResultSet rs) throws SQLException
    {
        APSPatient patient = new APSPatient();

        patient.setId(rs.getLong(ID));
        patient.setApsSite(new APSSite(rs.getLong("apsSiteId")));
        patient.setIcn(rs.getString("icn"));
        patient.setDfn(rs.getLong("dfn"));
        patient.setAccountNumber(rs.getString("oldAcntNum"));
        patient.setFirstName(rs.getString("firstName"));
        patient.setLastName(rs.getString("lastName"));
        patient.setMiddleName(rs.getString("middleName"));
        patient.setAddress1(rs.getString("address1"));
        patient.setAddress2(rs.getString("address2"));
        patient.setAddress3(rs.getString("address3"));
        patient.setCity(rs.getString("city"));
        patient.setState(rs.getString("state"));
        patient.setZipCode(rs.getString("zipCode"));
        patient.setCountry(rs.getString("country"));
        patient.setTotalAmountReceived(new AITCDollar(rs.getDouble("TotAmntRcv")));
        patient.setArAddress((BooleanChar.from(rs.getString("arAddressFlag")).isTrue()));
        patient.setLastBillPrepDate(rs.getDate("lastBillPrepDate"));
        patient.setNumOfPD(rs.getInt("numOfPD"));

        // Audit fields
        mapAuditFields(rs, patient);

        return patient;
    }

}
