package gov.va.cpss.dao;

import java.util.List;

import gov.va.cpss.model.cbs.CBSSiteTrans;
import gov.va.cpss.model.cbs.CBSStmt;

/**
 * Data Access Object interface for the CBSSiteTrans table.
 * 
 * @author Brad Pickle
 */
public interface CBSSiteTransDAO {

	/**
	 * Get the List of statement transactions (CBSSiteTrans) for the given
	 * CBSMessage messageId and primary site designated by stationNum.
	 * 
	 * @param cbsMessageId
	 *            return CBSSiteTrans records for CBSStmt records with this
	 *            messageId
	 * @param stationNum
	 *            return CBSSiteTrans records for CBSStmt records with a primary
	 *            station designated by this station number
	 * @return List<CBSSiteTrans> for all statements for the given messageId and
	 *         stationNum
	 */
	List<CBSSiteTrans> getMessageStatementTransForSite(long cbsMessageId, String stationNum);

	/**
	 * Get the List of statement transactions (CBSSiteTrans) associated with the
	 * given consolidated statement ID. The caller will have to sort the list
	 * into lists for their associated site statements.
	 * 
	 * @param stmtId
	 *            return CBSSiteStmt records for CBSStmt records with this
	 *            CBSStmt id
	 * @return List<CBSSiteStmt> for all statements for the given stmtId
	 */
	List<CBSSiteTrans> getAllByCBSSiteStmtID(long stmtId);

	/**
	 * Save the specified CBSSiteTrans to the database.
	 * 
	 * @param cbsSiteTrans
	 *            The CBSSiteTrans to save. The id, createdBy, createdDate,
	 *            modifiedBy and modifiedDate properties will be updated to
	 *            match the database state after the insert.
	 * @return Auto-generated index ID of the entry in the database.
	 */
	
	long save(CBSSiteTrans cbsSiteTrans);
	
	/**
	 * Batch save the nested CBSSiteTrans contained within the list of
	 * CBSStmts to the database. Unlike the save() method, none of the fields
	 * populated by the database will be updated (id will not be set). The
	 * CBSSiteStmt record id is assumed to be populated.
	 * 
	 * @param statements
	 *            The list of statements to save
	 */
	void saveBatch(List<CBSStmt> statements);

	/**
	 * Get CBSSiteTrans with the given id
	 * 
	 * @param id
	 *            id of the CBSSiteTrans to get
	 * @return CBSSiteTrans with given id
	 */
	CBSSiteTrans get(long id);

}
