package gov.va.cpss.cron;

/**
 * Utility methods for converting to and from cron expressions and time in
 * HH:mm:ss.
 */
public class CronUtils {

    /**
     * The delimiter for time formatting.
     */
    private static final String TIME_DELIMITER = ":";

    /**
     * The delimiter for cron formatting.
     */
    private static final String CRON_DELIMITER = " ";

    /**
     * The default suffix for daily cron expressions. Accepts all values for Day
     * of Month (*), Month (*), and no specific value for Day of Week (?). The
     * year field is not mandatory and thus not specified.
     */
    private static final String DAILY_CRON_SUFFIX = "* * ?";

    /**
     * The hour position of the cron expression.
     */
    private static final int CRON_HOUR_POSITION = 2;

    /**
     * Create a daily cron expression from a given HH:mm:ss.
     * 
     * @param time
     *            the time to parse
     * @return a cron daily expression for to run at the specified time
     */
    public static String getCronExpression(final String time) {

        StringBuilder cronExpressionBuilder = new StringBuilder();

        final String[] tokens = time.split(TIME_DELIMITER);

        for (int i = tokens.length - 1; i >= 0; --i) {
            cronExpressionBuilder.append(tokens[i]);
            cronExpressionBuilder.append(CRON_DELIMITER);
        }

        cronExpressionBuilder.append(DAILY_CRON_SUFFIX);

        return cronExpressionBuilder.toString();
    }

    /**
     * Create a time in HH:mm:ss from a daily cron expression.
     * 
     * @param cron
     *            the cron expression to obtail the run time
     * @return a time for which the cron expression runs
     */
    public static String getTimeFromCronExpression(final String cron) {

        StringBuilder timeExpressionBuilder = new StringBuilder();

        final String[] tokens = cron.split(CRON_DELIMITER);

        for (int i = CRON_HOUR_POSITION; i >= 0; --i) {

            timeExpressionBuilder.append(tokens[i]);
            if (i != 0) {
                timeExpressionBuilder.append(TIME_DELIMITER);
            }
        }

        return timeExpressionBuilder.toString();
    }

}
