package gov.va.cpss.service;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assume.assumeTrue;

import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import javax.sql.DataSource;

import org.apache.log4j.Logger;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.dao.CBSStmtDAO;
import gov.va.cpss.dao.PrintAckRecDAO;
import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.dao.impl.CBSDAOImplUtils;
import gov.va.cpss.dao.impl.CBSDAOTestWrapper;
import gov.va.cpss.dao.impl.ProcessStatusDAOImpl;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.cbs.CBSStmt;
import gov.va.cpss.model.printack.ADFileRecord;
import gov.va.cpss.model.printack.PrintAckFileRecord;
import gov.va.cpss.model.printack.PrintAckRec;

/**
 * Unit tests the UpdatePrintAckService class
 * 
 * @author Brad Pickle
 *
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml", "/dao-test-context.xml" })
public class UpdatePrintAckServiceTest {

	@Value("${run.integration.test:false}")
	private Boolean runIntegrationTest;

	@SuppressWarnings("unused")
	private Logger logger = Logger.getLogger(this.getClass().getCanonicalName());

	@Autowired
	private CBSDAOImplUtils cbsDAOImplUtils;

	@Autowired
	private CBSStmtDAO cbsStmtDao;

	@Autowired
	private PrintAckRecDAO printAckRecDao;

	@Autowired
	private UpdatePrintAckService updatePrintAckService;

	private JdbcTemplate jdbcTemplate;

	@Autowired
	public void setDataSource(DataSource dataSource) {
		jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/**
	 * Only run these tests if property is set to run integration test.
	 */
	@Before
	public final void beforeTest() {
		assumeTrue(runIntegrationTest);
	}

	/**
	 * Tests the initializePrintAck() method of UpdatePrintAckService. Reading
	 * PrintAckStatus.Status.INITIAL will fail.
	 */
	@Test(expected = RuntimeException.class)
	public final void testInitializePrintAckNoStatus() {
		(new UpdatePrintAckServiceTestWrapper() {

			@Override
			protected void testImpl() {
				failOnPrintAckStatus(ProcessStatus.Status.INITIAL);

				updatePrintAckService.initializePrintAck((PrintAckRec) null);
			}

		}).run(jdbcTemplate);
	}

	/**
	 * Tests the setPrintAckSuccess() method of UpdatePrintAckService. Reading
	 * PrintAckStatus.Status.SUCCESS will fail.
	 */
	@Test(expected = RuntimeException.class)
	public final void testSetPrintAckSuccessNoStatus() {
		(new UpdatePrintAckServiceTestWrapper() {

			@Override
			protected void testImpl() {
				failOnPrintAckStatus(ProcessStatus.Status.SUCCESS);

				updatePrintAckService.setPrintAckSuccess((PrintAckRec) null);
			}

		}).run(jdbcTemplate);
	}

	/**
	 * Tests the initializePrintAck() and setPrintAckSuccess() methods of
	 * UpdatePrintAckService.
	 */
	@Test
	public final void testInitializeSuccessPrintAck() {
		(new CBSDAOTestWrapper() {

			@Override
			protected void test() {
				final Timestamp dateReceived = new Timestamp(Calendar.getInstance().getTime().getTime());
				final String stationNum = "1";
				final int batchRunId = 1;
				final String fileName = "printAck.txt";

				final PrintAckRec printAckRec = new PrintAckRec();
				printAckRec.setDateReceived(dateReceived);
				printAckRec.setStationNum(stationNum);
				printAckRec.setFileName(fileName);
				printAckRec.setBatchRunId(batchRunId);

				final PrintAckRec originalPrintAckRec = cbsDAOImplUtils.copyPrintAckRec(printAckRec);

				final long id = updatePrintAckService.initializePrintAck(printAckRec);
				getPrintAckRecIdS().add(id);

				assertEquals(id, printAckRec.getId());
				assertTrue(printAckRec.getId() > 0);
				assertEquals(CBSDAOImplUtils.STATUS_INITIAL_ID, printAckRec.getStatusId());
				assertNotNull(printAckRec.getCreatedBy());
				assertNotNull(printAckRec.getCreatedDate());
				assertNotNull(printAckRec.getModifiedBy());
				assertNotNull(printAckRec.getModifiedDate());

				// Fields not set before save
				originalPrintAckRec.setId(printAckRec.getId());
				originalPrintAckRec.setStatusId(printAckRec.getStatusId());
				originalPrintAckRec.setCreatedBy(printAckRec.getCreatedBy());
				originalPrintAckRec.setCreatedDate(printAckRec.getCreatedDate());
				originalPrintAckRec.setModifiedBy(printAckRec.getModifiedBy());
				originalPrintAckRec.setModifiedDate(printAckRec.getModifiedDate());

				// Assert relevant fields equal with original
				cbsDAOImplUtils.comparePrintAckRec(printAckRec, originalPrintAckRec);

				final PrintAckRec newPrintAckRec = printAckRecDao.get(id);
				assertNotNull(newPrintAckRec);
				cbsDAOImplUtils.comparePrintAckRec(printAckRec, newPrintAckRec);

				updatePrintAckService.setPrintAckSuccess(printAckRec);
				assertEquals(CBSDAOImplUtils.STATUS_SUCCESS_ID, printAckRec.getStatusId());

				final PrintAckRec successPrintAckRec = printAckRecDao.get(id);
				assertNotNull(successPrintAckRec);
				cbsDAOImplUtils.comparePrintAckRec(printAckRec, successPrintAckRec);
			}

		}).run(jdbcTemplate);
	}

	/**
	 * Tests the processPrintAcks() and updateStmtStatusToAck() methods of
	 * UpdatePrintAckService. Testing failure to read CBSStmtStatus SENT and
	 * ACK.
	 */
	@Test
	public final void testFailReadCBSStmtStatuses() {
		final ProcessStatus.Status[] statuses = { ProcessStatus.Status.SENT, ProcessStatus.Status.ACK };

		for (ProcessStatus.Status status : statuses) {
			final ProcessStatus.Status failOnStatus = status;

			(new UpdatePrintAckServiceTestWrapper() {

				@Override
				protected void testImpl() {
					failOnCBSStmtStatus(failOnStatus); // Make getting this
														// status return null

					RuntimeException savedException = null;
					try {
						updatePrintAckService.processPrintAcks((PrintAckRec) null, (Date) null, (Date) null,
								(List<ADFileRecord>) null);
					} catch (RuntimeException e) {
						savedException = e;
					}
					assertTrue(savedException instanceof RuntimeException);

					RuntimeException savedException2 = null;
					try {
						updatePrintAckService.updateStmtStatusToAck((PrintAckRec) null);
					} catch (RuntimeException e) {
						savedException2 = e;
					}
					assertTrue(savedException2 instanceof RuntimeException);
				}

			}).run(jdbcTemplate);
		}
	}

	/**
	 * Tests the processPrintAcks() method of UpdatePrintAckService.
	 */
	@Test
	public final void testProcessPrintAcks() {
		//TODO: this test is hanging, fix
		if(true) {
			return;
		}
		
		(new CBSDAOTestWrapper() {

			@Override
			protected void test() {

				final int ackPatient1Key = 1;
				final int ackPatient2Key = 2;
				final int nonAckPatientKey = 3;

				final String ackPatient1ICN = CBSDAOImplUtils.getTestIcn(ackPatient1Key);
				final String ackPatient2ICN = CBSDAOImplUtils.getTestIcn(ackPatient2Key);
				final String nonAckPatientICN = CBSDAOImplUtils.getTestIcn(nonAckPatientKey);

				final long ackStationNum = 1L;
				final long nonAckStationNum = 2L;

				Date ackStatementDate = null;
				Date nonAckStatementDate = null;
				Date ackReceivedDate = null;
				Date ackPrintDate = null;
				try {
					ackStatementDate = (new SimpleDateFormat("MMddyyyy")).parse("06012016");
					nonAckStatementDate = (new SimpleDateFormat("MMddyyyy")).parse("05252016");
					ackReceivedDate = (new SimpleDateFormat("MMddyyyy")).parse("05312016");
					ackPrintDate = (new SimpleDateFormat("MMddyyyy")).parse("05302016");
				} catch (ParseException e) {
				}

				// Create a statement with a different statement date
				final CBSStmt nonAckStatementDateStmt = cbsDAOImplUtils.createSimpleTestCBSStmt(ackPatient1Key,
						CBSDAOImplUtils.STATUS_SENT_ID, ackStationNum, ackPatient1ICN, nonAckStatementDate);
				cbsDAOImplUtils.saveCBSStmt(nonAckStatementDateStmt);
				getCbsStmtIdS().add(nonAckStatementDateStmt.getId());
				getAccountIdS().add(nonAckStatementDateStmt.getAccountId());

				// Create a statement with a different station
				final CBSStmt nonAckStationStmt = cbsDAOImplUtils.createSimpleTestCBSStmt(ackPatient1Key,
						CBSDAOImplUtils.STATUS_SENT_ID, nonAckStationNum, ackPatient1ICN, ackStatementDate);
				cbsDAOImplUtils.saveCBSStmt(nonAckStationStmt);
				getCbsStmtIdS().add(nonAckStationStmt.getId());
				getAccountIdS().add(nonAckStationStmt.getAccountId());

				// Create a statement in NEW status
				final CBSStmt nonAckStatusStmt = cbsDAOImplUtils.createSimpleTestCBSStmt(ackPatient1Key,
						CBSDAOImplUtils.STATUS_NEW_ID, ackStationNum, ackPatient1ICN, ackStatementDate);
				cbsDAOImplUtils.saveCBSStmt(nonAckStatusStmt);
				getCbsStmtIdS().add(nonAckStatusStmt.getId());
				getAccountIdS().add(nonAckStatusStmt.getAccountId());

				// Create a statement with a different patient
				final CBSStmt nonAckPatientStmt = cbsDAOImplUtils.createSimpleTestCBSStmt(nonAckPatientKey,
						CBSDAOImplUtils.STATUS_SENT_ID, ackStationNum, nonAckPatientICN, ackStatementDate);
				cbsDAOImplUtils.saveCBSStmt(nonAckPatientStmt);
				getCbsStmtIdS().add(nonAckPatientStmt.getId());
				getAccountIdS().add(nonAckPatientStmt.getAccountId());

				// Create a statement with a different primary site
//				final CBSStmt nonAckPrimarySiteStmt = cbsDAOImplUtils.createDoubleTestCBSStmt(ackPatient1Key,
//						CBSDAOImplUtils.STATUS_SENT_ID, nonAckStationNum, ackStationNum);
//				nonAckPrimarySiteStmt.setStatementDate(ackStatementDate);
//				cbsDAOImplUtils.saveCBSStmt(nonAckPrimarySiteStmt);
//				getCbsStmtIdS().add(nonAckPrimarySiteStmt.getId());
//				getAccountIdS().add(nonAckPrimarySiteStmt.getAccountId());

				// Stmt for matching patient1 SENT status.
				final CBSStmt ackPatient1Stmt = cbsDAOImplUtils.createSimpleTestCBSStmt(ackPatient1Key,
						CBSDAOImplUtils.STATUS_SENT_ID, ackStationNum, ackPatient1ICN, ackStatementDate);
				cbsDAOImplUtils.saveCBSStmt(ackPatient1Stmt);
				getCbsStmtIdS().add(ackPatient1Stmt.getId());
				getAccountIdS().add(ackPatient1Stmt.getAccountId());

				// Stmt for matching patient2 ACK status.
				final CBSStmt ackPatient2Stmt = cbsDAOImplUtils.createSimpleTestCBSStmt(ackPatient2Key,
						CBSDAOImplUtils.STATUS_ACK_ID, ackStationNum, ackPatient2ICN, ackStatementDate);
				cbsDAOImplUtils.saveCBSStmt(ackPatient2Stmt);
				getCbsStmtIdS().add(ackPatient2Stmt.getId());
				getAccountIdS().add(ackPatient2Stmt.getAccountId());

				// Set up the PrintAckRec
				final int batchRunId = 1;
				final int printAckInitialStatusId = CBSDAOImplUtils.STATUS_INITIAL_ID;
				final String printAckFileName = "printAck.txt";

				final List<ADFileRecord> printAckDetailL = new ArrayList<ADFileRecord>(2);

				final ADFileRecord patient1AD = new ADFileRecord();
				patient1AD.setType(PrintAckFileRecord.DataType.AD);
				patient1AD.setPatientAccount(ackPatient1Stmt.getSiteStmtL().get(0).getSitePatient().getOldAcntNum());
				printAckDetailL.add(patient1AD);

				final ADFileRecord patient2AD = new ADFileRecord();
				patient2AD.setType(PrintAckFileRecord.DataType.AD);
				patient2AD.setPatientAccount(ackPatient2Stmt.getSiteStmtL().get(0).getSitePatient().getOldAcntNum());
				printAckDetailL.add(patient2AD);

				final PrintAckRec printAckRec = new PrintAckRec();
				printAckRec.setDateReceived(new Timestamp(ackReceivedDate.getTime()));
				printAckRec.setStationNum("" + ackStationNum);
				printAckRec.setStatusId(printAckInitialStatusId);
				printAckRec.setFileName(printAckFileName);
				printAckRec.setBatchRunId(batchRunId);
				printAckRec.setPrintAckDetailL(printAckDetailL);
				getPrintAckRecIdS().add(printAckRecDao.insert(printAckRec));
				assertTrue(printAckRec.getId() > 0);

				// Test that 2 patient statements are found and updated
				final List<ADFileRecord> missingPatientAccts = new ArrayList<ADFileRecord>(2);
				assertEquals(2, updatePrintAckService.processPrintAcks(printAckRec, ackStatementDate, ackPrintDate,
						missingPatientAccts));
				assertEquals(0, missingPatientAccts.size());

				final CBSStmt updatedAckPatient1Stmt = cbsStmtDao.get(ackPatient1Stmt.getId());
				assertNotNull(updatedAckPatient1Stmt);
				ackPatient1Stmt.setModifiedDate(updatedAckPatient1Stmt.getModifiedDate());
				ackPatient1Stmt.setPrintAckId(printAckRec.getId());
				ackPatient1Stmt.setPrintDate(ackPrintDate);
				ackPatient1Stmt.setStatusId(CBSDAOImplUtils.STATUS_ACK_ID);
				cbsDAOImplUtils.compareCBSStmt(updatedAckPatient1Stmt, ackPatient1Stmt);

				final CBSStmt updatedAckPatient2Stmt = cbsStmtDao.get(ackPatient2Stmt.getId());
				assertNotNull(updatedAckPatient2Stmt);
				ackPatient2Stmt.setModifiedDate(updatedAckPatient2Stmt.getModifiedDate());
				ackPatient2Stmt.setPrintAckId(printAckRec.getId());
				ackPatient2Stmt.setPrintDate(ackPrintDate);
				ackPatient2Stmt.setStatusId(CBSDAOImplUtils.STATUS_ACK_ID);
				cbsDAOImplUtils.compareCBSStmt(updatedAckPatient2Stmt, ackPatient2Stmt);

			}

		}).run(jdbcTemplate);

	}

	/**
	 * Extends the test wrapper to provide support for a status DAO proxy.
	 * 
	 * @author Brad Pickle
	 *
	 */
	private abstract class UpdatePrintAckServiceTestWrapper extends CBSDAOTestWrapper {
		private ProcessStatusDAO savePrintAckStatusDAO;
		private ProcessStatusDAO saveCBSStmtStatusDAO;

		@Override
		protected final void test() {
			try {
				testImpl();
			} finally {
				if (savePrintAckStatusDAO != null) {
					updatePrintAckService.setProcessStatusDAO(savePrintAckStatusDAO);
				}
				if (saveCBSStmtStatusDAO != null) {
					updatePrintAckService.setProcessStatusDAO(saveCBSStmtStatusDAO);
				}
			}
		}

		protected abstract void testImpl();

		private void savePrintAckStatusDAO() {
			if (savePrintAckStatusDAO == null) {
				savePrintAckStatusDAO = updatePrintAckService.getProcessStatusDAO();
			}
		}

		protected void failOnPrintAckStatus(final ProcessStatus.Status failEnumValue) {
			savePrintAckStatusDAO();
			overridePrintAckStatusDAO(new FailPrintAckStatusDAOImpl(failEnumValue, savePrintAckStatusDAO));

		}

		protected void overridePrintAckStatusDAO(final ProcessStatusDAO newDao) {
			savePrintAckStatusDAO();
			updatePrintAckService.setProcessStatusDAO(newDao);
		}

		private void saveCBSStmtStatusDAO() {
			if (saveCBSStmtStatusDAO == null) {
				saveCBSStmtStatusDAO = updatePrintAckService.getProcessStatusDAO();
			}
		}

		protected void failOnCBSStmtStatus(final ProcessStatus.Status failEnumValue) {
			saveCBSStmtStatusDAO();
			overrideCBSStmtStatusDAO(new FailCBSStmtStatusDAOImpl(failEnumValue, saveCBSStmtStatusDAO));

		}

		protected void overrideCBSStmtStatusDAO(final ProcessStatusDAO newDao) {
			saveCBSStmtStatusDAO();
			updatePrintAckService.setProcessStatusDAO(newDao);
		}

	}

	/**
	 * A PrintAckStatusDAO that will return null for a specified status.
	 * 
	 * @author Brad Pickle
	 *
	 */
	private class FailPrintAckStatusDAOImpl extends ProcessStatusDAOImpl {

		private ProcessStatus.Status failEnumValue;

		private ProcessStatusDAO savePrintAckStatusDAO;

		/**
		 * Instantiate FailPrintAckStatusDAOImpl for the given status.
		 * 
		 * @param failEnumValue
		 *            The status which should fail.
		 * @param savePrintAckStatusDAO
		 *            All status besides failEnumValue will be delegated to this
		 *            DAO.
		 */
		protected FailPrintAckStatusDAOImpl(ProcessStatus.Status failEnumValue,
				ProcessStatusDAO savePrintAckStatusDAO) {
			this.failEnumValue = failEnumValue;
			this.savePrintAckStatusDAO = savePrintAckStatusDAO;
		}

		/**
		 * Returns null if enumValue is the failEnumValue specified in the
		 * constructor. Otherwise returns value from the delegated DAO.
		 * 
		 * @param enumValue
		 *            Status to get.
		 * @return The status id, or null if failEnumValue.
		 */
		@Override
		public Integer getStatusFromEnum(ProcessStatus.Status enumValue) {
			return (failEnumValue == enumValue) ? null : savePrintAckStatusDAO.getStatusFromEnum(enumValue);
		}
	}

	/**
	 * A CBSStmtStatusDAO that will return null for a specified status.
	 * 
	 * @author Brad Pickle
	 *
	 */
	private class FailCBSStmtStatusDAOImpl extends ProcessStatusDAOImpl {

		private ProcessStatus.Status failEnumValue;

		private ProcessStatusDAO saveStmtStatusDAO;

		/**
		 * Instantiate FailCBSStmtStatusDAOImpl for the given status.
		 * 
		 * @param failEnumValue
		 *            The status which should fail.
		 * @param saveStmtStatusDAO
		 *            All status besides failEnumValue will be delegated to this
		 *            DAO.
		 */
		protected FailCBSStmtStatusDAOImpl(ProcessStatus.Status failEnumValue, ProcessStatusDAO saveStmtStatusDAO) {
			this.failEnumValue = failEnumValue;
			this.saveStmtStatusDAO = saveStmtStatusDAO;
		}

		/**
		 * Returns null if enumValue is the failEnumValue specified in the
		 * constructor. Otherwise returns value from the delegated DAO.
		 * 
		 * @param enumValue
		 *            Status to get.
		 * @return The status id, or null if failEnumValue.
		 */
		@Override
		public Integer getStatusFromEnum(ProcessStatus.Status enumValue) {
			return (failEnumValue == enumValue) ? null : saveStmtStatusDAO.getStatusFromEnum(enumValue);
		}
	}

}
