package gov.va.cpss.service;

import static org.junit.Assert.*;
import static org.junit.Assume.assumeTrue;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.sql.DataSource;

import org.apache.log4j.Logger;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;
import org.springframework.jdbc.datasource.init.ScriptUtils;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.dao.VistaAcntHistDAO;
import gov.va.cpss.dao.CBSAccountDAO;
import gov.va.cpss.dao.VistaAccountDAO;
import gov.va.cpss.model.DfnStationNumPair;
import gov.va.cpss.model.ProcessStatus.Status;
import gov.va.cpss.model.cbs.CBSAccount;
import gov.va.cpss.model.icn.VistaAccount;
import gov.va.cpss.model.icn.VistaAcntHist;

//import static gov.va.cpss.Constants.*;

/**
 * Unit tests for Updating ICN Across CBSS Business Service
 * 
 * (Sprint 2 - Story 325577 Update ICNs / Task 325578 Business service
 * implementation to update ICN across CBSS.)
 * 
 * Copyright HPE / VA June 6, 2016
 * 
 * @author Yiping Yao
 * @version 1.0.0
 *
 *
 *
 * Sprint 5 - Epic 349068, Tasks 370262 and 370266 DAO Implementation,
 *            and Implement ICN Update Service - Update based on new design.
 *
 * Copyright HPE / VA September 8, 2016
 * 
 * @author Yiping Yao
 * @version 2.0.0
 *
 */
@SuppressWarnings("nls")
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml", "/update-icn-test-context.xml",
		                            "/cpss-email.xml" })
public class UpdateICNServiceTest
{
	// Private Logger
	private final Logger logger = Logger.getLogger(this.getClass().getCanonicalName());

	@Value("${run.integration.test:false}")
	private Boolean runIntegrationTest;

	@Autowired
	private UpdateICNService updateICNService;

	@Autowired
	private VistaAcntHistDAO vistaAcntHistDAO;

	@Autowired
	private VistaAccountDAO vistaAccountDAO;

	@Autowired
	private CBSAccountDAO cbsAccountDAO;

	private List<VistaAcntHist> vistaAcntHistories = null;
	private List<VistaAccount> vistaAccounts = null;
	private List<CBSAccount> cbsAccounts = null;

	private Connection jdbcConnection = null;

	/**
	 * @param dataSource
	 */
	@Autowired
	public void setDataSource(DataSource dataSource)
	{
		try
		{
			this.jdbcConnection = dataSource.getConnection();
		}
		catch (SQLException e)
		{
			// Auto-generated catch block
			e.printStackTrace();
			this.logger.info("SQL Exception in getting JDBC connection.");
		}
	}

	/**
	 * Setup test data - insert into DB before tests.
	 * 
	 */
	@Before
	public void setUp()
	{
		assumeTrue(this.runIntegrationTest.booleanValue());

		// Resource sqlResource = new
		// ClassPathResource("sql/UpdateICNServiceTest-InsertData.sql");
		// ScriptUtils.executeSqlScript(this.jdbcConnection, sqlResource);
		executeTestData("sql/UpdateICNServiceTest01-DeleteData.sql");
		executeTestData("sql/UpdateICNServiceTest02-DeleteData.sql");
		executeTestData("sql/UpdateICNServiceTest03-DeleteData.sql");
		executeTestData("sql/UpdateICNServiceTest04-DeleteData.sql");
		executeTestData("sql/UpdateICNServiceTest05-DeleteData.sql");
		executeTestData("sql/UpdateICNServiceTest06-DeleteData.sql");
		//executeTestData("sql/UpdateICNServiceTest07-DeleteData.sql");
		//executeTestData("sql/UpdateICNServiceTest08-DeleteData.sql");
	}

	/**
	 * Tear down test data - delete from DB after tests.
	 * 
	 */
	@After
	public void tearDown()
	{
		if (this.runIntegrationTest.booleanValue())
		{
			// Resource sqlResource = new
			// ClassPathResource("sql/UpdateICNServiceTest-DeleteData.sql");
			// ScriptUtils.executeSqlScript(this.jdbcConnection, sqlResource);
			executeTestData("sql/UpdateICNServiceTest01-DeleteData.sql");
			executeTestData("sql/UpdateICNServiceTest02-DeleteData.sql");
			executeTestData("sql/UpdateICNServiceTest03-DeleteData.sql");
			executeTestData("sql/UpdateICNServiceTest04-DeleteData.sql");
			executeTestData("sql/UpdateICNServiceTest05-DeleteData.sql");
			executeTestData("sql/UpdateICNServiceTest06-DeleteData.sql");
			//executeTestData("sql/UpdateICNServiceTest07-DeleteData.sql");
			//executeTestData("sql/UpdateICNServiceTest08-DeleteData.sql");
		}
	}

	/**
	 * Test 0
	 * 
	 * Unit tests for updating ICN's with input VistA Account History
	 * 
	 */
	@Test
	public final void testUpdateICNsWithVistaAcntHistories()
	{
		this.logger.info("Test 0 starts: ");

		this.vistaAcntHistories = new ArrayList<>();

		// Build VistA Account History
		VistaAcntHist vistaAcntHist1 = new VistaAcntHist();

		// The old ICN and new ICN are the same.
		vistaAcntHist1.setOldIcn("ICN1234567890OLD1");
		vistaAcntHist1.setNewIcn("ICN1234567890OLD1");

		this.vistaAcntHistories.add(vistaAcntHist1);

		VistaAcntHist vistaAcntHist2 = new VistaAcntHist();

		// The status ID is processed.
		vistaAcntHist2.setOldIcn("ICN1234567890OLD2");
		vistaAcntHist2.setNewIcn("ICN1234567890NEW2");
		vistaAcntHist2.setStatus(Status.PROCESSED);

		this.vistaAcntHistories.add(vistaAcntHist2);

		Result<Integer[]> result = null;

		result = this.updateICNService.updateICN(this.vistaAcntHistories);

		this.logger.info("Return message: " + result.getMessage());

		assertFalse(result.isSuccessful());

		if (result.get() != null)
		{
			for (int i = 0; i < result.get().length; i++)
			{
				this.logger.info("Success or updated rows: " + result.get()[i]);

				assertTrue((result.get()[i].intValue() >= 0) ||
						   (result.get()[i].intValue() == Statement.SUCCESS_NO_INFO));
			}
		}

		this.logger.info("Test 0 ends.");
	}

	/**
	 * Test 1
	 * 
	 * Unit tests for updating ICN's with VistA Account History from DB
	 * 
	 * All existing CBS Accounts, also including not NEW ICN data and same ICN's
	 * data.
	 * 
	 */
	@Test
	public final void testUpdateICNsFromDB01()
	{
		this.logger.info("Test 1 starts: ");

		// Load test data
		executeTestData("sql/UpdateICNServiceTest01-InsertData.sql");

		// Using VistA Account History in DB
		this.vistaAcntHistories = null;

		Result<Integer[]> result = this.updateICNService.updateICN(this.vistaAcntHistories);

		// Check result
		checkResult(result);

		// Check data
		checkData();

		this.logger.info("Test 1 ends.");
	}

	/**
	 * Test 2
	 * 
	 * Unit tests for updating ICN's with VistA Account History from DB
	 * 
	 * All new CBS Accounts, also including not NEW ICN data and same ICN's
	 * data.
	 * 
	 */
	@Test
	public final void testUpdateICNsFromDB02()
	{
		this.logger.info("Test 2 starts: ");

		// Load test data
		executeTestData("sql/UpdateICNServiceTest02-InsertData.sql");

		// Using VistA Account History in DB
		this.vistaAcntHistories = null;

		Result<Integer[]> result = this.updateICNService.updateICN(this.vistaAcntHistories);

		// Check result
		checkResult(result);

		// Check data
		checkData();

		this.logger.info("Test 2 ends.");
	}

	/**
	 * Test 3
	 * 
	 * Unit tests for updating ICN's with VistA Account History from DB
	 * 
	 * Mixed new and existing CBS Accounts, also including not NEW ICN data and
	 * same ICN's data.
	 * 
	 */
	@Test
	public final void testUpdateICNsFromDB03()
	{
		this.logger.info("Test 3 starts: ");

		// Load test data
		executeTestData("sql/UpdateICNServiceTest03-InsertData.sql");

		// Using VistA Account History in DB
		this.vistaAcntHistories = null;

		Result<Integer[]> result = this.updateICNService.updateICN(this.vistaAcntHistories);

		// Check result
		checkResult(result);

		// Check data
		checkData();

		this.logger.info("Test 3 ends.");
	}

	/**
	 * Test 4
	 * 
	 * Unit tests for updating ICN's with VistA Account History from DB
	 * 
	 * Mixed new and existing CBS Accounts, also including not NEW ICN data and
	 * same ICN's data, as well as duplicated VistA Account History data before
	 * processing, for both existing and new CBS Accounts.
	 * 
	 */
	@Test
	public final void testUpdateICNsFromDB04()
	{
		this.logger.info("Test 4 starts: ");

		// Load test data
		executeTestData("sql/UpdateICNServiceTest04-InsertData.sql");

		// Using VistA Account History in DB
		this.vistaAcntHistories = null;

		Result<Integer[]> result = this.updateICNService.updateICN(this.vistaAcntHistories);

		// Check result
		checkResult(result);

		// Check data
		checkData();

		this.logger.info("Test 4 ends.");
	}

	/**
	 * Test 5
	 * 
	 * Unit tests for updating ICN's with VistA Account History from DB
	 * 
	 * Mixed new and existing CBS Accounts, also including not NEW ICN data and
	 * same ICN's data, as well as duplicated VistA Account History data after
	 * processing, for both existing and new CBS Accounts.
	 * 
	 */
	@Test
	public final void testUpdateICNsFromDB05()
	{
		this.logger.info("Test 5 starts: ");

		// Load test data
		executeTestData("sql/UpdateICNServiceTest05-InsertData.sql");

		// Using VistA Account History in DB
		this.vistaAcntHistories = null;

		Result<Integer[]> result = this.updateICNService.updateICN(this.vistaAcntHistories);

		// Check result
		checkResult(result);

		// Check data
		checkData();

		this.logger.info("Test 5 ends.");
	}

	/**
	 * Test 6
	 * 
	 * Unit tests for updating ICN's with VistA Account History from DB
	 * 
	 * Mixed new and existing CBS Accounts, also including not NEW ICN data and
	 * same ICN's data, with splitting ICN's:
	 * 
	 * Before, DFN / Site 1 - Old ICN 1 DFN / Site 2 - Old ICN 1
	 * 
	 * After, DFN / Site 1 - New ICN 1 DFN / Site 2 - New ICN 2
	 * 
	 */
	@Test
	public final void testUpdateICNsFromDB06()
	{
		this.logger.info("Test 6 starts: ");

		// Load test data
		executeTestData("sql/UpdateICNServiceTest06-InsertData.sql");

		// Using VistA Account History in DB
		this.vistaAcntHistories = null;

		Result<Integer[]> result = this.updateICNService.updateICN(this.vistaAcntHistories);

		// Check result
		checkResult(result);

		// Check data
		checkData();

		this.logger.info("Test 6 ends.");
	}

	/**
	 * Test 7
	 * 
	 * Unit tests for updating ICN's with VistA Account History from DB
	 * 
	 * Mixed new and existing CBS Accounts, also including not NEW ICN data and
	 * same ICN's data, with combining ICN's:
	 * 
	 * Before, DFN / Site 1 - Old ICN 1 DFN / Site 1 - Old ICN 2
	 * 
	 * After, DFN / Site 1 - New ICN 1
	 * 
	 * (This test case may need to be verified, due to VistA Account primary
	 * keys only being DFN and StationNum / Site.)
	 */
	//@Test
	public final void testUpdateICNsFromDB07()
	{
		this.logger.info("Test 7 starts: ");

		 // Load test data
		 //executeTestData("sql/UpdateICNServiceTest07-InsertData.sql");

		 // Using VistA Account History in DB
		 this.vistaAcntHistories = null;

		 //Result<Integer[]> result = this.updateICNService.updateICN(this.vistaAcntHistories);

		 // Check result
		 //checkResult(result);

		 // Check data
		 //checkData();

		this.logger.info("Test 7 ends.");
	}

	/**
	 * Test 8 - Brad Pickle - 1/5/2017
	 * 
	 * Unit tests for updating ICN's with VistA Account History from DB
	 * 
	 * Update 17 ICNs in one run to test batch update logic for more than
	 * BATCH_SELECT_SQL_SMALL updates.
	 * 
	 * Tests that for two VistaAcntHist with same DFN/StationNum, the one with
	 * latest modifiedDate will be used.
	 * 
	 */
    //TODO: re-enable, this breaks everything
	//@Test
	public final void testUpdateICNsFromDB08()
	{
		this.logger.info("Test 8 starts: ");

		// Load test data
		//executeTestData("sql/UpdateICNServiceTest08-InsertData.sql");

		// Using VistA Account History in DB
		this.vistaAcntHistories = null;

		//Result<Integer[]> result = this.updateICNService.updateICN(this.vistaAcntHistories);

		// Check result
		//checkResult(result);

		// Check data
		//checkData();

		this.logger.info("Test 8 ends.");
	}

	// Execute test data SQL script file
	private void executeTestData(String sqlFile)
	{
		Resource sqlResource = new ClassPathResource(sqlFile);
		ScriptUtils.executeSqlScript(this.jdbcConnection, sqlResource);
	}

	// Check result
	private void checkResult(Result<Integer[]> inResult)
	{
		this.logger.info("Return message: " + inResult.getMessage());

		assertTrue(inResult.isSuccessful());

		if (inResult.get() != null)
		{
			for (int i = 0; i < inResult.get().length; i++)
			{
				this.logger.info("Success or updated rows: " + inResult.get()[i]);

				assertTrue((inResult.get()[i].intValue() >= 0) ||
						   (inResult.get()[i].intValue() == Statement.SUCCESS_NO_INFO));
			}
		}
	}

	// Check data in DB
	private void checkData()
	{
		// Check VistA Account History
		this.vistaAcntHistories = this.vistaAcntHistDAO.select(Status.PROCESSED);

		this.logger.debug("Number of VistA Account History before removing duplicate: " + this.vistaAcntHistories.size());

		assertNotNull(this.vistaAcntHistories);
		assertFalse(this.vistaAcntHistories.isEmpty());

		// Remove duplicated VistA Account History
		// Also use LinkedHashSet to preserve the order
		Set<VistaAcntHist> tempVistaAcntHistories = new LinkedHashSet<>();
		tempVistaAcntHistories.addAll(this.vistaAcntHistories);
		this.vistaAcntHistories.clear();
		this.vistaAcntHistories.addAll(tempVistaAcntHistories);

		this.logger.debug("Number of VistA Account History after removing duplicate: " + this.vistaAcntHistories.size());

		String[] newICNs = new String[this.vistaAcntHistories.size()];
		String[] oldICNs = new String[this.vistaAcntHistories.size()];

		for (int i = 0; i < this.vistaAcntHistories.size(); i++)
		{
			newICNs[i] = this.vistaAcntHistories.get(i).getNewIcn();
			oldICNs[i] = this.vistaAcntHistories.get(i).getOldIcn();

			if (!newICNs[i].equals(oldICNs[i]))
			{
				assertTrue(this.vistaAcntHistories.get(i).getOldCbssAcntId() > 0);
			}
		}

		// Identify the latest VistaAcntHist records if there are duplicates for
		// any DFN/SiteNumber
		final Set<DfnStationNumPair> uniquePairs = new HashSet<>();
		final Map<DfnStationNumPair, VistaAcntHist> latestDupHists = new HashMap<>();

		for (VistaAcntHist vistaAcntHist : this.vistaAcntHistories)
		{
			final long dfn = vistaAcntHist.getDfn();
			final String stationNum = vistaAcntHist.getStationNum();
			final DfnStationNumPair pair = new DfnStationNumPair(dfn, stationNum);

			if (uniquePairs.contains(pair))
			{
				// More than one VistaAcntHist for this pair

				// Pick the one with the last modified date and save it
				VistaAcntHist latestDupHist = latestDupHists.get(pair);
				if ( (latestDupHist == null) ||
				     (latestDupHist.getModifiedDate().before(vistaAcntHist.getModifiedDate())) )
				{
					latestDupHist = vistaAcntHist;
				}

				latestDupHists.put(pair, latestDupHist);
			}
			else
			{
				uniquePairs.add(pair);
			}
		}

		this.logger.debug("Number of duplicate pair Vista Account History: " + latestDupHists.size());

		for (VistaAcntHist vistaAcntHist : latestDupHists.values())
		{
			this.logger.debug("- for duplicate DFN:" + vistaAcntHist.getDfn() + ", StationNum:" +
					          vistaAcntHist.getStationNum() + ", expecting newICN:" + vistaAcntHist.getNewIcn());
		}

		// Check CBS Account
		final Set<String> newICNSet = new HashSet<>();

		for (String icn : newICNs)
		{
			newICNSet.add(icn);
		}

		this.cbsAccounts = this.cbsAccountDAO.batchSelect(newICNSet.toArray(new String[0]));

		assertNotNull(this.cbsAccounts);
		assertFalse(this.cbsAccounts.isEmpty());
		assertEquals(this.cbsAccounts.size(), newICNSet.size());

		for (CBSAccount cbsAccount : this.cbsAccounts)
		{
			assertNotNull(cbsAccount);
		}

		// Check CBS Account
		final Set<String> oldICNSet = new HashSet<>();

		for (String icn : oldICNs)
		{
			oldICNSet.add(icn);
		}

		final List<CBSAccount> oldCBSAccounts = this.cbsAccountDAO.batchSelect(oldICNSet.toArray(new String[0]));

		assertNotNull(oldCBSAccounts);
		assertFalse(oldCBSAccounts.isEmpty());
		assertEquals(oldCBSAccounts.size(), oldICNSet.size());

		for (CBSAccount cbsAccount : oldCBSAccounts)
		{
			assertNotNull(cbsAccount);
		}

		// Check VistA Account
		this.vistaAccounts = this.vistaAccountDAO.batchSelect(newICNs);

		assertNotNull(this.vistaAccounts);
		assertFalse(this.vistaAccounts.isEmpty());

		boolean hasCBSAccount;

		for (VistaAccount vistaAccount : this.vistaAccounts)
		{
			hasCBSAccount = false;

			// If multiple hists for this dfn/sitenum, validate the correct one was used
			final DfnStationNumPair pair = new DfnStationNumPair(vistaAccount.getDfn(), vistaAccount.getStationNum());

			if (latestDupHists.containsKey(pair))
			{
				assertTrue(latestDupHists.get(pair).getNewIcn().equals(vistaAccount.getIcn()));
			}

			for (VistaAcntHist vistaAcntHistory : this.vistaAcntHistories)
			{
				if (vistaAccount.getIcn().equals(vistaAcntHistory.getNewIcn()))
				{
					for (CBSAccount cbsAccount : this.cbsAccounts)
					{
						if (vistaAcntHistory.getNewIcn().equals(cbsAccount.getIcn()))
						{
							assertTrue(vistaAccount.getCbssAcntId() == cbsAccount.getId());

							hasCBSAccount = true;
							this.cbsAccounts.remove(cbsAccount);
							break;
						}
					}

					this.vistaAcntHistories.remove(vistaAcntHistory);
					break;
				}
			}

			assertTrue(hasCBSAccount);
		}
	}
}
