package gov.va.cpss.service;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static org.junit.Assume.assumeTrue;

import java.io.BufferedReader;
import java.io.File;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

/**
 * Integration Unit Tests to test success and failure cases of SFTP Service.
 * Various test files are referenced in src/test/resources/fps directory.
 * 
 * @author DNS
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-sftp-service.xml" })
public class FtpServiceIT {

	@SuppressWarnings("unused")
	private Logger logger = Logger.getLogger(this.getClass().getCanonicalName());

	@Value("${run.integration.test:false}")
	private Boolean runIntegrationTest;

	private final String DATA_DIRECTORY = "psdata";

	@Autowired
	private SftpService sftpService;

	/**
	 * Only run these tests if property is set to run integration test.
	 */
	@Before
	public final void beforeTest() {
		assumeTrue(runIntegrationTest);
	}
	
	@Test
	public final void testRemove() {

		final String filename = "CCPC-to-CPSS_empty.txt";
		final String localFilepath = new File(this.getClass().getClassLoader().getResource("fps/" + filename).getFile())
				.getAbsolutePath();
		final String targetDirectory = getArchiveDirectory();

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, DATA_DIRECTORY));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, targetDirectory));

		// FTP test files to server.
		assertTrue(sftpService.ftpFileToServer(localFilepath, DATA_DIRECTORY));
		assertTrue(sftpService.ftpMoveFileFromDirectoryToDirectory(filename, DATA_DIRECTORY, targetDirectory));
		assertTrue(sftpService.ftpFileToServer(localFilepath, DATA_DIRECTORY));

		// Ensure test server has the files.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename, DATA_DIRECTORY));
		assertTrue(sftpService.ftpFileExistsInDirectory(filename, targetDirectory));

		// Remove file from the specified directories.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, DATA_DIRECTORY));
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, targetDirectory));

		// Verify file is no longer in the target directories.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, DATA_DIRECTORY));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, targetDirectory));
	}

	@Test
	public final void testMove() {

		final String filename = "CCPC-to-CPSS_empty.txt";
		final String localFilepath = new File(this.getClass().getClassLoader().getResource("fps/" + filename).getFile())
				.getAbsolutePath();
		final String targetDirectory = getErrorDirectory();

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, targetDirectory));

		// FTP test file to server.
		assertTrue(sftpService.ftpFileToServer(localFilepath));

		// Move file to the specified directory.
		assertTrue(sftpService.ftpMoveFileToDirectory(filename, targetDirectory));

		// Verify file is no longer in the original directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, DATA_DIRECTORY));

		// Verify file is in the target directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename, targetDirectory));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, targetDirectory));
	}

	@Test
	public final void testRenameFileInDirectory() {

		final String filename = "CCPC-to-CPSS_empty.txt";
		final String localFilepath = new File(this.getClass().getClassLoader().getResource("fps/" + filename).getFile())
				.getAbsolutePath();
		final String targetFilename = filename + ".121110.sendcbs";
		final String targetDirectory = getErrorDirectory();

		try {

			// Ensure test server is in expected initial state.
			assertFalse(sftpService.ftpFileExistsInDirectory(filename, targetDirectory));

			// Ensure test server is in expected initial state.
			assertFalse(sftpService.ftpFileExistsInDirectory(filename, targetDirectory));

			// FTP test file to server.
			assertTrue(sftpService.ftpFileToServerWithName(localFilepath, targetFilename, targetDirectory));

			// Verify file is in the target directory.
			assertTrue(sftpService.ftpFileExistsInDirectory(targetFilename, targetDirectory));

			Thread.sleep(4000);

			// Move file to the specified directory.
			assertTrue(sftpService.ftpRenameFileInDirectory(targetFilename, filename, targetDirectory));

			// Verify file is no longer in the original directory.
			assertFalse(sftpService.ftpFileExistsInDirectory(targetFilename, targetDirectory));

			// Verify file is in the target directory.
			assertTrue(sftpService.ftpFileExistsInDirectory(filename, targetDirectory));

			Thread.sleep(4000);

			// Cleanup.
			assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, targetDirectory));

		} catch (Exception e) {
			fail(e.getMessage());
		}
	}

	@Test
	public final void testPut() {

		final String filename = "CPSS-to-CCPC_PS_invalid_sequence_number.txt";
		final String localFilepath = new File(this.getClass().getClassLoader().getResource("fps/" + filename).getFile())
				.getAbsolutePath();

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, DATA_DIRECTORY));

		// FTP test file to server.
		assertTrue(sftpService.ftpFileToServer(localFilepath, DATA_DIRECTORY));

		// Verify file is in the target directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename, DATA_DIRECTORY));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, DATA_DIRECTORY));
	}

	@Test
	public final void testPutRename() {

		final String filename = "CPSS-to-CCPC_PS_invalid_sequence_number.txt";
		final String localFilepath = new File(this.getClass().getClassLoader().getResource("fps/" + filename).getFile())
				.getAbsolutePath();
		final String targetFilename = filename + ".121110.sendcbs";

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, DATA_DIRECTORY));

		// FTP test file to server.
		assertTrue(sftpService.ftpFileToServerWithName(localFilepath, targetFilename, DATA_DIRECTORY));

		// Verify file is in the target directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(targetFilename, DATA_DIRECTORY));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(targetFilename, DATA_DIRECTORY));
	}

	@Test
	public final void testPutEmpty() {

		final String targetFilename = "test.don";

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(targetFilename, DATA_DIRECTORY));

		// FTP test file to server.
		assertTrue(sftpService.ftpEmptyFileToServerWithName(targetFilename, DATA_DIRECTORY));

		// Verify file is in the target directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(targetFilename, DATA_DIRECTORY));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(targetFilename, DATA_DIRECTORY));
	}

	@Test
	public final void testPutString() {

		final String targetFilename = "test.txt";

		final String content = "This is row 1\nThis is row 2!";

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(targetFilename, DATA_DIRECTORY));

		// FTP test file to server.
		assertTrue(sftpService.ftpStringAsFileToServerWithName(content, targetFilename, DATA_DIRECTORY));

		// Verify file is in the target directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(targetFilename, DATA_DIRECTORY));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(targetFilename, DATA_DIRECTORY));
	}

	@Test
	public final void testLs() {

		final String filename = "CPSS-to-CCPC_PS_invalid_sequence_number.txt";
		final String localFilepath = new File(this.getClass().getClassLoader().getResource("fps/" + filename).getFile())
				.getAbsolutePath();

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, DATA_DIRECTORY));

		// FTP test file to server.
		assertTrue(sftpService.ftpFileToServer(localFilepath, DATA_DIRECTORY));

		// Verify file is in the target directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename, DATA_DIRECTORY));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, DATA_DIRECTORY));
	}

	@Test
	public final void testFileInfoLs() {

		final String filename1 = "CPSS-to-CCPC_PS_invalid_sequence_number.txt";
		final String localFilepath1 = new File(
				this.getClass().getClassLoader().getResource("fps/" + filename1).getFile()).getAbsolutePath();
		final String filename2 = "CPSS-to-CCPC_PS_invalid_end_of_file.txt";
		final String localFilepath2 = new File(
				this.getClass().getClassLoader().getResource("fps/" + filename2).getFile()).getAbsolutePath();
		final String filename3 = "CPSS-to-CCPC_PH_invalid_end_of_file.txt";
		final String localFilepath3 = new File(
				this.getClass().getClassLoader().getResource("fps/" + filename3).getFile()).getAbsolutePath();

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename1, DATA_DIRECTORY));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename2, DATA_DIRECTORY));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename3, DATA_DIRECTORY));

		// FTP test file to server.
		assertTrue(sftpService.ftpFileToServer(localFilepath1, DATA_DIRECTORY));

		try {
			Thread.sleep(3000);
		} catch (Exception e) {
			fail("Failure sleeping");
		}

		assertTrue(sftpService.ftpFileToServer(localFilepath3, DATA_DIRECTORY));

		try {
			Thread.sleep(3000);
		} catch (Exception e) {
			fail("Failure sleeping");
		}

		assertTrue(sftpService.ftpFileToServer(localFilepath2, DATA_DIRECTORY));

		// ftpGetFileInfoListInDirectory has been removed from SftpService to
		// remove the dependency on Spring Integration. - Brad
		// Test to see if the list comes back with the proper files.
		// List<FileInfo<?>> fileList =
		// sftpService.ftpGetFileInfoListInDirectory(DATA_DIRECTORY);
		// assertTrue(fileList.size() == 3);
		// List<String> fileNameList = new ArrayList<String>();
		// fileNameList.add(filename1);
		// fileNameList.add(filename2);
		// fileNameList.add(filename3);
		// for(FileInfo<?> fileInfo: fileList) {
		// assertTrue(fileNameList.contains(fileInfo.getFilename()));
		// }

		// Ensure the dates are sorted.
		List<String> serverFileList = sftpService.ftpGetFileListWithExtensionInDirectory(DATA_DIRECTORY, ".txt");
		assertTrue(serverFileList.size() == 3);

		assertTrue(serverFileList.get(0).equals(filename1));
		assertTrue(serverFileList.get(1).equals(filename3));
		assertTrue(serverFileList.get(2).equals(filename2));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename1, DATA_DIRECTORY));
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename2, DATA_DIRECTORY));
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename3, DATA_DIRECTORY));
	}

	@Test
	public final void testGet() {

		final String filename = "CPSS-to-CCPC_successful.txt";
		final String localFilepath = new File(this.getClass().getClassLoader().getResource("fps/" + filename).getFile())
				.getAbsolutePath();

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, DATA_DIRECTORY));

		// FTP test file to server.
		assertTrue(sftpService.ftpFileToServer(localFilepath, DATA_DIRECTORY));

		// Verify file is in the target directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename, DATA_DIRECTORY));

		// Verify the file is read as expected.
		SftpStreamSession sss = null;
		try {
			sss = sftpService.openFileStream(filename, DATA_DIRECTORY);
			assertNotNull(sss);
			List<String> fileContentsL = new ArrayList<>();
			BufferedReader reader = new BufferedReader(new InputStreamReader(sss.getInputStream(), "utf-8"));
			String line = reader.readLine();
			// Loop until thread is interrupted or no more lines available.
			while (line != null) {
				fileContentsL.add(line);
				line = reader.readLine();
			}
			assertNotNull(fileContentsL);
			assertFalse(fileContentsL.isEmpty());
			assertEquals(28, fileContentsL.size());
		} catch (Exception e) {
			fail("Unexpected exception: " + e.getMessage());
		} finally {
			if (sss != null) {
				sss.close();
			}
		}

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, DATA_DIRECTORY));
	}

	@Test
	public final void testGetByExtension() {

		final String dataFilename = "CCPC-to-CPSS_empty.txt";
		final String indicatorFilename = "CCPC-to-CPSS_empty.don";
		final String localFilepath = new File(
				this.getClass().getClassLoader().getResource("fps/" + dataFilename).getFile()).getAbsolutePath();

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(dataFilename, DATA_DIRECTORY));
		assertFalse(sftpService.ftpFileExistsInDirectory(indicatorFilename, DATA_DIRECTORY));

		// FTP test file to server.
		assertTrue(sftpService.ftpFileToServer(localFilepath, DATA_DIRECTORY));
		assertTrue(sftpService.ftpFileToServerWithName(localFilepath, indicatorFilename, DATA_DIRECTORY));

		// Verify file is in the target directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(dataFilename, DATA_DIRECTORY));
		assertTrue(sftpService.ftpFileExistsInDirectory(indicatorFilename, DATA_DIRECTORY));

		List<String> fileL = sftpService.ftpGetFileListWithExtensionInDirectory(DATA_DIRECTORY, ".don");

		// Verify only one file.
		assertEquals(1, fileL.size());

		// Verify the file is read as expected.
		try {
			String filename = fileL.get(0);

			// Build the input file from the indicator file.
			filename = filename.substring(0, filename.lastIndexOf(".don")) + ".txt";

			// Only attempt to process if the filename is not null and not
			// empty.
			if ((filename != null) && !filename.isEmpty()) {
				assertEquals(filename, dataFilename);
			} else {
				fail("Attempted to process an invalid filename that was null or empty string");
			}

		} catch (Exception e) {
			fail("Unexpected exception: " + e.getMessage());
		}

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(dataFilename, DATA_DIRECTORY));
		assertTrue(sftpService.ftpRemoveFileFromDirectory(indicatorFilename, DATA_DIRECTORY));
	}

	private String getArchiveDirectory() {
		return DATA_DIRECTORY + "/" + "archive";
	}

	private String getErrorDirectory() {
		return DATA_DIRECTORY + "/" + "error";
	}

}