package gov.va.cpss.model;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

import java.util.UnknownFormatConversionException;

import org.junit.Test;

import gov.va.cpss.model.AITCDollar.Sign;

/** 
 * Test for AITCDollar class.
 * 
 *
 */
public class AITCDollarTest {

	@Test
	public void testSign() {
		final Sign plus = Sign.PLUS;

		assertEquals(plus.getValue(), '+');
		assertEquals(plus.getSign(), 1);
		assertEquals(plus, Sign.from('+'));
		assertEquals(plus, Sign.from("+"));
		assertEquals(plus, Sign.from("+xxx"));
		assertEquals(plus, Sign.fromEnd("+"));
		assertEquals(plus, Sign.fromEnd("9+"));
		assertEquals(plus, Sign.fromEnd("11+"));
		assertEquals(plus, Sign.fromEnd("0000000000+"));
		assertEquals(plus, Sign.from(0d));
		assertEquals(plus, Sign.from(1d));

		final Sign minus = Sign.MINUS;

		assertEquals(minus.getValue(), '-');
		assertEquals(minus.getSign(), -1);
		assertEquals(minus, Sign.from('-'));
		assertEquals(minus, Sign.from("-"));
		assertEquals(minus, Sign.from("-xxx"));
		assertEquals(minus, Sign.fromEnd("-"));
		assertEquals(minus, Sign.fromEnd("9-"));
		assertEquals(minus, Sign.fromEnd("11-"));
		assertEquals(minus, Sign.fromEnd("0000000000-"));
		assertEquals(minus, Sign.from(-1d));

	}

	@Test(expected = UnknownFormatConversionException.class)
	public void testWrongSignFromChar1() {
		Sign.from((char) 0);
	}

	@Test(expected = UnknownFormatConversionException.class)
	public void testWrongSignFromChar2() {
		Sign.from('a');
	}

	@Test(expected = UnknownFormatConversionException.class)
	public void testWrongSignFromString1() {
		Sign.from((String) null);
	}

	@Test(expected = UnknownFormatConversionException.class)
	public void testWrongSignFromString2() {
		Sign.from("");
	}

	@Test(expected = UnknownFormatConversionException.class)
	public void testWrongSignFromString3() {
		Sign.from("a");
	}

	@Test(expected = UnknownFormatConversionException.class)
	public void testWrongSignFromString4() {
		Sign.from("aa");
	}

	@Test(expected = UnknownFormatConversionException.class)
	public void testWrongSignFromString5() {
		Sign.from("aa+");
	}

	@Test(expected = UnknownFormatConversionException.class)
	public void testWrongSignFromEndString1() {
		Sign.fromEnd((String) null);
	}

	@Test(expected = UnknownFormatConversionException.class)
	public void testWrongSignFromEndString2() {
		Sign.fromEnd("");
	}

	@Test(expected = UnknownFormatConversionException.class)
	public void testWrongSignFromEndString3() {
		Sign.fromEnd("a");
	}

	@Test(expected = UnknownFormatConversionException.class)
	public void testWrongSignFromEndString4() {
		Sign.fromEnd("aa");
	}

	@Test(expected = UnknownFormatConversionException.class)
	public void testWrongSignFromEndString5() {
		Sign.fromEnd("+aa");
	}

	@Test
	public void testConstructor() {
		final AITCDollar dollar1 = new AITCDollar(99d);

		assertEquals(dollar1.getDouble(), 99d, 0d);
		assertEquals(dollar1.toString(), "9900+");

		final AITCDollar dollar2 = new AITCDollar(-111d);

		assertEquals(dollar2.getDouble(), -111d, 0d);
		assertEquals(dollar2.toString(), "11100-");
	}

	@Test
	public void testFromString() {
		final AITCDollar posDollar1 = AITCDollar.fromString("9999999999+");
		assertNotNull(posDollar1);
		assertEquals(posDollar1.getDouble(), 99999999.99d, 0d);
		assertEquals(posDollar1.toString(), "9999999999+");

		final AITCDollar posDollar2 = AITCDollar.fromString("888+");
		assertNotNull(posDollar2);
		assertEquals(posDollar2.getDouble(), 8.88d, 0d);
		assertEquals(posDollar2.toString(), "888+");

		final AITCDollar posDollar3 = AITCDollar.fromString("77+");
		assertNotNull(posDollar3);
		assertEquals(posDollar3.getDouble(), 0.77d, 0d);
		assertEquals(posDollar3.toString(), "77+");

		final AITCDollar posDollar4 = AITCDollar.fromString("6+");
		assertNotNull(posDollar4);
		assertEquals(posDollar4.getDouble(), 0.06d, 0d);
		assertEquals(posDollar4.toString(), "6+");

		final AITCDollar posDollar5 = AITCDollar.fromString("0000000000+");
		assertNotNull(posDollar5);
		assertEquals(posDollar5.getDouble(), 0d, 0d);
		assertEquals(posDollar5.toString(), "0+");

		final AITCDollar posDollar6 = AITCDollar.fromString("0+");
		assertNotNull(posDollar6);
		assertEquals(posDollar6.getDouble(), 0d, 0d);
		assertEquals(posDollar6.toString(), "0+");

		final AITCDollar negDollar1 = AITCDollar.fromString("9999999999-");
		assertNotNull(negDollar1);
		assertEquals(negDollar1.getDouble(), -99999999.99d, 0d);
		assertEquals(negDollar1.toString(), "9999999999-");

		final AITCDollar negDollar2 = AITCDollar.fromString("888-");
		assertNotNull(negDollar2);
		assertEquals(negDollar2.getDouble(), -8.88d, 0d);
		assertEquals(negDollar2.toString(), "888-");

		final AITCDollar negDollar3 = AITCDollar.fromString("77-");
		assertNotNull(negDollar3);
		assertEquals(negDollar3.getDouble(), -0.77d, 0d);
		assertEquals(negDollar3.toString(), "77-");

		final AITCDollar negDollar4 = AITCDollar.fromString("6-");
		assertNotNull(negDollar4);
		assertEquals(negDollar4.getDouble(), -0.06d, 0d);
		assertEquals(negDollar4.toString(), "6-");

		final AITCDollar negDollar5 = AITCDollar.fromString("0000000000-");
		assertNotNull(negDollar5);
		assertEquals(negDollar5.getDouble(), 0d, 0d);
		assertEquals(negDollar5.toString(), "0+");

		final AITCDollar negDollar6 = AITCDollar.fromString("0-");
		assertNotNull(negDollar6);
		assertEquals(negDollar6.getDouble(), 0d, 0d);
		assertEquals(negDollar6.toString(), "0+");

	}

	@Test(expected = IllegalArgumentException.class)
	public void testNullString() {
		AITCDollar.fromString(null);
	}

	@Test(expected = IllegalArgumentException.class)
	public void testEmptyString() {
		AITCDollar.fromString("+");
	}

	@Test(expected = IllegalArgumentException.class)
	public void testLength1String() {
		AITCDollar.fromString("+");
	}

	@Test(expected = IllegalArgumentException.class)
	public void testInvalidString() {
		AITCDollar.fromString("xx");
	}

	@Test
	public final void testPaddedString() {
		assertEquals("00004005+", new AITCDollar(40.05).toPaddedString(9));
		assertEquals("00004005-", new AITCDollar(-40.05).toPaddedString(9));
		assertEquals("00004000+", new AITCDollar(40.0).toPaddedString(9));
		assertEquals("00004000-", new AITCDollar(-40.0).toPaddedString(9));
		assertEquals("00000400-", new AITCDollar(-4).toPaddedString(9));
		assertEquals("00000400+", new AITCDollar(4).toPaddedString(9));
		assertEquals("12345678+", new AITCDollar(123456.78).toPaddedString(9));
		assertEquals("4005+", new AITCDollar(40.05).toPaddedString(5));
	}

	@Test(expected = IndexOutOfBoundsException.class)
	public final void testPaddedStringTooSmall() {
		new AITCDollar(40.05).toPaddedString(4);
	}

}
