package gov.va.cpss.mirth;

import static org.junit.Assert.*;
import static org.junit.Assume.assumeTrue;

import java.io.File;
import java.io.IOException;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.Iterator;
import java.util.List;

import javax.sql.DataSource;

import org.apache.log4j.Logger;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;
import org.springframework.jdbc.datasource.init.ScriptUtils;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.util.FileCopyUtils;

import com.dumbster.smtp.SmtpMessage;

import gov.va.cpss.dao.CBSAccountDAO;
import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.dao.VistaAccountDAO;
import gov.va.cpss.dao.VistaAcntHistDAO;
import gov.va.cpss.email.MockEmailServer;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.icn.VistaAcntHist;

/**
 * Integration tests for the Mirth CBSS Update ICN channels.
 * 
 * @author Brad Pickle
 *
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml",  "/test-email-server.xml", "/cpss-context.xml", "/mirth-test-context.xml" })
public class UpdateIcnTest {

	private final Logger logger = Logger.getLogger(this.getClass().getCanonicalName());

	@Value("${run.integration.mirth.test:false}")
	private Boolean runIntegrationMirthTest;

	private Connection jdbcConnection = null;

	@Autowired
	private ProcessStatusDAO processStatusDao;

	@Autowired
	private VistaAccountDAO vistaAccountDao;

	@Autowired
	private VistaAcntHistDAO vistaAcntHistDao;
	
	@Autowired
	private CBSAccountDAO cbsAccountDao;

	@Autowired
	private MockEmailServer<SmtpMessage> emailServer;

	private File mirthMockMVIHL7StagingDirectory;

	/**
	 * @param dataSource
	 */
	@Autowired
	public void setDataSource(DataSource dataSource) {
		try {
			this.jdbcConnection = dataSource.getConnection();
		} catch (SQLException e) {
			// Auto-generated catch block
			e.printStackTrace();
			this.logger.info("SQL Exception in getting JDBC connection.");
		}
	}

	/**
	 * The local directory path to stage HL7 message files for the Mock MVI
	 * sending channel to send to CBSS.
	 * 
	 * @param mirthMockMVIHL7StagingDirectory
	 */
	@Autowired
	public void setMirthMockMVIHL7StagingDirectory(final String mirthMockMVIHL7StagingDirectory) {
		this.mirthMockMVIHL7StagingDirectory = new File(mirthMockMVIHL7StagingDirectory);
	}

	/**
	 * Only run these tests if property is set to run integration mirth test.
	 * Setup test data - insert into DB before tests.
	 */
	@Before
	public final void beforeTest() {
		assumeTrue(runIntegrationMirthTest);
		setUp();
	}

	/**
	 * Setup test data - insert into DB before tests.
	 * 
	 */
	private void setUp() {
		if (runIntegrationMirthTest) {
			emailServer.stop();
			emailServer.start();
			Resource sqlResource = new ClassPathResource("mirth/sql/UpdateIcnTest-InsertData.sql");
			ScriptUtils.executeSqlScript(this.jdbcConnection, sqlResource);
		}
	}

	/**
	 * Tear down test data - delete from DB after tests.
	 * 
	 */
	@After
	public void tearDown() {
		if (runIntegrationMirthTest) {
			emailServer.stop();
			Resource sqlResource = new ClassPathResource("mirth/sql/UpdateIcnTest-DeleteData.sql");
			ScriptUtils.executeSqlScript(this.jdbcConnection, sqlResource);
		}
	}

	@Test
	public void testA24() throws Exception {

		final int newStatusId = processStatusDao.getStatusFromEnum(ProcessStatus.Status.NEW);

		final long dfn1 = 111111111111111L;

		final String stationNum1 = "REG1";

		final String oldICN1 = "ICN1234567890REG1";
		final String newICN1 = "ICNNEW4567890REG1";

		assertTrue(vistaAccountDao.exists(dfn1, stationNum1));

		copyToMockMVIHL7StagingDirectory("mirth/hl7/ADT-A24-Test1.hl7");

		long totalWaitTimeMillis = 0L;

		totalWaitTimeMillis += MirthTestUtils.waitForCompletion(
				MirthTestUtils.WAIT_TIME_MAX_MILLIS - totalWaitTimeMillis, waitForVistaAcntHist(dfn1, stationNum1));
		checkVistaAcntHist(dfn1, stationNum1, oldICN1, newICN1, cbsAccountDao.getByICN(oldICN1), newStatusId);
		
		assertEquals(0, emailServer.getReceivedEmailSize());
	}

	@Test
	public void testA43() throws Exception {

		final int newStatusId = processStatusDao.getStatusFromEnum(ProcessStatus.Status.NEW);

		final long dfn2 = 222222222222222L;

		final String stationNum2 = "REG2";

		final String oldICN2 = "ICN1234567890REG2";
		final String newICN2 = "ICNNEW4567890REG2";

		assertTrue(vistaAccountDao.exists(dfn2, stationNum2));

		copyToMockMVIHL7StagingDirectory("mirth/hl7/ADT-A43-Test1.hl7");

		long totalWaitTimeMillis = 0L;

		totalWaitTimeMillis += MirthTestUtils.waitForCompletion(
				MirthTestUtils.WAIT_TIME_MAX_MILLIS - totalWaitTimeMillis, waitForVistaAcntHist(dfn2, stationNum2));
		checkVistaAcntHist(dfn2, stationNum2, oldICN2, newICN2, cbsAccountDao.getByICN(oldICN2), newStatusId);
		
		assertEquals(0, emailServer.getReceivedEmailSize());
	}

	@Test
	public void testMVICommitRejectForALAck() throws Exception {

		final int newStatusId = processStatusDao.getStatusFromEnum(ProcessStatus.Status.NEW);

		final long dfn3 = 333333333333333L;

		final String stationNum3 = "REG3";

		final String oldICN3 = "ICN1234567890REG3";
		final String newICN3 = "ICNNEW4567890REG3";

		assertTrue(vistaAccountDao.exists(dfn3, stationNum3));

		copyToMockMVIHL7StagingDirectory("mirth/hl7/ADT-A24-Reject.hl7");

		long totalWaitTimeMillis = 0L;

		totalWaitTimeMillis += MirthTestUtils.waitForCompletion(
				MirthTestUtils.WAIT_TIME_MAX_MILLIS - totalWaitTimeMillis, waitForVistaAcntHist(dfn3, stationNum3));
		checkVistaAcntHist(dfn3, stationNum3, oldICN3, newICN3, cbsAccountDao.getByICN(oldICN3), newStatusId);
		
		// Wait for error notification emails
		totalWaitTimeMillis += MirthTestUtils.waitForCompletion(
				MirthTestUtils.WAIT_TIME_MAX_MILLIS - totalWaitTimeMillis, waitForErrorEmail(1));
		
		assertEquals(1, emailServer.getReceivedEmailSize());
		Iterator<SmtpMessage> iter = emailServer.getReceivedEmail();
		assertNotNull(iter);
		assertTrue(iter.hasNext());
		SmtpMessage message = iter.next();
		assertNotNull(message);
		assertEquals("Update ICNs", message.getHeaderValue("Subject"));
		String messageBody = message.getBody();
		assertNotNull(messageBody);
		messageBody.contains("Message Type: ACK");
		messageBody.contains("Trigger Event: A24");

	}

	private void copyToMockMVIHL7StagingDirectory(final String hl7FilePath) throws IOException {
		assertNotNull(mirthMockMVIHL7StagingDirectory);
		assertTrue(mirthMockMVIHL7StagingDirectory.isDirectory());

		final ClassLoader classLoader = this.getClass().getClassLoader();
		final File hl7File = new File(classLoader.getResource(hl7FilePath).getFile());
		assertTrue(hl7File.isFile());

		final File destinationHL7File = new File(mirthMockMVIHL7StagingDirectory, hl7File.getName());

		FileCopyUtils.copy(hl7File, destinationHL7File);
	}

	private CompletionIndicator waitForVistaAcntHist(final long dfn, final String stationNum) {
		return new CompletionIndicator() {
			@Override
			public boolean complete() {
				return existsVistaAcntHist(dfn, stationNum);
			}
		};
	}

	private boolean existsVistaAcntHist(final long dfn, final String stationNum) {
		List<VistaAcntHist> vistaAcntHistL = vistaAcntHistDao.selectByDfnSite(dfn, stationNum);

		return ((vistaAcntHistL != null) && (vistaAcntHistL.size() > 0));
	}
	
	private void checkVistaAcntHist(final long dfn, final String stationNum, final String oldICN, final String newICN,
			final long oldCbssAcntId, final int statusId) {
		List<VistaAcntHist> vistaAcntHistL = vistaAcntHistDao.selectByDfnSite(dfn, stationNum);
		assertNotNull(vistaAcntHistL);
		assertEquals(1, vistaAcntHistL.size());
		VistaAcntHist vistaAcntHist = vistaAcntHistL.get(0);
		assertNotNull(vistaAcntHist);
		assertEquals(dfn, vistaAcntHist.getDfn());
		assertEquals(stationNum, vistaAcntHist.getStationNum());
		assertEquals(oldICN, vistaAcntHist.getOldIcn());
		assertEquals(newICN, vistaAcntHist.getNewIcn());
		assertEquals(oldCbssAcntId, vistaAcntHist.getOldCbssAcntId());
		assertEquals(statusId, vistaAcntHist.getStatusId());
	}

	private CompletionIndicator waitForErrorEmail(final int expectedNumEmails) {
		return new CompletionIndicator() {
			@Override
			public boolean complete() {
				return (emailServer.getReceivedEmailSize() >= expectedNumEmails);
			}
		};
	}

}
