package gov.va.cpss.mirth;

import static org.junit.Assert.*;
import static org.junit.Assume.assumeTrue;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.sql.DataSource;

import org.apache.log4j.Logger;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;
import org.springframework.jdbc.datasource.init.ScriptUtils;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import com.dumbster.smtp.SmtpMessage;

import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.dao.VistaAccountDAO;
import gov.va.cpss.dao.VistaAcntHistDAO;
import gov.va.cpss.email.MockEmailServer;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.icn.VistaAccount;
import gov.va.cpss.model.icn.VistaAcntHist;

/**
 * Integration tests for the Mirth CBSS Send Register DFN channel.
 * 
 * @author Brad Pickle
 *
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/test-email-server.xml", "/cpss-context.xml" })
public class RegisterDfnTest {

	private final Logger logger = Logger.getLogger(this.getClass().getCanonicalName());

	@Value("${run.integration.mirth.test:false}")
	private Boolean runIntegrationMirthTest;

	private Connection jdbcConnection = null;

	@Autowired
	private ProcessStatusDAO processStatusDao;

	@Autowired
	private VistaAccountDAO vistaAccountDao;

	@Autowired
	private VistaAcntHistDAO vistaAcntHistDao;

	@Autowired
	private MockEmailServer<SmtpMessage> emailServer;

	/**
	 * @param dataSource
	 */
	@Autowired
	public void setDataSource(DataSource dataSource) {
		try {
			this.jdbcConnection = dataSource.getConnection();
		} catch (SQLException e) {
			// Auto-generated catch block
			e.printStackTrace();
			this.logger.info("SQL Exception in getting JDBC connection.");
		}
	}

	/**
	 * Only run these tests if property is set to run integration mirth test.
	 * Setup test data - insert into DB before tests.
	 */
	@Before
	public final void beforeTest() {
		assumeTrue(runIntegrationMirthTest);
		setUp();
	}

	/**
	 * Setup test data - insert into DB before tests.
	 * 
	 */
	private void setUp() {
		if (runIntegrationMirthTest) {
			emailServer.stop();
			emailServer.start();
			Resource sqlResource = new ClassPathResource("mirth/sql/RegisterDfnTest-InsertData.sql");
			ScriptUtils.executeSqlScript(this.jdbcConnection, sqlResource);
		}
	}

	/**
	 * Tear down test data - delete from DB after tests.
	 * 
	 */
	@After
	public void tearDown() {
		if (runIntegrationMirthTest) {
			emailServer.stop();
			Resource sqlResource = new ClassPathResource("mirth/sql/RegisterDfnTest-DeleteData.sql");
			ScriptUtils.executeSqlScript(this.jdbcConnection, sqlResource);
		}
	}

	@Test
	public void test() throws InterruptedException {
		final int successStatusId = processStatusDao.getStatusFromEnum(ProcessStatus.Status.SUCCESS);
		final int ackStatusId = processStatusDao.getStatusFromEnum(ProcessStatus.Status.ACK);
		final int errorStatusId = processStatusDao.getStatusFromEnum(ProcessStatus.Status.ERROR);
		final int newStatusId = processStatusDao.getStatusFromEnum(ProcessStatus.Status.NEW);

		final long dfn1 = 111111111111111L;
		final long dfn2 = 222222222222222L;
		final long dfn3 = 333333333333333L;
		final long dfn4 = 444444444444444L;
		final long dfn5 = 555555555555555L;
		final long dfn6 = 666666666666666L;

		final String stationNum1 = "REG1";
		final String stationNum2 = "REG2";
		final String stationNum3 = "REG3";
		final String stationNum4 = "REG4";
		final String stationNum5 = "REG5";
		final String stationNum6 = "REG6";

		final String oldICN5 = "ICN1234567890REG5";
		final String newICN5 = "ICNNEW4567890REG5";

		assertTrue(vistaAccountDao.exists(dfn1, stationNum1));
		assertTrue(vistaAccountDao.exists(dfn2, stationNum2));
		assertTrue(vistaAccountDao.exists(dfn3, stationNum3));
		assertTrue(vistaAccountDao.exists(dfn4, stationNum4));
		assertTrue(vistaAccountDao.exists(dfn5, stationNum5));
		assertTrue(vistaAccountDao.exists(dfn6, stationNum6));

		long totalWaitTimeMillis = 0L;

		// DFN1/StationNum1 starts in NEW status, expect SUCCESS
		totalWaitTimeMillis += MirthTestUtils.waitForCompletion(
				MirthTestUtils.WAIT_TIME_MAX_MILLIS - totalWaitTimeMillis,
				waitForVistaAccountStatus(dfn1, stationNum1, successStatusId));

		// DFN2/StationNum2 starts in ACK status/modifiedDate one day ago,
		// expect SUCCESS
		totalWaitTimeMillis += MirthTestUtils.waitForCompletion(
				MirthTestUtils.WAIT_TIME_MAX_MILLIS - totalWaitTimeMillis,
				waitForVistaAccountStatus(dfn2, stationNum2, successStatusId));

		// DFN3/StationNum3 starts in NEW status, MockMVI will not respond,
		// expect ACK
		totalWaitTimeMillis += MirthTestUtils.waitForCompletion(
				MirthTestUtils.WAIT_TIME_MAX_MILLIS - totalWaitTimeMillis,
				waitForVistaAccountStatus(dfn3, stationNum3, ackStatusId));

		// DFN4/StationNum4 starts in NEW status, MockMVI will respond with
		// error, expect ERROR
		totalWaitTimeMillis += MirthTestUtils.waitForCompletion(
				MirthTestUtils.WAIT_TIME_MAX_MILLIS - totalWaitTimeMillis,
				waitForVistaAccountStatus(dfn4, stationNum4, errorStatusId));

		// DFN5/StationNum5 starts in NEW status, expect SUCCESS
		totalWaitTimeMillis += MirthTestUtils.waitForCompletion(
				MirthTestUtils.WAIT_TIME_MAX_MILLIS - totalWaitTimeMillis,
				waitForVistaAccountStatus(dfn5, stationNum5, successStatusId));

		// DFN6/StationNum6 starts in NEW status, MockMVI will send CR commit
		// level ack, expect ERROR
		totalWaitTimeMillis += MirthTestUtils.waitForCompletion(
				MirthTestUtils.WAIT_TIME_MAX_MILLIS - totalWaitTimeMillis,
				waitForVistaAccountStatus(dfn6, stationNum6, errorStatusId));

		// Wait for error notification emails
		totalWaitTimeMillis += MirthTestUtils.waitForCompletion(
				MirthTestUtils.WAIT_TIME_MAX_MILLIS - totalWaitTimeMillis, waitForErrorEmail(3));
		
		final VistaAccount vistaAccount = vistaAccountDao.getVistaAccount(dfn5, stationNum5);
		assertNotNull(vistaAccount);

		List<VistaAcntHist> vistaAcntHistL = vistaAcntHistDao.selectByDfnSite(dfn5, stationNum5);
		assertNotNull(vistaAcntHistL);
		assertEquals(1, vistaAcntHistL.size());
		VistaAcntHist vistaAcntHist = vistaAcntHistL.get(0);
		assertNotNull(vistaAcntHist);
		assertEquals(dfn5, vistaAcntHist.getDfn());
		assertEquals(stationNum5, vistaAcntHist.getStationNum());
		assertEquals(oldICN5, vistaAcntHist.getOldIcn());
		assertEquals(newICN5, vistaAcntHist.getNewIcn());
		assertEquals(vistaAccount.getCbssAcntId(), vistaAcntHist.getOldCbssAcntId());
		assertEquals(newStatusId, vistaAcntHist.getStatusId());

		assertEquals(3, emailServer.getReceivedEmailSize());

		Map<String, SmtpMessage> messages = new HashMap<String, SmtpMessage>();
		Pattern p = Pattern.compile("Site Number: ([^\\v]*)");
		for (Iterator<SmtpMessage> iter = emailServer.getReceivedEmail(); iter.hasNext();) {
			SmtpMessage message = iter.next();
			assertNotNull(message);

			String messageBody = message.getBody();
			assertNotNull(messageBody);

			Matcher m = p.matcher(messageBody);
			assertTrue(m.find());
			String siteNumber = m.group(1);
			messages.put(siteNumber, message);
		}

		final Map<String, String> dfnMap = new HashMap<String, String>();
		dfnMap.put(stationNum2, dfn2+"");
		dfnMap.put(stationNum4, dfn4+"");
		dfnMap.put(stationNum6, dfn6+"");
		for (String stationNum : Arrays.asList(stationNum2, stationNum4, stationNum6)) {
			SmtpMessage message = messages.get(stationNum);
			assertNotNull(message);
			
			assertEquals("Register DFNs", message.getHeaderValue("Subject"));
			String messageBody = message.getBody();
			assertTrue(messageBody.contains("DFN: " + dfnMap.get(stationNum)));
		}

	}

	private CompletionIndicator waitForVistaAccountStatus(final long dfn, final String stationNum, final int statusId) {
		return new CompletionIndicator() {
			@Override
			public boolean complete() {
				return getVistaAccountStatus(dfn, stationNum, statusId);
			}
		};
	}

	private boolean getVistaAccountStatus(final long dfn, final String stationNum, final int statusId) {

		final VistaAccount vistaAccount = vistaAccountDao.getVistaAccount(dfn, stationNum);
		assertNotNull(vistaAccount);

		return vistaAccount.getStatusId() == statusId;
	}

	private CompletionIndicator waitForErrorEmail(final int expectedNumEmails) {
		return new CompletionIndicator() {
			@Override
			public boolean complete() {
				return (emailServer.getReceivedEmailSize() >= expectedNumEmails);
			}
		};
	}

}
