package gov.va.cpss.mirth;

import static org.junit.Assert.*;
import static org.junit.Assume.assumeTrue;

import java.io.File;
import java.io.IOException;
import java.util.Iterator;

import org.apache.log4j.Logger;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.util.FileCopyUtils;

import com.dumbster.smtp.SmtpMessage;

import gov.va.cpss.email.MockEmailServer;

/**
 * Integration tests for the Mirth CBSS Update ICN channels.
 * 
 * @author Brad Pickle
 *
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml",  "/test-email-server.xml", "/cpss-context.xml", "/mirth-test-context.xml" })
public class CBSSReceiverTest {

	@SuppressWarnings("unused")
	private final Logger logger = Logger.getLogger(this.getClass().getCanonicalName());

	@Value("${run.integration.mirth.test:false}")
	private Boolean runIntegrationMirthTest;

	@Autowired
	private MockEmailServer<SmtpMessage> emailServer;

	private File mirthMockMVIHL7StagingDirectory;

	/**
	 * The local directory path to stage HL7 message files for the Mock MVI
	 * sending channel to send to CBSS.
	 * 
	 * @param mirthMockMVIHL7StagingDirectory
	 */
	@Autowired
	public void setMirthMockMVIHL7StagingDirectory(final String mirthMockMVIHL7StagingDirectory) {
		this.mirthMockMVIHL7StagingDirectory = new File(mirthMockMVIHL7StagingDirectory);
	}

	/**
	 * Only run these tests if property is set to run integration mirth test.
	 */
	@Before
	public final void beforeTest() {
		assumeTrue(runIntegrationMirthTest);
		setUp();
	}

	/**
	 * Start email server
	 * 
	 */
	private void setUp() {
		if (runIntegrationMirthTest) {
			emailServer.stop();
			emailServer.start();
		}
	}

	/**
	 * Stop email server
	 * 
	 */
	@After
	public void tearDown() {
		if (runIntegrationMirthTest) {
			emailServer.stop();
		}
	}

	@Test
	public void testUnhandledMessageType() throws Exception {

		copyToMockMVIHL7StagingDirectory("mirth/hl7/ADT-A31-Test1.hl7");

		long totalWaitTimeMillis = 0L;

		totalWaitTimeMillis += MirthTestUtils.waitForCompletion(
				MirthTestUtils.WAIT_TIME_MAX_MILLIS - totalWaitTimeMillis, waitForErrorEmail());
		
		assertEquals(1, emailServer.getReceivedEmailSize());
		
		Iterator<SmtpMessage> iter = emailServer.getReceivedEmail();
		assertNotNull(iter);
		assertTrue(iter.hasNext());
		SmtpMessage message = iter.next();
		assertNotNull(message);
		assertEquals("Update ICNs", message.getHeaderValue("Subject"));
		String messageBody = message.getBody();
		assertNotNull(messageBody);
		assertTrue(messageBody.contains("Message Type: ADT"));
		assertTrue(messageBody.contains("Trigger Event: A31"));
	}

	private void copyToMockMVIHL7StagingDirectory(final String hl7FilePath) throws IOException {
		assertNotNull(mirthMockMVIHL7StagingDirectory);
		assertTrue(mirthMockMVIHL7StagingDirectory.isDirectory());

		final ClassLoader classLoader = this.getClass().getClassLoader();
		final File hl7File = new File(classLoader.getResource(hl7FilePath).getFile());
		assertTrue(hl7File.isFile());

		final File destinationHL7File = new File(mirthMockMVIHL7StagingDirectory, hl7File.getName());

		FileCopyUtils.copy(hl7File, destinationHL7File);
	}

	private CompletionIndicator waitForErrorEmail() {
		return new CompletionIndicator() {
			@Override
			public boolean complete() {
				return (emailServer.getReceivedEmailSize() > 0);
			}
		};
	}

}
