package gov.va.cpss.job.loadbill;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assume.assumeTrue;

import java.util.Calendar;

import javax.sql.DataSource;

import org.apache.commons.lang.time.DateUtils;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.dao.SiteBalDAO;
import gov.va.cpss.model.bal.SiteBalance;

/**
 * Integration tests to test outcomes of the Load Bill batch job. NOTE: These
 * tests bypass the FTP server. Test files are found in
 * src/test/resources/loadbill.
 * 
 * @author Andrew Vance
 *
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml", "/cpss-batch.xml",
		"/cpss-batch-load-bill-data.xml", "/load-bill-data-job-test-context.xml",
		"/load-bill-data-job-bypass-ftp-test-context.xml", "/cpss-email.xml" })
public class LoadBillDataJobBatchIT {

	@Value("${run.integration.test:false}")
	private Boolean runIntegrationTest;

	@Autowired
	SiteBalDAO siteBalDAO;

	@Autowired
	TestLoadBillDataJob testLoadBillDataJob;

	private JdbcTemplate jdbcTemplate;

	@Autowired
	public void setDataSource(DataSource dataSource) {
		jdbcTemplate = new JdbcTemplate(dataSource);
	}

	@Before
	public final void beforeTest() {
		assumeTrue(runIntegrationTest);
		cleanup();

	}

	@After
	public final void afterTest() {
		if (runIntegrationTest) {
			cleanup();
		}
	}

	@Test
	public void testSuccessfulJob() {
		// These values must match payment_update_2.txt
		final long acntNum = 1112223334445557l;
		final String stationNum = "442";
		final double balance = 40.00;
		final Calendar oldestBillDateC = Calendar.getInstance();
		oldestBillDateC.set(2016, Calendar.MAY, 6);
		final Calendar processDateC = Calendar.getInstance();
		processDateC.set(2016, Calendar.JUNE, 7);

		testLoadBillDataJob.setFilePath("loadbill/payment_update_2.txt");

		boolean runSuccess = testLoadBillDataJob.testJob(true);

		assertTrue(runSuccess);

		final SiteBalance siteBalance = siteBalDAO.getByAcctNumAndStationNum(acntNum, stationNum);

		assertNotNull(siteBalance);
		assertEquals(balance, siteBalance.getBalance(), 0);
		// DateUtils.truncate is called to ensure the same date is verified
		// rather than the same millisecond.
		assertEquals(DateUtils.truncate(oldestBillDateC.getTime(), Calendar.DATE),
				DateUtils.truncate(siteBalance.getOldestBillDate(), Calendar.DATE));
		assertEquals(DateUtils.truncate(processDateC.getTime(), Calendar.DATE),
				DateUtils.truncate(siteBalance.getProcessDate(), Calendar.DATE));

	}

	/**
	 * Test the case that an existing patient's record is updated with new data.
	 */
	@Test
	public void testUpdateExisting() {
		// These values must match payment_update_3.txt and
		// payment_update_3_existing.txt
		final long acntNum = 1112223334445551l;
		final String stationNum = "442";
		final double balanceOriginal = 40.00;
		final double balanceUpdated = 123.45;

		// Insert original record
		testLoadBillDataJob.setFilePath("loadbill/payment_update_3.txt");
		boolean runSuccessOriginal = testLoadBillDataJob.testJob(true);
		assertTrue(runSuccessOriginal);
		final SiteBalance siteBalanceOriginal = siteBalDAO.getByAcctNumAndStationNum(acntNum, stationNum);
		assertNotNull(siteBalanceOriginal);
		assertEquals(balanceOriginal, siteBalanceOriginal.getBalance(), 0);

		// Update record
		testLoadBillDataJob.setFilePath("loadbill/payment_update_3_existing.txt");
		boolean runSuccessUpdated = testLoadBillDataJob.testJob(true);
		assertTrue(runSuccessUpdated);
		final SiteBalance siteBalanceUpdated = siteBalDAO.getByAcctNumAndStationNum(acntNum, stationNum);
		assertEquals(balanceUpdated, siteBalanceUpdated.getBalance(), 0);
	}

	public void cleanup() {
		final String sql = "DELETE FROM SiteBal WHERE DFN like ?";
		jdbcTemplate.update(sql, "111222333444555%");
		final String sql2 = "DELETE FROM vistaAccount WHERE DFN like ?";
		jdbcTemplate.update(sql2, "111222333444555%");
	}
}