package gov.va.cpss.job.fps;

import java.io.File;
import java.util.Arrays;
import java.util.List;

import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.JobParameters;

import gov.va.cpss.job.ProcessFPSDataJob;
import gov.va.cpss.model.BatchJob;
import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.fps.PSReceived;

/**
 * Test class that tests the correct processing execution of the
 * ProcessFPSDataJob using local file for test instead of ftp. It is important
 * to note that it bypasses the quartz scheduling portion of the batch process.
 * The intent is to support integration testing of the actual input file
 * processing business logic. This class extends ProcessFPSDataJob simply to
 * facilitate testing in the ProcessFPSDataJobBatchIT.
 * 
 * @author DNS
 */
public class ProcessFPSDataJobTest extends ProcessFPSDataJob {

	private PSReceived received;
	private JobExecution execution;

	/**
	 * Get the PSReceived object. This object is checked for expected values for
	 * verification tests.
	 * 
	 * @return PSReceived object.
	 */
	public PSReceived getReceived() {
		return received;
	}

	public JobExecution getExecution() {
		return execution;
	}

	/**
	 * Set an invalid file path.
	 */
	public void setInvalidFilePath() {
		filename = "fps/fake_file_path.txt";
	}

	/**
	 * Sets a local project file to use for test.
	 */
	public void setFilePath(final String fp) {
		ClassLoader classLoader = this.getClass().getClassLoader();
		File file = new File(classLoader.getResource(fp).getFile());
		this.filename = file.getAbsolutePath();
	}

	/**
	 * Method based on the parent class runJob method to capture the PSReceived
	 * object for verification tests. This method tests the supporting methods
	 * to verify proper handling of an input file.
	 * 
	 * @return Boolean value indicating if successful or not.
	 */
	public boolean testJob() {

		boolean successful = false;

		BatchJob bj = batchService.getBatchJobByName(name);
		BatchRun bR = batchService.startRun(bj.getId());

		resetMessages();

		received = initializeJob(bR.getId());

		JobParameters parameters = getParameters(received);

		execution = executeJob(getJob(), parameters);

		successful = fpsService.endProcessFpsDataJob(execution, received);

		if (successful) {
			completeRun(bR);
		} else {
			errorRun(bR);
		}

		return successful;
	}

	/**
	 * Method based on the parent class runJob method to run a job. This method
	 * tests the ftp interface to verify proper handling of an ftp input.
	 * 
	 * @return Boolean value indicating if successful or not.
	 */
	public boolean testFtpJob() {

		BatchJob bj = batchService.getBatchJobByName(name);
		BatchRun bR = batchService.startRun(bj.getId());

		resetMessages();

		final boolean successful = super.runJob(bR, getJob());

		if (successful) {
			completeRun(bR);
		} else {
			errorRun(bR);
		}

		return successful;
	}

	/**
	 * Method based on the parent class runJob method to run a job. This method
	 * tests the ftp interface to verify proper handling of an erroneous ftp
	 * input.
	 * 
	 * @return Boolean value indicating if successful or not.
	 */
	public boolean testErrorFtpJob() {

		BatchJob bj = batchService.getBatchJobByName(name);
		BatchRun bR = batchService.startRun(bj.getId());

		resetMessages();

		List<String> fileL = Arrays.asList("fake_file_path.don");

		final boolean successful = processList(bR.getId(), getJob(), fileL);

		if (successful) {
			completeRun(bR);
		} else {
			errorRun(bR);
		}

		return successful;
	}
}
