package gov.va.cpss.job.fps;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.io.File;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.batch.core.ExitStatus;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.job.CbssJobProcessingConstants;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.fps.PSReceived;

/**
 * Integration Unit Tests to test success and failure cases of Process FPS Data
 * batch processing. NOTE: These tests bypass the ftp server. Various test files
 * are referenced in src/test/resources/fps directory.
 * 
 * @author DNS 
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml", "/cpss-batch.xml",
		"/process-fps-data-job-test-context.xml", "/process-fps-data-job-ftp-test-context.xml", "/cpss-email.xml" })
public class ProcessFPSDataJobBatchIT {

	@Autowired
	ProcessFPSDataJobTest processFPSDataJobTest;

	@Autowired
	ProcessStatusDAO processStatusDAO;

	@Test
	public final void testInvalidZeroNumberPatientsJob() {
		
		final String filename = "fps/CPSS-to-CCPC_PS_invalid_zero_number_patients.txt";

		processFPSDataJobTest.setFilePath(filename);

		boolean runSuccess = processFPSDataJobTest.testJob();

		PSReceived pR = processFPSDataJobTest.getReceived();
		
		// Check PSReceived
		assertFalse(runSuccess);
		assertEquals(ProcessStatus.Status.ERROR, processStatusDAO.getStatusType(pR.getStatusId()).getStatus());
		assertEquals(0, pR.getNumOfSite());
		assertEquals(0, pR.getNumOfStatement());

		// Check exit status
		assertEquals(CbssJobProcessingConstants.PROCESSING_FAILURE_STATUS,
				processFPSDataJobTest.getExecution().getExitStatus().getExitCode());
		assertEquals(
				"\n\nError Message:\n" + (new File(filename)).getName() + ": "
						+ "Attempted to process PS with invalid total statement count (0)",
				processFPSDataJobTest.getErrorMessage());
	}
	
	@Test
	public final void testInvalidZeroNumberDetailsJob() {
		
		final String filename = "fps/CPSS-to-CCPC_PH_invalid_zero_number_details.txt";

		processFPSDataJobTest.setFilePath(filename);

		boolean runSuccess = processFPSDataJobTest.testJob();

		PSReceived pR = processFPSDataJobTest.getReceived();
		
		// Check PSReceived
		assertFalse(runSuccess);
		assertEquals(ProcessStatus.Status.ERROR, processStatusDAO.getStatusType(pR.getStatusId()).getStatus());
		assertEquals(0, pR.getNumOfSite());
		assertEquals(0, pR.getNumOfStatement());

		// Check exit status
		assertEquals(CbssJobProcessingConstants.PROCESSING_FAILURE_STATUS,
				processFPSDataJobTest.getExecution().getExitStatus().getExitCode());
		assertEquals(
				"\n\nError Message:\n" + (new File(filename)).getName() + ": "
						+ "Attempted to process PH with invalid total details count (0)",
				processFPSDataJobTest.getErrorMessage());
	}

	@Test
	public final void testSuccessfulJob() {

		processFPSDataJobTest.setFilePath("fps/CPSS-to-CCPC_successful.txt");

		boolean runSuccess = processFPSDataJobTest.testJob();

		PSReceived pR = processFPSDataJobTest.getReceived();

		// Check PSReceived
		assertTrue(runSuccess);
		assertEquals(ProcessStatus.Status.NEW, processStatusDAO.getStatusType(pR.getStatusId()).getStatus());
		assertEquals(3, pR.getNumOfSite());
		assertEquals(6, pR.getNumOfStatement());

		// Check exit status
		assertEquals(ExitStatus.COMPLETED, processFPSDataJobTest.getExecution().getExitStatus());
		assertNull(processFPSDataJobTest.getErrorMessage());
	}

	@Test
	public final void testInvalidSequenceNumberJob() {

		final String filename = "fps/CPSS-to-CCPC_PS_invalid_sequence_number.txt";

		processFPSDataJobTest.setFilePath(filename);

		boolean runSuccess = processFPSDataJobTest.testJob();

		PSReceived pR = processFPSDataJobTest.getReceived();

		// Check PSReceived
		assertFalse(runSuccess);
		assertEquals(ProcessStatus.Status.ERROR, processStatusDAO.getStatusType(pR.getStatusId()).getStatus());
		assertEquals(0, pR.getNumOfSite());
		assertEquals(0, pR.getNumOfStatement());

		// Check exit status
		assertEquals(CbssJobProcessingConstants.PROCESSING_FAILURE_STATUS,
				processFPSDataJobTest.getExecution().getExitStatus().getExitCode());
		assertEquals(
				"\n\nError Message:\n" + (new File(filename)).getName() + ": "
						+ "Attempted to process PS with invalid seq num (5) but expected (1)",
				processFPSDataJobTest.getErrorMessage());
	}

	@Test
	public final void testIncompleteFilePSJob() {

		final String filename = "fps/CPSS-to-CCPC_PS_invalid_end_of_file.txt";

		processFPSDataJobTest.setFilePath(filename);

		boolean runSuccess = processFPSDataJobTest.testJob();

		PSReceived pR = processFPSDataJobTest.getReceived();

		// Check PSReceived
		assertFalse(runSuccess);
		assertEquals(ProcessStatus.Status.ERROR, processStatusDAO.getStatusType(pR.getStatusId()).getStatus());
		assertEquals(0, pR.getNumOfSite());
		assertEquals(0, pR.getNumOfStatement());

		// Check exit status
		assertEquals(CbssJobProcessingConstants.INCOMPLETE_FILE_ERROR_STATUS,
				processFPSDataJobTest.getExecution().getExitStatus().getExitCode());
		assertEquals("\n\nError Message:\n" + (new File(filename)).getName() + ": "
				+ "PS count (1) of (2) indicates incomplete sequence", processFPSDataJobTest.getErrorMessage());
	}

	@Test
	public final void testIncompleteFilePHJob() {

		final String filename = "fps/CPSS-to-CCPC_PH_invalid_end_of_file.txt";

		processFPSDataJobTest.setFilePath(filename);

		boolean runSuccess = processFPSDataJobTest.testJob();

		PSReceived pR = processFPSDataJobTest.getReceived();

		// Check PSReceived
		assertFalse(runSuccess);
		assertEquals(ProcessStatus.Status.ERROR, processStatusDAO.getStatusType(pR.getStatusId()).getStatus());
		assertEquals(0, pR.getNumOfSite());
		assertEquals(0, pR.getNumOfStatement());

		// Check exit status
		assertEquals(CbssJobProcessingConstants.INCOMPLETE_FILE_ERROR_STATUS,
				processFPSDataJobTest.getExecution().getExitStatus().getExitCode());
		assertEquals(
				"\n\nError Message:\n" + (new File(filename)).getName() + ": "
						+ "PH count (3) of (4) indicates incomplete statement",
				processFPSDataJobTest.getErrorMessage());
	}

	@Test
	public final void testIncompleteFilePDJob() {

		final String filename = "fps/CPSS-to-CCPC_PD_invalid_end_of_file.txt";

		processFPSDataJobTest.setFilePath(filename);

		boolean runSuccess = processFPSDataJobTest.testJob();

		PSReceived pR = processFPSDataJobTest.getReceived();

		// Check PSReceived
		assertFalse(runSuccess);
		assertEquals(ProcessStatus.Status.ERROR, processStatusDAO.getStatusType(pR.getStatusId()).getStatus());
		assertEquals(0, pR.getNumOfSite());
		assertEquals(0, pR.getNumOfStatement());

		// Check exit status
		assertEquals(CbssJobProcessingConstants.INCOMPLETE_FILE_ERROR_STATUS,
				processFPSDataJobTest.getExecution().getExitStatus().getExitCode());
		assertEquals("\n\nError Message:\n" + (new File(filename)).getName() + ": "
				+ "PD count (1) of (2) indicates incomplete details", processFPSDataJobTest.getErrorMessage());
	}

	@Test
	public final void testInvalidStatementCountLowJob() {

		final String filename = "fps/CPSS-to-CCPC_PS_invalid_statement_count_low.txt";

		processFPSDataJobTest.setFilePath(filename);

		boolean runSuccess = processFPSDataJobTest.testJob();

		PSReceived pR = processFPSDataJobTest.getReceived();

		// Check PSReceived
		assertFalse(runSuccess);
		assertEquals(ProcessStatus.Status.ERROR, processStatusDAO.getStatusType(pR.getStatusId()).getStatus());
		assertEquals(0, pR.getNumOfSite());
		assertEquals(0, pR.getNumOfStatement());

		// Check exit status
		assertEquals(CbssJobProcessingConstants.PROCESSING_FAILURE_STATUS,
				processFPSDataJobTest.getExecution().getExitStatus().getExitCode());
		assertEquals(
				"\n\nError Message:\n" + (new File(filename)).getName() + ": "
						+ "PH count (3) of (4) indicates incomplete statement",
				processFPSDataJobTest.getErrorMessage());
	}

	@Test
	public final void testInvalidStatementCountHighJob() {

		final String filename = "fps/CPSS-to-CCPC_PS_invalid_statement_count_high.txt";

		processFPSDataJobTest.setFilePath(filename);

		boolean runSuccess = processFPSDataJobTest.testJob();

		PSReceived pR = processFPSDataJobTest.getReceived();

		// Check PSReceived
		assertFalse(runSuccess);
		assertEquals(ProcessStatus.Status.ERROR, processStatusDAO.getStatusType(pR.getStatusId()).getStatus());
		assertEquals(0, pR.getNumOfSite());
		assertEquals(0, pR.getNumOfStatement());

		// Check exit status
		assertEquals(CbssJobProcessingConstants.PROCESSING_FAILURE_STATUS,
				processFPSDataJobTest.getExecution().getExitStatus().getExitCode());
		assertEquals(
				"\n\nError Message:\n" + (new File(filename)).getName() + ": "
						+ "Attempted to process PH with unexpected count (3) but expected (1)",
				processFPSDataJobTest.getErrorMessage());
	}

	@Test
	public final void testInvalidDetailsCountLowJob() {

		final String filename = "fps/CPSS-to-CCPC_PH_invalid_details_count_low.txt";

		processFPSDataJobTest.setFilePath(filename);

		boolean runSuccess = processFPSDataJobTest.testJob();

		PSReceived pR = processFPSDataJobTest.getReceived();

		// Check PSReceived
		assertFalse(runSuccess);
		assertEquals(ProcessStatus.Status.ERROR, processStatusDAO.getStatusType(pR.getStatusId()).getStatus());
		assertEquals(0, pR.getNumOfSite());
		assertEquals(0, pR.getNumOfStatement());

		// Check exit status
		assertEquals(CbssJobProcessingConstants.PROCESSING_FAILURE_STATUS,
				processFPSDataJobTest.getExecution().getExitStatus().getExitCode());
		assertEquals("\n\nError Message:\n" + (new File(filename)).getName() + ": "
				+ "PD count (2) of (3) indicates incomplete details", processFPSDataJobTest.getErrorMessage());
	}

	@Test
	public final void testInvalidDetailsCountHighJob() {

		final String filename = "fps/CPSS-to-CCPC_PH_invalid_details_count_high.txt";

		processFPSDataJobTest.setFilePath(filename);

		boolean runSuccess = processFPSDataJobTest.testJob();

		PSReceived pR = processFPSDataJobTest.getReceived();

		// Check PSReceived
		assertFalse(runSuccess);
		assertEquals(ProcessStatus.Status.ERROR, processStatusDAO.getStatusType(pR.getStatusId()).getStatus());
		assertEquals(0, pR.getNumOfSite());
		assertEquals(0, pR.getNumOfStatement());

		// Check exit status
		assertEquals(CbssJobProcessingConstants.PROCESSING_FAILURE_STATUS,
				processFPSDataJobTest.getExecution().getExitStatus().getExitCode());
		assertEquals(
				"\n\nError Message:\n" + (new File(filename)).getName() + ": "
						+ "Attempted to process PD with unexpected count (2) but expected (1)",
				processFPSDataJobTest.getErrorMessage());
	}

	@Test
	public final void testInvalidType() {

		final String filename = "fps/CCPC-to-CPSS_invalid_type.txt";

		processFPSDataJobTest.setFilePath(filename);

		boolean runSuccess = processFPSDataJobTest.testJob();

		PSReceived pR = processFPSDataJobTest.getReceived();

		// Check PSReceived
		assertFalse(runSuccess);
		assertEquals(ProcessStatus.Status.ERROR, processStatusDAO.getStatusType(pR.getStatusId()).getStatus());
		assertEquals(0, pR.getNumOfSite());
		assertEquals(0, pR.getNumOfStatement());

		// Check exit status
		assertEquals(CbssJobProcessingConstants.READ_FAILURE_STATUS,
				processFPSDataJobTest.getExecution().getExitStatus().getExitCode());
		assertTrue(
				processFPSDataJobTest.getErrorMessage().contains("Cause: Could not find a matching pattern for key"));
	}

	@Test
	public final void testBadFilePath() {

		processFPSDataJobTest.setInvalidFilePath();

		boolean runSuccess = processFPSDataJobTest.testJob();

		PSReceived pR = processFPSDataJobTest.getReceived();

		// Check PSReceived
		assertFalse(runSuccess);
		assertEquals(ProcessStatus.Status.ERROR, processStatusDAO.getStatusType(pR.getStatusId()).getStatus());
		assertEquals(0, pR.getNumOfSite());
		assertEquals(0, pR.getNumOfStatement());

		// Check exit status
		assertEquals(CbssJobProcessingConstants.FILE_OPEN_ERROR_STATUS,
				processFPSDataJobTest.getExecution().getExitStatus().getExitCode());
		assertTrue(processFPSDataJobTest.getErrorMessage().contains("Input resource must exist"));
	}

	@Test
	public final void testEmptyFilePath() {

		final String filename = "fps/CCPC-to-CPSS_empty.txt";

		processFPSDataJobTest.setFilePath(filename);

		boolean runSuccess = processFPSDataJobTest.testJob();

		PSReceived pR = processFPSDataJobTest.getReceived();

		// Check PSReceived
		assertFalse(runSuccess);
		assertEquals(ProcessStatus.Status.ERROR, processStatusDAO.getStatusType(pR.getStatusId()).getStatus());
		assertEquals(0, pR.getNumOfSite());
		assertEquals(0, pR.getNumOfStatement());

		// Check exit status
		assertEquals(CbssJobProcessingConstants.EMPTY_FILE_ERROR_STATUS,
				processFPSDataJobTest.getExecution().getExitStatus().getExitCode());
		assertEquals("\n\nError Message:\n" + (new File(filename)).getName() + ": " + "Input file is empty",
				processFPSDataJobTest.getErrorMessage());
	}

	@Test
	public final void testBadAddressFlagReadFormat() {

		final String filename = "fps/CCPC-to-CPSS_PH_bad_address_flag.txt";

		processFPSDataJobTest.setFilePath(filename);

		boolean runSuccess = processFPSDataJobTest.testJob();

		PSReceived pR = processFPSDataJobTest.getReceived();

		// Check PSReceived
		assertFalse(runSuccess);
		assertEquals(ProcessStatus.Status.ERROR, processStatusDAO.getStatusType(pR.getStatusId()).getStatus());
		assertEquals(0, pR.getNumOfSite());
		assertEquals(0, pR.getNumOfStatement());

		// Check exit status
		assertEquals(CbssJobProcessingConstants.READ_FAILURE_STATUS,
				processFPSDataJobTest.getExecution().getExitStatus().getExitCode());
		assertTrue(processFPSDataJobTest.getErrorMessage()
				.contains("Field error in object 'target' on field 'addressFlag': rejected value [R]"));
	}

	@Test
	public final void testBadAddressFlagReadFormat2() {

		final String filename = "fps/CCPC-to-CPSS_PH_bad_address_flag_empty.txt";

		processFPSDataJobTest.setFilePath(filename);

		boolean runSuccess = processFPSDataJobTest.testJob();

		PSReceived pR = processFPSDataJobTest.getReceived();

		// Check PSReceived
		assertFalse(runSuccess);
		assertEquals(ProcessStatus.Status.ERROR, processStatusDAO.getStatusType(pR.getStatusId()).getStatus());
		assertEquals(0, pR.getNumOfSite());
		assertEquals(0, pR.getNumOfStatement());

		// Check exit status
		assertEquals(CbssJobProcessingConstants.READ_FAILURE_STATUS,
				processFPSDataJobTest.getExecution().getExitStatus().getExitCode());
		assertTrue(processFPSDataJobTest.getErrorMessage()
				.contains("Field error in object 'target' on field 'addressFlag': rejected value []"));
	}

	@Test
	public final void testBadCobolFormat() {

		final String filename = "fps/CCPC-to-CPSS_PS_invalid_statement_val.txt";

		processFPSDataJobTest.setFilePath(filename);

		boolean runSuccess = processFPSDataJobTest.testJob();

		PSReceived pR = processFPSDataJobTest.getReceived();

		// Check PSReceived
		assertFalse(runSuccess);
		assertEquals(ProcessStatus.Status.ERROR, processStatusDAO.getStatusType(pR.getStatusId()).getStatus());
		assertEquals(0, pR.getNumOfSite());
		assertEquals(0, pR.getNumOfStatement());

		// Check exit status
		assertEquals(CbssJobProcessingConstants.READ_FAILURE_STATUS,
				processFPSDataJobTest.getExecution().getExitStatus().getExitCode());
		assertTrue(processFPSDataJobTest.getErrorMessage()
				.contains("Field error in object 'target' on field 'statementVal': rejected value [0000055442Z]"));
	}
}