package gov.va.cpss.job.fps;

import org.junit.Before;
import static org.junit.Assume.assumeTrue;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.io.File;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.service.SftpService;

/**
 * Integration Unit Tests to test success and failure cases of Process FPS Data
 * batch processing using the ftp interface. NOTE: These tests use the ftp
 * server. Various test files are referenced in src/test/resources/fps
 * directory.
 * 
 * @author DNS   
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml", "/cpss-batch.xml",
		"/process-fps-data-job-ftp-test-context.xml", "/cpss-email.xml" })
public class ProcessFPSDataJobBatchFtpIT {

	@Value("${run.integration.test:false}")
	private Boolean runIntegrationTest;
	
	@Autowired
	private ProcessFPSDataJobTest processFPSDataJobTest;

	@Autowired
	private SftpService sftpService;
	
	@Autowired
	private String processFPSServerTargetDirectory;
	@Autowired
	private String processFPSServerArchiveTargetDirectory;
	
	@Autowired
	private String processFPSServerErrorTargetDirectory;
	/**
	 * Only run these tests if property is set to run integration test.
	 */
	@Before
    public final void beforeTest() {
    	assumeTrue(runIntegrationTest);
    }
	
	@Test
	public final void testInvalidZeroNumberPatientsJob() {

		final String filename = "CPSS-to-CCPC_PS_invalid_zero_number_patients.txt";
		final String filepath = "fps/" + filename;

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// FTP test file to server.
		assertTrue(ftpFileToServer(filepath));

		// Run the job.
		final boolean runSuccess = processFPSDataJobTest.testFtpJob();

		// Verify the job was unsuccessful.
		assertFalse(runSuccess);
		assertEquals(
				"\n\nError Message:\n" + (new File(filename)).getName() + ": "
						+ "Attempted to process PS with invalid total statement count (0)",
				processFPSDataJobTest.getErrorMessage());

		// Verify file is not in the data directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));

		// Verify file is not in the archive directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));

		// Verify file is in the error directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, getErrorDirectory()));
	}
	
	@Test
	public final void testInvalidZeroNumberDetailsJob() {

		final String filename = "CPSS-to-CCPC_PH_invalid_zero_number_details.txt";
		final String filepath = "fps/" + filename;

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// FTP test file to server.
		assertTrue(ftpFileToServer(filepath));

		// Run the job.
		final boolean runSuccess = processFPSDataJobTest.testFtpJob();

		// Verify the job was unsuccessful.
		assertFalse(runSuccess);
		assertEquals(
				"\n\nError Message:\n" + (new File(filename)).getName() + ": "
						+ "Attempted to process PH with invalid total details count (0)",
				processFPSDataJobTest.getErrorMessage());

		// Verify file is not in the data directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));

		// Verify file is not in the archive directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));

		// Verify file is in the error directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, getErrorDirectory()));
	}

	@Test
	public final void testSuccessfulJob() {

		final String filename = "CPSS-to-CCPC_successful.txt";
		final String filepath = "fps/" + filename;

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// FTP test file to server.
		assertTrue(ftpFileToServer(filepath));

		// Run the job.
		final boolean runSuccess = processFPSDataJobTest.testFtpJob();

		// Verify the job was successful.
		assertTrue(runSuccess);

		// Check exit status
		assertNull(processFPSDataJobTest.getErrorMessage());

		// Verify file is not in the data directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));

		// Verify file is in the archive directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));

		// Verify file is not in the error directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, getArchiveDirectory()));
	}

	@Test
	public final void testInvalidSequenceNumberJob() {

		final String filename = "CPSS-to-CCPC_PS_invalid_sequence_number.txt";
		final String filepath = "fps/" + filename;

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// FTP test file to server.
		assertTrue(ftpFileToServer(filepath));

		// Run the job.
		final boolean runSuccess = processFPSDataJobTest.testFtpJob();

		// Verify the job was unsuccessful.
		assertFalse(runSuccess);
		assertEquals(
				"\n\nError Message:\n" + (new File(filename)).getName() + ": "
						+ "Attempted to process PS with invalid seq num (5) but expected (1)",
				processFPSDataJobTest.getErrorMessage());

		// Verify file is not in the data directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));

		// Verify file is not in the archive directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));

		// Verify file is in the error directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, getErrorDirectory()));
	}

	@Test
	public final void testIncompleteFilePSJob() {

		final String filename = "CPSS-to-CCPC_PS_invalid_end_of_file.txt";
		final String filepath = "fps/" + filename;

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// FTP test file to server.
		assertTrue(ftpFileToServer(filepath));

		// Run the job.
		final boolean runSuccess = processFPSDataJobTest.testFtpJob();

		// Verify the job was unsuccessful.
		assertFalse(runSuccess);
		assertEquals("\n\nError Message:\n" + (new File(filename)).getName() + ": "
				+ "PS count (1) of (2) indicates incomplete sequence", processFPSDataJobTest.getErrorMessage());

		// Verify file is not in the data directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));

		// Verify file is not in the archive directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));

		// Verify file is in the error directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, getErrorDirectory()));
	}

	@Test
	public final void testIncompleteFilePHJob() {

		final String filename = "CPSS-to-CCPC_PH_invalid_end_of_file.txt";
		final String filepath = "fps/" + filename;

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// FTP test file to server.
		assertTrue(ftpFileToServer(filepath));

		// Run the job.
		final boolean runSuccess = processFPSDataJobTest.testFtpJob();

		// Verify the job was unsuccessful.
		assertFalse(runSuccess);
		assertEquals(
				"\n\nError Message:\n" + (new File(filename)).getName() + ": "
						+ "PH count (3) of (4) indicates incomplete statement",
				processFPSDataJobTest.getErrorMessage());

		// Verify file is not in the data directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));

		// Verify file is not in the archive directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));

		// Verify file is in the error directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, getErrorDirectory()));
	}

	@Test
	public final void testIncompleteFilePDJob() {

		final String filename = "CPSS-to-CCPC_PD_invalid_end_of_file.txt";
		final String filepath = "fps/" + filename;

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// FTP test file to server.
		assertTrue(ftpFileToServer(filepath));

		// Run the job.
		final boolean runSuccess = processFPSDataJobTest.testFtpJob();

		// Verify the job was unsuccessful.
		assertFalse(runSuccess);
		assertEquals("\n\nError Message:\n" + (new File(filename)).getName() + ": "
				+ "PD count (1) of (2) indicates incomplete details", processFPSDataJobTest.getErrorMessage());

		// Verify file is not in the data directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));

		// Verify file is not in the archive directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));

		// Verify file is in the error directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, getErrorDirectory()));
	}

	@Test
	public final void testInvalidStatementCountLowJob() {

		final String filename = "CPSS-to-CCPC_PS_invalid_statement_count_low.txt";
		final String filepath = "fps/" + filename;

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// FTP test file to server.
		assertTrue(ftpFileToServer(filepath));

		// Run the job.
		final boolean runSuccess = processFPSDataJobTest.testFtpJob();

		// Verify the job was unsuccessful.
		assertFalse(runSuccess);
		assertEquals(
				"\n\nError Message:\n" + (new File(filename)).getName() + ": "
						+ "PH count (3) of (4) indicates incomplete statement",
				processFPSDataJobTest.getErrorMessage());

		// Verify file is not in the data directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));

		// Verify file is not in the archive directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));

		// Verify file is in the error directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, getErrorDirectory()));
	}

	@Test
	public final void testInvalidStatementCountHighJob() {

		final String filename = "CPSS-to-CCPC_PS_invalid_statement_count_high.txt";
		final String filepath = "fps/" + filename;

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// FTP test file to server.
		assertTrue(ftpFileToServer(filepath));

		// Run the job.
		final boolean runSuccess = processFPSDataJobTest.testFtpJob();

		// Verify the job was unsuccessful.
		assertFalse(runSuccess);
		assertEquals(
				"\n\nError Message:\n" + (new File(filename)).getName() + ": "
						+ "Attempted to process PH with unexpected count (3) but expected (1)",
				processFPSDataJobTest.getErrorMessage());

		// Verify file is not in the data directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));

		// Verify file is not in the archive directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));

		// Verify file is in the error directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, getErrorDirectory()));
	}

	@Test
	public final void testInvalidDetailsCountLowJob() {

		final String filename = "CPSS-to-CCPC_PH_invalid_details_count_low.txt";
		final String filepath = "fps/" + filename;

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// FTP test file to server.
		assertTrue(ftpFileToServer(filepath));

		// Run the job.
		final boolean runSuccess = processFPSDataJobTest.testFtpJob();

		// Verify the job was unsuccessful.
		assertFalse(runSuccess);
		assertEquals("\n\nError Message:\n" + (new File(filename)).getName() + ": "
				+ "PD count (2) of (3) indicates incomplete details", processFPSDataJobTest.getErrorMessage());

		// Verify file is not in the data directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));

		// Verify file is not in the archive directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));

		// Verify file is in the error directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, getErrorDirectory()));
	}

	@Test
	public final void testInvalidDetailsCountHighJob() {

		final String filename = "CPSS-to-CCPC_PH_invalid_details_count_high.txt";
		final String filepath = "fps/" + filename;

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// FTP test file to server.
		assertTrue(ftpFileToServer(filepath));

		// Run the job.
		final boolean runSuccess = processFPSDataJobTest.testFtpJob();

		// Verify the job was unsuccessful.
		assertFalse(runSuccess);
		assertEquals(
				"\n\nError Message:\n" + (new File(filename)).getName() + ": "
						+ "Attempted to process PD with unexpected count (2) but expected (1)",
				processFPSDataJobTest.getErrorMessage());

		// Verify file is not in the data directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));

		// Verify file is not in the archive directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));

		// Verify file is in the error directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, getErrorDirectory()));
	}

	@Test
	public final void testInvalidType() {

		final String filename = "CCPC-to-CPSS_invalid_type.txt";
		final String filepath = "fps/" + filename;

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// FTP test file to server.
		assertTrue(ftpFileToServer(filepath));

		// Run the job.
		final boolean runSuccess = processFPSDataJobTest.testFtpJob();

		// Verify the job was unsuccessful.
		assertFalse(runSuccess);
		assertTrue(
				processFPSDataJobTest.getErrorMessage().contains("Cause: Could not find a matching pattern for key"));

		// Verify file is not in the data directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));

		// Verify file is not in the archive directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));

		// Verify file is in the error directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, getErrorDirectory()));
	}

	@Test
	public final void testNoFiles() {

		// Run the job.
		final boolean runSuccess = processFPSDataJobTest.testFtpJob();

		// Verify the job was unsuccessful.
		assertFalse(runSuccess);
		assertEquals("\n\nError Message:\n" + "No files found on ftp server to process",
				processFPSDataJobTest.getErrorMessage());
	}

	@Test
	public final void testEmptyFilePath() {

		// Note that empty file causes IOException when reading via ftp.

		final String filename = "CCPC-to-CPSS_empty.txt";
		final String filepath = "fps/" + filename;

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// FTP test file to server.
		assertTrue(ftpFileToServer(filepath));

		// Run the job.
		final boolean runSuccess = processFPSDataJobTest.testFtpJob();

		// Verify the job was unsuccessful.
		assertFalse(runSuccess);
		assertTrue(processFPSDataJobTest.getErrorMessage().contains("Input file is empty"));

		// Verify file is not in the data directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));

		// Verify file is not in the archive directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));

		// Verify file is in the error directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, getErrorDirectory()));
	}

	@Test
	public final void testBadAddressFlagReadFormat() {

		final String filename = "CCPC-to-CPSS_PH_bad_address_flag.txt";
		final String filepath = "fps/" + filename;

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// FTP test file to server.
		assertTrue(ftpFileToServer(filepath));

		// Run the job.
		final boolean runSuccess = processFPSDataJobTest.testFtpJob();

		// Verify the job was unsuccessful.
		assertFalse(runSuccess);
		assertTrue(processFPSDataJobTest.getErrorMessage()
				.contains("Field error in object 'target' on field 'addressFlag': rejected value [R]"));

		// Verify file is not in the data directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));

		// Verify file is not in the archive directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));

		// Verify file is in the error directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, getErrorDirectory()));
	}

	@Test
	public final void testBadAddressFlagReadFormat2() {

		final String filename = "CCPC-to-CPSS_PH_bad_address_flag_empty.txt";
		final String filepath = "fps/" + filename;

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// FTP test file to server.
		assertTrue(ftpFileToServer(filepath));

		// Run the job.
		final boolean runSuccess = processFPSDataJobTest.testFtpJob();

		// Verify the job was unsuccessful.
		assertFalse(runSuccess);
		assertTrue(processFPSDataJobTest.getErrorMessage()
				.contains("Field error in object 'target' on field 'addressFlag': rejected value []"));

		// Verify file is not in the data directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));

		// Verify file is not in the archive directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));

		// Verify file is in the error directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, getErrorDirectory()));
	}

	@Test
	public final void testBadCobolFormat() {

		final String filename = "CCPC-to-CPSS_PS_invalid_statement_val.txt";
		final String filepath = "fps/" + filename;

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// FTP test file to server.
		assertTrue(ftpFileToServer(filepath));

		// Run the job.
		final boolean runSuccess = processFPSDataJobTest.testFtpJob();

		// Verify the job was unsuccessful.
		assertFalse(runSuccess);
		assertTrue(processFPSDataJobTest.getErrorMessage()
				.contains("Field error in object 'target' on field 'statementVal': rejected value [0000055442Z]"));

		// Verify file is not in the data directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, processFPSServerTargetDirectory));

		// Verify file is not in the archive directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename, getArchiveDirectory()));

		// Verify file is in the error directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename, getErrorDirectory()));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename, getErrorDirectory()));
	}

	@Test
	public final void testBadFilePath() {

		// Run the job.
		final boolean runSuccess = processFPSDataJobTest.testErrorFtpJob();

		// Verify the job was unsuccessful.
		assertFalse(runSuccess);
		assertTrue(processFPSDataJobTest.getErrorMessage().contains("Unable to open ftp connection because failed file size request"));
	}

	@Test
	public final void testMultiSuccessJob() {

		final String filename1 = "CPSS-to-CCPC_successful.txt";
		final String filename2 = "CPSS-to-CCPC-double.txt";

		final String filepath1 = "fps/" + filename1;
		final String filepath2 = "cbs/" + filename2;

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename1, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename1, getArchiveDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename1, getErrorDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename2, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename2, getArchiveDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename2, getErrorDirectory()));

		// FTP test files to server.
		assertTrue(ftpFileToServer(filepath1));
		assertTrue(ftpFileToServer(filepath2));

		// Run the job.
		final boolean runSuccess = processFPSDataJobTest.testFtpJob();

		// Verify the job was successful.
		assertTrue(runSuccess);

		// Check exit status
		assertNull(processFPSDataJobTest.getErrorMessage());

		// Verify files are not in the data directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename1, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename2, processFPSServerTargetDirectory));

		// Verify files are in the archive directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename1, getArchiveDirectory()));
		assertTrue(sftpService.ftpFileExistsInDirectory(filename2, getArchiveDirectory()));

		// Verify files are not in the error directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename1, getErrorDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename2, getErrorDirectory()));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename1, getArchiveDirectory()));
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename2, getArchiveDirectory()));
	}

	@Test
	public final void testMultiErrorJob() {

		final String filename1 = "CPSS-to-CCPC_successful.txt";
		final String filename2 = "CPSS-to-CCPC_PS_invalid_sequence_number.txt";
		final String filename3 = "CPSS-to-CCPC_PS_invalid_end_of_file.txt";

		final String filepath1 = "fps/" + filename1;
		final String filepath2 = "fps/" + filename2;
		final String filepath3 = "fps/" + filename3;

		// Ensure test server is in expected initial state.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename1, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename1, getArchiveDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename1, getErrorDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename2, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename2, getArchiveDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename2, getErrorDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename3, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename3, getArchiveDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename3, getErrorDirectory()));

		// FTP test files to server.
		assertTrue(ftpFileToServer(filepath1));
		assertTrue(ftpFileToServer(filepath2));
		assertTrue(ftpFileToServer(filepath3));

		// Run the job.
		final boolean runSuccess = processFPSDataJobTest.testFtpJob();

		// Verify the job was unsuccessful.
		assertFalse(runSuccess);

		// Verify 2 errors can be found in the error message.
		// Note the order of processed file is not guaranteed.
		assertTrue(processFPSDataJobTest.getErrorMessage()
				.equals("\n\nError Message:\n" + (new File(filename2)).getName() + ": "
						+ "Attempted to process PS with invalid seq num (5) but expected (1)\n"
						+ (new File(filename3)).getName() + ": " + "PS count (1) of (2) indicates incomplete sequence")
				|| processFPSDataJobTest.getErrorMessage()
						.equals("\n\nError Message:\n" + (new File(filename3)).getName() + ": "
								+ "PS count (1) of (2) indicates incomplete sequence\n"
								+ (new File(filename2)).getName() + ": "
								+ "Attempted to process PS with invalid seq num (5) but expected (1)"));

		// Verify files are not in the data directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename1, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename2, processFPSServerTargetDirectory));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename3, processFPSServerTargetDirectory));

		// Verify successful file is in the archive directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename1, getArchiveDirectory()));

		// Verify erroneous files are not in the archive directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename2, getArchiveDirectory()));
		assertFalse(sftpService.ftpFileExistsInDirectory(filename3, getArchiveDirectory()));

		// Verify successful file is not in the error directory.
		assertFalse(sftpService.ftpFileExistsInDirectory(filename1, getErrorDirectory()));

		// Verify erroneous files are in the error directory.
		assertTrue(sftpService.ftpFileExistsInDirectory(filename2, getErrorDirectory()));
		assertTrue(sftpService.ftpFileExistsInDirectory(filename3, getErrorDirectory()));

		// Cleanup.
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename1, getArchiveDirectory()));
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename2, getErrorDirectory()));
		assertTrue(sftpService.ftpRemoveFileFromDirectory(filename3, getErrorDirectory()));
	}
	
	private String getArchiveDirectory() {
		return processFPSServerTargetDirectory + "/" + processFPSServerArchiveTargetDirectory ;
	}

	private String getErrorDirectory() {
		return processFPSServerTargetDirectory + "/" + processFPSServerErrorTargetDirectory;
	}
	
	private boolean ftpFileToServer(final String filepath) {
		// Transfer data file.
		final boolean dataFileResult = sftpService.ftpFileToServer(new File(this.getClass().getClassLoader().getResource(filepath).getFile()).getAbsolutePath(), processFPSServerTargetDirectory);
		
		// Transfer don file.
		final String emptyFilename = "CCPC-to-CPSS_empty.txt";
		final String emptyFilepath = new File(
				this.getClass().getClassLoader().getResource("fps/" + emptyFilename).getFile()).getAbsolutePath();
		
		final String dataFilename = new File(this.getClass().getClassLoader().getResource(filepath).getFile()).getName();
		final String indicatorFilename = dataFilename.substring(0, dataFilename.lastIndexOf(".txt")) + ".don";
		
		final boolean indicatorFileResult = sftpService.ftpFileToServerWithName(emptyFilepath, indicatorFilename, processFPSServerTargetDirectory);
		
		return (dataFileResult && indicatorFileResult);
	}

}