package gov.va.cpss.job.appsprintack;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assume.assumeTrue;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.sql.DataSource;

import org.apache.log4j.Logger;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import java.util.Calendar;

import gov.va.cpss.dao.APSStmtDAO;
import gov.va.cpss.dao.BatchRunProcessDAO;
import gov.va.cpss.dao.CBSAccountDAO;
import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.dao.impl.APPSDAOTestWrapper;
import gov.va.cpss.model.AITCDollar;
import gov.va.cpss.model.BatchRunProcess;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.ProcessStatus.Status;
import gov.va.cpss.model.apps.APSStmt;
import gov.va.cpss.service.SftpService;

/**
 * Automated tests for UpdateAPPSPrintAckJob. 
 * 
 * @author Andrew Vance
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml", "/dao-test-context.xml",
		"/cpss-batch.xml", "/update-apps-print-ack-test-context.xml", "/cpss-email.xml" })
public class UpdateAppsPrintAckJobTest {
	
	public static final String TEST_FILE_DIR_PATH = "appsprintack/";
	
	private final Logger testLogger = Logger.getLogger(this.getClass().getCanonicalName());
	
	@Value("${run.integration.test:false}")
	private Boolean runIntegrationTest;
	
	@Autowired
	private TestUpdateAppsPrintAckJob testUpdateAppsPrintAckJob;
	
	@Autowired
	private SftpService sftpService;
	
	@Autowired
	private APSStmtDAO apsStmtDao;
	
	@Autowired
	private CBSAccountDAO cbsAccountDao;
	
	@Autowired
	private BatchRunProcessDAO batchRunProcessDao;
	
	@Autowired
	private ProcessStatusDAO processStatusDao;
	
	@Autowired
	private String updatePrintAckAPPSServerTargetDirectory;
	
	@SuppressWarnings("unused")
	@Autowired
	private String updatePrintAckAPPSServerErrorTargetDirectory;

	@SuppressWarnings("unused")
	@Autowired
	private String updatePrintAckAPPSServerArchiveTargetDirectory;
	
	private JdbcTemplate jdbcTemplate;
	
	private long testCbsAccountId1;
	
	private long testCbsAccountId2;
	
	private String testIcn1 = "TESTICNTESTICN123";
	
	private String testIcn2 = "TESTICNTESTICN456";
	
	@Autowired
	public void setDatasource(DataSource dataSource) {
		jdbcTemplate = new JdbcTemplate(dataSource);
	}
	
	@Before
	public final void beforeTest(){
		assumeTrue(runIntegrationTest);
		
		//Create test CBSAccounts
		testCbsAccountId1 = cbsAccountDao.save("TESTICNTESTICN123");
		testCbsAccountId2 = cbsAccountDao.save("TESTICNTESTICN456");
	}
	
	@After
	public final void afterTest(){
		removeApsStmt(testCbsAccountId1);
		removeApsStmt(testCbsAccountId2);
		
		//TODO: create method in cbsAccountDao to delete
		jdbcTemplate.update("delete from cbsAccount where id="+testCbsAccountId1+" or id="+testCbsAccountId2);
	}
	
	/**
	 * Simple test case with 1 PA and 1 AD.
	 */
	@Test
	public void testSimpleSuccessful() {
		final String fileName = "apps-print-ack-test-simple-successful.txt";
		
		final List<String> expectedSuccessFileNames = new ArrayList<String>();
		expectedSuccessFileNames.add(fileName);
		
		final List<String> expectedFailFileNames = null;
		
		final String fileContents = "PA^001^001^0001^2016^05052016^}AD^"+testIcn1+"^}~";
		
		writeInputFile(fileName, fileContents);
		
		(new FileTestWrapper(expectedSuccessFileNames, expectedFailFileNames) {

			@Override
			protected void test() {
				(new APPSDAOTestWrapper() {

					@Override
					protected void test() {						
						final int expectedFileCount = 1;
						final String stmtDate = "05052016";
						Date ackPrintDate = null;
						try {
							ackPrintDate = (new SimpleDateFormat("MMddyyyy")).parse("05052016");
						} catch (ParseException e) {
						}						
						final int stmtYear = 2016;
						final long cbsAccountId = testCbsAccountId1;
						
						//Build target APS Stmt
						final long apsStmtId = buildApsStmt(cbsAccountId, stmtYear, stmtDate, jdbcTemplate);	
						
						boolean success = testUpdateAppsPrintAckJob.testJob();						
						
						//Verify BatchRunProcess was created successfully
						List<Long> batchRunProcessIdList = testUpdateAppsPrintAckJob.getBatchRunProcessIdList();
						assertNotNull(batchRunProcessIdList);
						assertTrue(!batchRunProcessIdList.isEmpty());
						
						final long batchRunProcessId = batchRunProcessIdList.get(0);
						assertTrue(batchRunProcessId > 0);
						
						final BatchRunProcess batchRunProcess = batchRunProcessDao.get(batchRunProcessId);
						assertNotNull(batchRunProcess);
						
						assertTrue(success);

						assertEquals(batchRunProcess.getStatusId(), (int)processStatusDao.getStatusFromEnum(Status.SUCCESS));
						
						final int fileCount = testUpdateAppsPrintAckJob.getFileCount();						
						assertEquals(expectedFileCount, fileCount);
						
						//Verify APSStmt was updated successfully
						final APSStmt apsStmt = apsStmtDao.get(apsStmtId);
						assertNotNull(apsStmt);
						
						assertEquals(apsStmt.getAckBatchRunId(), batchRunProcessId);
						
						assertEquals(apsStmt.getPrintDate(), ackPrintDate);
						
						assertEquals(apsStmt.getStatusId(), (int)processStatusDao.getStatusFromEnum(ProcessStatus.Status.ACK));				
					}					
				}).run(jdbcTemplate);
			}
			
		}).run();
	}
	
	/**
	 * Test case with 1 PA and 2 ADs, all valid. Expect 2 APS received/updated.
	 */
	@Test
	public void test1PA2ADvalid() {
		final String filename = "apps-print-ack-test-1pa2ad-valid.txt";
		
		final List<String> expectedSuccessFileNames = new ArrayList<String>();
		expectedSuccessFileNames.add(filename);
		
		final List<String> expectedFailFileNames = null;
		
		final String fileContents = "PA^001^001^0002^2016^05052016^}AD^"+testIcn1
				+"^}AD^"+testIcn2+"^}~";
		
		writeInputFile(filename, fileContents);
		
		(new FileTestWrapper(expectedSuccessFileNames, expectedFailFileNames) {

			@Override
			protected void test() {
				(new APPSDAOTestWrapper() {
					
					@Override
					protected void test() {
						final long cbsAccountId1 = testCbsAccountId1;
						final long cbsAccountId2 = testCbsAccountId2;
						final int stmtYear = 2016;
						final String statementDate = "05052016";
						Date expectedPrintDate = null;
						try {
							expectedPrintDate = (new SimpleDateFormat("MMddyyyy")).parse(statementDate);
						} 
						catch (ParseException e) {							
						}
						
						final long apsStmtId1 = buildApsStmt(cbsAccountId1, stmtYear, statementDate, jdbcTemplate);
						final long apsStmtId2 = buildApsStmt(cbsAccountId2, stmtYear, statementDate, jdbcTemplate);
						
						final boolean runSuccess = testUpdateAppsPrintAckJob.testJob();
						
						final List<Long> batchRunProcessIdList = testUpdateAppsPrintAckJob.getBatchRunProcessIdList();
						assertNotNull(batchRunProcessIdList);
						assertEquals(1, batchRunProcessIdList.size());
						
						assertTrue(runSuccess);						
						
						assertEquals(2L, testUpdateAppsPrintAckJob.getTotalAPSReceived());
						assertEquals(2L, testUpdateAppsPrintAckJob.getTotalAPSUpdated());
						
						//Verify APSStmt 1
						final APSStmt apsStmt1 = apsStmtDao.get(apsStmtId1);
						assertNotNull(apsStmt1);
						assertEquals(apsStmt1.getAckBatchRunId(), (long)batchRunProcessIdList.get(0));
						assertEquals(apsStmt1.getPrintDate(), expectedPrintDate);
						assertEquals(apsStmt1.getStatusId(), (int)processStatusDao.getStatusFromEnum(ProcessStatus.Status.ACK));
						
						//Verify APSStmt 2
						final APSStmt apsStmt2 = apsStmtDao.get(apsStmtId2);
						assertNotNull(apsStmt2);
						assertEquals(apsStmt2.getAckBatchRunId(), (long)batchRunProcessIdList.get(0));
						assertEquals(apsStmt2.getPrintDate(), expectedPrintDate);
						assertEquals(apsStmt2.getStatusId(), (int)processStatusDao.getStatusFromEnum(ProcessStatus.Status.ACK));
					}
				}).run(jdbcTemplate);
			}
			
		}).run();
	}
	
	/**
	 * Test case with 1 PA and 1 invalid AD (doesn't match with any CBSAccount). Expect failure. Note this test
	 * does not need to be dynamically written because the AD is supposed to use an invalid account number.
	 */
	@Test
	public void testInvalidAD() {
		final String filename = "apps-print-ack-test-invalid-ad.txt";
		
		final List<String> expectedSuccessFileNames = null;
		
		final List<String> expectedFailFileNames = new ArrayList<String>();
		expectedFailFileNames.add(filename);
		
		(new FileTestWrapper(expectedSuccessFileNames, expectedFailFileNames) {

			@Override
			protected void test() {
				(new APPSDAOTestWrapper() {
					
					@Override
					protected void test() {
						final boolean runSuccess = testUpdateAppsPrintAckJob.testJob();
						
						final List<Long> batchRunProcessIdList = testUpdateAppsPrintAckJob.getBatchRunProcessIdList();
						assertNotNull(batchRunProcessIdList);
						assertEquals(0, batchRunProcessIdList.size());
						
						assertFalse(runSuccess);
						
						assertNotNull(testUpdateAppsPrintAckJob.getErrorMessage());
						
						assertEquals(1, testUpdateAppsPrintAckJob.getFailedFileList().size());
						
						assertEquals(0L, testUpdateAppsPrintAckJob.getTotalAPSReceived());
						assertEquals(0L, testUpdateAppsPrintAckJob.getTotalAPSUpdated());
					}
				}).run(jdbcTemplate);
			}			
		}).run();
	}
	
	/**
	 * Test case with 1 PA and 0 AD. Expect failure. This test file does not need to be dynamically generated because
	 * there is no account number associated with it at all.
	 */
	@Test
	public void testMissingAD() {
		final String filename = "apps-print-ack-test-missing-ad.txt";
		
		final List<String> expectedSuccessFileNames = null;
		
		final List<String> expectedFailFileNames = new ArrayList<String>();
		expectedFailFileNames.add(filename);
		
		(new FileTestWrapper(expectedSuccessFileNames, expectedFailFileNames) {

			@Override
			protected void test() {
				(new APPSDAOTestWrapper() {
					
					@Override
					protected void test() {
						final boolean runSuccess = testUpdateAppsPrintAckJob.testJob();
						
						final List<Long> batchRunProcessIdList = testUpdateAppsPrintAckJob.getBatchRunProcessIdList();
						assertNotNull(batchRunProcessIdList);
						assertEquals(0, batchRunProcessIdList.size());
						
						assertFalse(runSuccess);
						
						assertNotNull(testUpdateAppsPrintAckJob.getErrorMessage());
						
						assertEquals(1, testUpdateAppsPrintAckJob.getFailedFileList().size());
						
						assertEquals(0L, testUpdateAppsPrintAckJob.getTotalAPSReceived());
						assertEquals(0L, testUpdateAppsPrintAckJob.getTotalAPSUpdated());
					}
				}).run(jdbcTemplate);
			}			
		}).run();
	}
	
	/**
	 * Test case with 1 PA and 2 AD but the PA only specifies 1 AD. Expect failure. This test file does not need to be dynamically
	 * generated because the test will never actually access an account.
	 */
	@Test
	public void testExtraAD() {
		final String filename = "apps-print-ack-test-extra-ad.txt";
		
		final List<String> expectedSuccessFileNames = null;
		
		final List<String> expectedFailFileNames = new ArrayList<String>();
		expectedFailFileNames.add(filename);
		
		(new FileTestWrapper(expectedSuccessFileNames, expectedFailFileNames) {

			@Override
			protected void test() {
				(new APPSDAOTestWrapper() {
					
					@Override
					protected void test() {
						final boolean runSuccess = testUpdateAppsPrintAckJob.testJob();
						
						final List<Long> batchRunProcessIdList = testUpdateAppsPrintAckJob.getBatchRunProcessIdList();
						assertNotNull(batchRunProcessIdList);
						assertEquals(0, batchRunProcessIdList.size());
						
						assertFalse(runSuccess);
						
						assertNotNull(testUpdateAppsPrintAckJob.getErrorMessage());
						
						assertEquals(1, testUpdateAppsPrintAckJob.getFailedFileList().size());
						
						assertEquals(0L, testUpdateAppsPrintAckJob.getTotalAPSReceived());
						assertEquals(0L, testUpdateAppsPrintAckJob.getTotalAPSUpdated());
					}
				}).run(jdbcTemplate);
			}			
		}).run();
	}
	
	/**
	 * Test with 2 PAs each with 1 AD. Expect 2 APS received/updated.
	 */
	@Test
	public void testMultipleRecords() {
		final String filename = "apps-print-ack-test-multiple-records.txt";
		
		final List<String> expectedSuccessFileNames = new ArrayList<String>();
		expectedSuccessFileNames.add(filename);
		
		final List<String> expectedFailFileNames = null;
		
		final String fileContents = "PA^001^^0001^2016^05052016^}AD^"+testIcn1+"^}~\n" +
									"PA^002^002^0001^2016^05172016^}AD^"+testIcn2+"^}~";
		
		writeInputFile(filename, fileContents);
		
		(new FileTestWrapper(expectedSuccessFileNames, expectedFailFileNames) {

			@Override
			protected void test() {
				(new APPSDAOTestWrapper() {
					
					@Override
					protected void test() {
						final long cbsAccountId1 = testCbsAccountId1;
						final long cbsAccountId2 = testCbsAccountId2;
						final int stmtYear = 2016;
						final String statementDate1 = "05052016";
						final String statementDate2 = "05172016";
						Date expectedPrintDate1 = null;
						Date expectedPrintDate2 = null;
						try {
							expectedPrintDate1 = (new SimpleDateFormat("MMddyyyy")).parse(statementDate1);
							expectedPrintDate2 = (new SimpleDateFormat("MMddyyyy")).parse(statementDate2);
						} 
						catch (ParseException e) {							
						}
						
						final long apsStmtId1 = buildApsStmt(cbsAccountId1, stmtYear, statementDate1, jdbcTemplate);
						final long apsStmtId2 = buildApsStmt(cbsAccountId2, stmtYear, statementDate2, jdbcTemplate);
						
						final boolean runSuccess = testUpdateAppsPrintAckJob.testJob();
						
						final List<Long> batchRunProcessIdList = testUpdateAppsPrintAckJob.getBatchRunProcessIdList();
						assertNotNull(batchRunProcessIdList);
						assertEquals(2, batchRunProcessIdList.size());
						
						assertTrue(runSuccess);
						
						//Verify first APSStmt
						APSStmt apsStmt1 = apsStmtDao.get(apsStmtId1);
						assertNotNull(apsStmt1);
						assertEquals(apsStmt1.getStatusId(), (int)processStatusDao.getStatusFromEnum(ProcessStatus.Status.ACK));
						assertEquals(apsStmt1.getAckBatchRunId(), (long)batchRunProcessIdList.get(0));
						assertEquals(apsStmt1.getPrintDate(), expectedPrintDate1);
						
						//Verify second APSStmt		
						APSStmt apsStmt2 = apsStmtDao.get(apsStmtId2);
						assertNotNull(apsStmt2);
						assertEquals(apsStmt2.getStatusId(), (int)processStatusDao.getStatusFromEnum(ProcessStatus.Status.ACK));
						assertEquals(apsStmt2.getAckBatchRunId(), (long)batchRunProcessIdList.get(1));
						assertEquals(apsStmt2.getPrintDate(), expectedPrintDate2);
					}
				}).run(jdbcTemplate);
			}
			
		}).run();
	}
	
	private void writeInputFile(final String filename, final String contents) {
		
		final File output = new File(this.getClass().getClassLoader().getResource(TEST_FILE_DIR_PATH + filename).getFile());
		
		PrintWriter writer;
		try {
			writer = new PrintWriter(output);
		} catch (FileNotFoundException e) {
			testLogger.debug(e);
			return;
		}
		
		writer.println(contents);
		
		writer.close();
	}
	
	private boolean ftpFileToServer(final String filepath) {
		// Transfer data file.
		final boolean dataFileResult = sftpService.ftpFileToServer(
				new File(this.getClass().getClassLoader().getResource(filepath).getFile()).getAbsolutePath(),
				updatePrintAckAPPSServerTargetDirectory);

		// Transfer don file.
		final String emptyFilename = "CCPC-to-CPSS_empty.txt";
		final String emptyFilepath = new File(
				this.getClass().getClassLoader().getResource("fps/" + emptyFilename).getFile()).getAbsolutePath();

		final String dataFilename = new File(this.getClass().getClassLoader().getResource(filepath).getFile())
				.getName();
		final String indicatorFilename = dataFilename.substring(0, dataFilename.lastIndexOf(".txt")) + ".don";

		final boolean indicatorFileResult = sftpService.ftpFileToServerWithName(emptyFilepath, indicatorFilename,
				updatePrintAckAPPSServerTargetDirectory);

		return (dataFileResult && indicatorFileResult);
	}
	
	/**
	 * Inserts a test APSStmt and APSSiteStmt into the database.
	 * @param cbsAccountId the CBSS account number used to create statements
	 * @param stmtYear
	 * @param statementDate
	 * @param jdbcTemplate
	 * @return apsStmtId of the inserted APSStmt
	 */
	private synchronized long buildApsStmt(long cbsAccountId, int stmtYear, String statementDate, JdbcTemplate jdbcTemplate) {
		// TODO: For demo purposes, database entries are manually inserted. It would be preferable to make these
		// queries through the DAO layer but these aren't implemented yet. -Andrew Vance 1-18-2017
		
		//Create APS Stmt
		APSStmt apsStmt = new APSStmt(cbsAccountId);
		//TODO: Pending decision on APSStmt stmtYear field. -Andrew Vance 2-28-2017
		//apsStmt.setStmtYear(stmtYear);
		apsStmt.setTotalAmountReceived(new AITCDollar(270));
		apsStmt.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.SENT));
		Date ackPrintDate = null;
		try {
			ackPrintDate = (new SimpleDateFormat("MMddyyyy")).parse(statementDate);
		} catch (ParseException e) {
		}
		apsStmt.setStatementDate(ackPrintDate);
		apsStmt.setProcessDate(Calendar.getInstance().getTime());		
		apsStmt.setId(apsStmtDao.save(apsStmt));

		//Create APS Site Stmt
		final long apsStmtId = apsStmt.getId();
		jdbcTemplate.update("insert into apssitestmt (stmtid, stationnum, stmtyear, processdate, totalamntrcv, araddressflag, lastbillprepdate)"
				+ "values ('"+ apsStmtId + "', '123', '" + stmtYear + "', CURRENT_TIMESTAMP, 100, 'N', CURRENT_TIMESTAMP)");	
		
		return apsStmtId;
	}
	
	/**
	 * Deletes test APSStmt and APSSiteStmt from database.
	 * @param cbsAccountId id of the CBS account associated with the APSStmt to delete
	 */
	private void removeApsStmt(final long cbsAccountId) {
		//TODO: Same with prepareDatabase(), these queries should be handled through DAO layer -Andrew Vance 1/18/2017
		
		jdbcTemplate.update("delete from apssitestmt where stmtid=(select id from apsstmt where accountId="+cbsAccountId+")");
		
		jdbcTemplate.update("delete from apsstmt where accountId="+cbsAccountId);
	}
	
	private abstract class FileTestWrapper {
		
		private List<String> expectSuccessFileNames;
		private List<String> expectFailFileNames;
		
		protected FileTestWrapper(final List<String> expectSuccessFileNames, final List<String> expectFailFileNames) {
			this.expectSuccessFileNames = expectSuccessFileNames;
			this.expectFailFileNames = expectFailFileNames;
		}
		
		private void cleanup() {
			final List<List<String>> fileNameLists = new ArrayList<List<String>>(2);
			fileNameLists.add(expectSuccessFileNames);
			fileNameLists.add(expectFailFileNames);

			for (List<String> fileNameL : fileNameLists) {
				if (fileNameL == null)
					continue;

				for (String fileName : fileNameL) {
					sftpService.ftpRemoveFileFromDirectory(fileName, testUpdateAppsPrintAckJob.getDataDirectory());
					sftpService.ftpRemoveFileFromDirectory(fileName, testUpdateAppsPrintAckJob.getArchiveDirectory());
					sftpService.ftpRemoveFileFromDirectory(fileName, testUpdateAppsPrintAckJob.getErrorDirectory());
				}
			}
		}
		
		private void checkSetup() {
			final List<List<String>> fileNameLists = new ArrayList<List<String>>(2);
			fileNameLists.add(expectSuccessFileNames);
			fileNameLists.add(expectFailFileNames);
			
			for (List<String> fileNameL : fileNameLists) {
				if(fileNameL == null)
					continue;
				
				for(String fileName : fileNameL) {
					assertFalse(sftpService.ftpFileExistsInDirectory(fileName, testUpdateAppsPrintAckJob.getDataDirectory(),
							true));
					assertFalse(sftpService.ftpFileExistsInDirectory(fileName, testUpdateAppsPrintAckJob.getArchiveDirectory(), true));
					assertFalse(sftpService.ftpFileExistsInDirectory(fileName, testUpdateAppsPrintAckJob.getErrorDirectory(), true));
				}
			}
		}
		
		private void stageFiles() {			
			final List<List<String>> fileNameLists = new ArrayList<List<String>>(2);
			fileNameLists.add(expectSuccessFileNames);
			fileNameLists.add(expectFailFileNames);

			for (List<String> fileNameL : fileNameLists) {
				if (fileNameL == null) {
					continue;
				}					

				for (String fileName : fileNameL) {
					assertTrue(ftpFileToServer(TEST_FILE_DIR_PATH + fileName));
				}
			}
		}
		
		private void checkDirectories() {
			if (expectSuccessFileNames != null) {
				for (String fileName : expectSuccessFileNames) {
					// Verify file is not in the data directory.
					assertFalse(sftpService.ftpFileExistsInDirectory(fileName, testUpdateAppsPrintAckJob.getDataDirectory(),
							true));
					// Verify file is in the archive directory.
					assertTrue(sftpService.ftpFileExistsInDirectory(fileName,
							testUpdateAppsPrintAckJob.getArchiveDirectory(), true));
					// Verify file is not in the error directory.
					assertFalse(sftpService.ftpFileExistsInDirectory(fileName,
							testUpdateAppsPrintAckJob.getErrorDirectory(), true));
				}
			}
			if (expectFailFileNames != null) {
				for (String fileName : expectFailFileNames) {
					// Verify file is not in the data directory.
					assertFalse(sftpService.ftpFileExistsInDirectory(fileName, testUpdateAppsPrintAckJob.getDataDirectory(),
							true));
					// Verify file is not in the archive directory.
					assertFalse(sftpService.ftpFileExistsInDirectory(fileName,
							testUpdateAppsPrintAckJob.getArchiveDirectory(), true));
					// Verify file is in the error directory.
					assertTrue(sftpService.ftpFileExistsInDirectory(fileName, testUpdateAppsPrintAckJob.getErrorDirectory(),
							true));
				}
			}
		}
		
		protected void run() {
			checkSetup();	
			
			try {
				stageFiles();
				test();
				checkDirectories();
			} finally {
				cleanup();
			}
		}
		
		protected abstract void test();
	}
}
