package gov.va.cpss.job;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assume.assumeTrue;

import java.io.File;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.sql.DataSource;

import org.apache.log4j.Logger;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.dao.CBSStmtDAO;
import gov.va.cpss.dao.PrintAckRecDAO;
import gov.va.cpss.dao.impl.CBSDAOImplUtils;
import gov.va.cpss.dao.impl.CBSDAOTestWrapper;
import gov.va.cpss.model.cbs.CBSStmt;
import gov.va.cpss.model.printack.ADFileRecord;
import gov.va.cpss.model.printack.PrintAckRec;
import gov.va.cpss.service.SftpService;

/**
 * Unit tests for UpdatePrintAckJob.
 * 
 * @author Brad Pickle
 *
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml", "/dao-test-context.xml",
		"/cpss-batch.xml", "/update-print-ack-test-context.xml", "/cpss-email.xml" })
public class UpdatePrintAckJobTest {

	public static final String TEST_FILE_DIR_PATH = "printack/";

	private final Logger testLogger = Logger.getLogger(this.getClass().getCanonicalName());

	@Value("${run.integration.test:false}")
	private Boolean runIntegrationTest;

	@Autowired
	private TestUpdatePrintAckJob testUpdatePrintAckJob;

	@Autowired
	private SftpService sftpService;

	@Autowired
	private CBSDAOImplUtils cbsDAOImplUtils;

	@Autowired
	private CBSStmtDAO cbsStmtDao;

	@Autowired
	private PrintAckRecDAO printAckRecDao;

	@Autowired
	private String updatePrintAckServerTargetDirectory;

	@SuppressWarnings("unused")
	@Autowired
	private String updatePrintAckServerErrorTargetDirectory;

	@SuppressWarnings("unused")
	@Autowired
	private String updatePrintAckServerArchiveTargetDirectory;

	private JdbcTemplate jdbcTemplate;

	@Autowired
	public void setDataSource(DataSource dataSource) {
		jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/**
	 * Only run these tests if property is set to run integration test.
	 */
	@Before
	public final void beforeTest() {
		assumeTrue(runIntegrationTest);
	}

	/**
	 * Test against printack-test-1.txt. 1 valid PA with 5 missing AD. 0 CBS
	 * received/updated.
	 */
	@Test
	public void test() {
		final String fileName = "printack-test-1.txt";
		final List<String> expectedSuccessFileNames = null;

		final List<String> expectedFailFileNames = new ArrayList<String>(1);
		expectedFailFileNames.add(fileName);

		(new FileTestWrapper(expectedSuccessFileNames, expectedFailFileNames) {

			@Override
			protected void test() {
				(new CBSDAOTestWrapper() {

					@Override
					protected void test() {
						final boolean runSuccess = testUpdatePrintAckJob.testJob();
						List<Long> printAckRecIdList = testUpdatePrintAckJob.getPrintAckRecIdList();
						assertNotNull(printAckRecIdList);
						getPrintAckRecIdS().addAll(printAckRecIdList);
						assertEquals(1, printAckRecIdList.size());
						final long printAckRecId = printAckRecIdList.get(0);
						assertTrue(printAckRecId > 0);

						// Verify the job was successful.
						assertFalse(runSuccess);

						final PrintAckRec printAckRec = printAckRecDao.get(printAckRecId);
						assertNotNull(printAckRec);

						final String stationNum = "442";

						assertEquals(testUpdatePrintAckJob.getBatchRunId(), printAckRec.getBatchRunId());
						assertTrue(printAckRec.getDateReceived().getTime() > 0);
						assertEquals(fileName, printAckRec.getFileName());
						assertEquals(stationNum, printAckRec.getStationNum());
						assertEquals(CBSDAOImplUtils.STATUS_ERROR_ID, printAckRec.getStatusId());

						// Check exit status
						assertNotNull(testUpdatePrintAckJob.getErrorMessage());

						assertEquals(0L, testUpdatePrintAckJob.getTotalCBSReceived());
						assertEquals(0L, testUpdatePrintAckJob.getTotalCBSUpdated());
					}

				}).run(jdbcTemplate);
			}

		}).run();
	}

	/**
	 * Test against printack-test-3.txt. 1 valid PA with 2 AD. 2 CBS
	 * received/updated
	 */
	@Test
	public void testFile3() {
		final String fileName = "printack-test-3.txt";
		final List<String> expectedSuccessFileNames = new ArrayList<String>(1);
		expectedSuccessFileNames.add(fileName);

		final List<String> expectedFailFileNames = null;

		(new FileTestWrapper(expectedSuccessFileNames, expectedFailFileNames) {

			@Override
			protected void test() {
				(new CBSDAOTestWrapper() {

					@Override
					protected void test() {
						final List<CBSStmt> cbsStmtL = createCBSSet1(this);

						final boolean runSuccess = testUpdatePrintAckJob.testJob();
						List<Long> printAckRecIdList = testUpdatePrintAckJob.getPrintAckRecIdList();
						assertNotNull(printAckRecIdList);
						getPrintAckRecIdS().addAll(printAckRecIdList);
						assertEquals(1, printAckRecIdList.size());
						final long printAckRecId = printAckRecIdList.get(0);
						assertTrue(printAckRecId > 0);

						// Verify the job was successful.
						assertTrue(runSuccess);

						Date ackPrintDate = null;
						try {
							ackPrintDate = (new SimpleDateFormat("MMddyyyy")).parse("05312016");
						} catch (ParseException e) {
						}

						final CBSStmt cbsStmt1 = cbsStmtDao.get(cbsStmtL.get(0).getId());
						final CBSStmt cbsStmt2 = cbsStmtDao.get(cbsStmtL.get(1).getId());

						assertNotNull(cbsStmt1);
						assertNotNull(cbsStmt2);

						assertEquals(ackPrintDate, cbsStmt1.getPrintDate());
						assertEquals(ackPrintDate, cbsStmt2.getPrintDate());

						assertEquals(CBSDAOImplUtils.STATUS_ACK_ID, cbsStmt1.getStatusId());
						assertEquals(CBSDAOImplUtils.STATUS_ACK_ID, cbsStmt2.getStatusId());

						assertEquals(printAckRecId, cbsStmt1.getPrintAckId());
						assertEquals(printAckRecId, cbsStmt2.getPrintAckId());

						final String stationNum = "442";

						final PrintAckRec printAckRec = printAckRecDao.get(printAckRecId);
						assertNotNull(printAckRec);

						assertEquals(testUpdatePrintAckJob.getBatchRunId(), printAckRec.getBatchRunId());
						assertTrue(printAckRec.getDateReceived().getTime() > 0);
						assertEquals(fileName, printAckRec.getFileName());
						assertEquals(stationNum, printAckRec.getStationNum());
						assertEquals(CBSDAOImplUtils.STATUS_SUCCESS_ID, printAckRec.getStatusId());

						// Check exit status
						assertNull(testUpdatePrintAckJob.getErrorMessage());

						assertEquals(2L, testUpdatePrintAckJob.getTotalCBSReceived());
						assertEquals(2L, testUpdatePrintAckJob.getTotalCBSUpdated());
					}

				}).run(jdbcTemplate);
			}

		}).run();
	}

	/**
	 * Test against printack-test-valid-invalid-records.txt. 1 valid PA with 2
	 * AD. 2 CBS received/updated, but expected to be rolled back. Followed by
	 * invalid PA.
	 */
	@Test
	public void testValidInvalid() {
		final String fileName = "printack-test-valid-invalid-records.txt";
		final List<String> expectedSuccessFileNames = null;

		final List<String> expectedFailFileNames = new ArrayList<String>(1);
		expectedFailFileNames.add(fileName);

		(new FileTestWrapper(expectedSuccessFileNames, expectedFailFileNames) {

			@Override
			protected void test() {
				(new CBSDAOTestWrapper() {

					@Override
					protected void test() {
						final List<CBSStmt> cbsStmtL = createCBSSet1(this);

						final boolean runSuccess = testUpdatePrintAckJob.testJob();
						List<Long> printAckRecIdList = testUpdatePrintAckJob.getPrintAckRecIdList();
						assertNotNull(printAckRecIdList);
						getPrintAckRecIdS().addAll(printAckRecIdList);
						assertEquals(1, printAckRecIdList.size());
						final long printAckRecId = printAckRecIdList.get(0);
						assertTrue(printAckRecId > 0);

						assertFalse(runSuccess);

						final CBSStmt cbsStmt1 = cbsStmtDao.get(cbsStmtL.get(0).getId());
						final CBSStmt cbsStmt2 = cbsStmtDao.get(cbsStmtL.get(1).getId());

						assertNotNull(cbsStmt1);
						assertNotNull(cbsStmt2);

						assertEquals(null, cbsStmt1.getPrintDate());
						assertEquals(null, cbsStmt2.getPrintDate());

						assertEquals(CBSDAOImplUtils.STATUS_SENT_ID, cbsStmt1.getStatusId());
						assertEquals(CBSDAOImplUtils.STATUS_SENT_ID, cbsStmt2.getStatusId());

						assertEquals(0, cbsStmt1.getPrintAckId());
						assertEquals(0, cbsStmt2.getPrintAckId());

						final String stationNum = "442";

						final PrintAckRec printAckRec = printAckRecDao.get(printAckRecId);
						assertNotNull(printAckRec);

						assertEquals(testUpdatePrintAckJob.getBatchRunId(), printAckRec.getBatchRunId());
						assertTrue(printAckRec.getDateReceived().getTime() > 0);
						assertEquals(fileName, printAckRec.getFileName());
						assertEquals(stationNum, printAckRec.getStationNum());
						assertEquals(CBSDAOImplUtils.STATUS_ERROR_ID, printAckRec.getStatusId());

						// Check exit status
						assertNotNull(testUpdatePrintAckJob.getErrorMessage());

						assertEquals(0L, testUpdatePrintAckJob.getTotalCBSReceived());
						assertEquals(0L, testUpdatePrintAckJob.getTotalCBSUpdated());
					}

				}).run(jdbcTemplate);
			}

		}).run();
	}

	/**
	 * Test against printack-test-missing-AD.txt.
	 */
	@Test
	public void testMissingAD() {
		final List<String> expectedSuccessFileNames = null;

		final List<String> expectedFailFileNames = new ArrayList<String>(1);
		expectedFailFileNames.add("printack-test-missing-AD.txt");

		(new FileTestWrapper(expectedSuccessFileNames, expectedFailFileNames) {

			@Override
			protected void test() {
				final boolean runSuccess = testUpdatePrintAckJob.testJob();
				List<Long> printAckRecIdList = testUpdatePrintAckJob.getPrintAckRecIdList();
				assertNotNull(printAckRecIdList);
				assertEquals(0, printAckRecIdList.size());

				assertFalse(runSuccess);

				// Check exit status
				assertNotNull(testUpdatePrintAckJob.getErrorMessage());

				assertEquals(0L, testUpdatePrintAckJob.getTotalCBSReceived());
				assertEquals(0L, testUpdatePrintAckJob.getTotalCBSUpdated());
			}

		}).run();
	}

	/**
	 * Test against printack-test-extra-AD.txt.
	 */
	@Test
	public void testExtraAD() {
		final List<String> expectedSuccessFileNames = null;

		final List<String> expectedFailFileNames = new ArrayList<String>(1);
		expectedFailFileNames.add("printack-test-extra-AD.txt");

		(new FileTestWrapper(expectedSuccessFileNames, expectedFailFileNames) {

			@Override
			protected void test() {
				final boolean runSuccess = testUpdatePrintAckJob.testJob();
				List<Long> printAckRecIdList = testUpdatePrintAckJob.getPrintAckRecIdList();
				assertNotNull(printAckRecIdList);
				assertEquals(0, printAckRecIdList.size());

				assertFalse(runSuccess);

				// Check exit status
				assertNotNull(testUpdatePrintAckJob.getErrorMessage());

				assertEquals(0L, testUpdatePrintAckJob.getTotalCBSReceived());
				assertEquals(0L, testUpdatePrintAckJob.getTotalCBSUpdated());
			}

		}).run();
	}

	/**
	 * Test against printack-test-multi-stations-lines.txt. 4 valid PA with 2 AD
	 * or 1 AD each. 6 CBS received/updated
	 */
	@Test
	public void testMultiStationsLines() {
		final String fileName = "printack-test-multi-stations-lines.txt";
		final List<String> expectedSuccessFileNames = new ArrayList<String>(1);
		expectedSuccessFileNames.add(fileName);

		final List<String> expectedFailFileNames = null;

		(new FileTestWrapper(expectedSuccessFileNames, expectedFailFileNames) {

			@Override
			protected void test() {
				(new CBSDAOTestWrapper() {

					@Override
					protected void test() {
						final List<CBSStmt> cbsStmtL = createCBSSet2(this);
						testLogger.debug(CBSDAOImplUtils.cbsStmtListToString(cbsStmtL, true));

						final boolean runSuccess = testUpdatePrintAckJob.testJob();
						List<Long> printAckRecIdList = testUpdatePrintAckJob.getPrintAckRecIdList();
						assertNotNull(printAckRecIdList);
						getPrintAckRecIdS().addAll(printAckRecIdList);
						assertEquals(4, printAckRecIdList.size());
						final long printAckRecId1 = printAckRecIdList.get(0);
						final long printAckRecId2 = printAckRecIdList.get(1);
						final long printAckRecId3 = printAckRecIdList.get(2);
						final long printAckRecId4 = printAckRecIdList.get(3);

						assertTrue(printAckRecId1 > 0);
						assertTrue(printAckRecId2 > 0);
						assertTrue(printAckRecId3 > 0);
						assertTrue(printAckRecId4 > 0);

						assertTrue(runSuccess);

						Date ackPrintDate1 = null;
						try {
							ackPrintDate1 = (new SimpleDateFormat("MMddyyyy")).parse("05312016");
						} catch (ParseException e) {
						}

						final CBSStmt cbsStmt1 = cbsStmtDao.get(cbsStmtL.get(0).getId());
						final CBSStmt cbsStmt2 = cbsStmtDao.get(cbsStmtL.get(1).getId());
						final CBSStmt cbsStmt3 = cbsStmtDao.get(cbsStmtL.get(2).getId());
						final CBSStmt cbsStmt4 = cbsStmtDao.get(cbsStmtL.get(3).getId());

						assertNotNull(cbsStmt1);
						assertNotNull(cbsStmt2);
						assertNotNull(cbsStmt3);
						assertNotNull(cbsStmt4);

						assertEquals(ackPrintDate1, cbsStmt1.getPrintDate());
						assertEquals(ackPrintDate1, cbsStmt2.getPrintDate());
						assertEquals(ackPrintDate1, cbsStmt3.getPrintDate());
						assertEquals(ackPrintDate1, cbsStmt4.getPrintDate());

						assertEquals(CBSDAOImplUtils.STATUS_ACK_ID, cbsStmt1.getStatusId());
						assertEquals(CBSDAOImplUtils.STATUS_ACK_ID, cbsStmt2.getStatusId());
						assertEquals(CBSDAOImplUtils.STATUS_ACK_ID, cbsStmt3.getStatusId());
						assertEquals(CBSDAOImplUtils.STATUS_ACK_ID, cbsStmt4.getStatusId());

						assertEquals(printAckRecId1, cbsStmt1.getPrintAckId());
						assertEquals(printAckRecId3, cbsStmt2.getPrintAckId());
						assertEquals(printAckRecId2, cbsStmt3.getPrintAckId());
						assertEquals(printAckRecId4, cbsStmt4.getPrintAckId());

						final String stationNum1 = "442";
						final String stationNum2 = "555";

						final PrintAckRec printAckRec1 = printAckRecDao.get(printAckRecId1);
						assertNotNull(printAckRec1);

						assertEquals(testUpdatePrintAckJob.getBatchRunId(), printAckRec1.getBatchRunId());
						assertTrue(printAckRec1.getDateReceived().getTime() > 0);
						assertEquals(fileName, printAckRec1.getFileName());
						assertEquals(stationNum1, printAckRec1.getStationNum());
						assertEquals(CBSDAOImplUtils.STATUS_SUCCESS_ID, printAckRec1.getStatusId());

						final PrintAckRec printAckRec2 = printAckRecDao.get(printAckRecId2);
						assertNotNull(printAckRec2);

						assertEquals(testUpdatePrintAckJob.getBatchRunId(), printAckRec2.getBatchRunId());
						assertTrue(printAckRec2.getDateReceived().getTime() > 0);
						assertEquals(fileName, printAckRec2.getFileName());
						assertEquals(stationNum1, printAckRec2.getStationNum());
						assertEquals(CBSDAOImplUtils.STATUS_SUCCESS_ID, printAckRec2.getStatusId());

						final PrintAckRec printAckRec3 = printAckRecDao.get(printAckRecId3);
						assertNotNull(printAckRec3);

						assertEquals(testUpdatePrintAckJob.getBatchRunId(), printAckRec3.getBatchRunId());
						assertTrue(printAckRec3.getDateReceived().getTime() > 0);
						assertEquals(fileName, printAckRec3.getFileName());
						assertEquals(stationNum2, printAckRec3.getStationNum());
						assertEquals(CBSDAOImplUtils.STATUS_SUCCESS_ID, printAckRec3.getStatusId());

						final PrintAckRec printAckRec4 = printAckRecDao.get(printAckRecId4);
						assertNotNull(printAckRec4);

						assertEquals(testUpdatePrintAckJob.getBatchRunId(), printAckRec4.getBatchRunId());
						assertTrue(printAckRec4.getDateReceived().getTime() > 0);
						assertEquals(fileName, printAckRec4.getFileName());
						assertEquals(stationNum2, printAckRec4.getStationNum());
						assertEquals(CBSDAOImplUtils.STATUS_SUCCESS_ID, printAckRec4.getStatusId());

						assertEquals(printAckRec1.getDateReceived(), printAckRec2.getDateReceived());
						assertEquals(printAckRec1.getDateReceived(), printAckRec3.getDateReceived());
						assertEquals(printAckRec1.getDateReceived(), printAckRec4.getDateReceived());

						// Check exit status
						assertNull(testUpdatePrintAckJob.getErrorMessage());

						assertEquals(6L, testUpdatePrintAckJob.getTotalCBSReceived());
						assertEquals(6L, testUpdatePrintAckJob.getTotalCBSUpdated());
					}

				}).run(jdbcTemplate);
			}

		}).run();
	}

	/**
	 * Test against ACK_07142016.txt. AD records are on new lines.
	 */
	@Test
	public void testMultiLineMessages() {
		final String fileName = "ACK_07142016.txt";
		final List<String> expectedSuccessFileNames = null;

		final List<String> expectedFailFileNames = new ArrayList<String>(1);
		expectedFailFileNames.add(fileName);

		(new FileTestWrapper(expectedSuccessFileNames, expectedFailFileNames) {

			@Override
			protected void test() {
				(new CBSDAOTestWrapper() {

					@Override
					protected void test() {
						final boolean runSuccess = testUpdatePrintAckJob.testJob();
						List<Long> printAckRecIdList = testUpdatePrintAckJob.getPrintAckRecIdList();
						assertNotNull(printAckRecIdList);
						getPrintAckRecIdS().addAll(printAckRecIdList);
						assertEquals(1, printAckRecIdList.size());
						final long printAckRecId = printAckRecIdList.get(0);
						assertTrue(printAckRecId > 0);

						assertFalse(runSuccess);

						List<ADFileRecord> missingPatientAcctList = testUpdatePrintAckJob.getMissingPatientAcctList();
						assertTrue(ack07142016MissingPatientsMatch(missingPatientAcctList));

					}

				}).run(jdbcTemplate);
			}

		}).run();
	}

	private boolean ack07142016MissingPatientsMatch(List<ADFileRecord> missingPatientAcctList) {

		final String[] expectedMissingAr = { "442068357209APADA", "442131639772ARAIZ", "442153271743ALM",
				"442168317532ANGEL", "442189397286ASNIC", "442198331238ARICA", "442217558120BARCO", "442226459057ATKER",
				"442261225442BARBU", "442269351395BARBA", "442270374253ASCHB", "442278614565BARGE", "442280212324BARDI",
				"442321352119AVELA", "442396393480BASTE", "442453578767BUESI", "442467490614BUECK", "442476374589ARMAS",
				"442479935181BAB", "442480272976ARIZO", "442481577667ANKRO", "442484279654BANGS", "442501193457BARAS",
				"442503491929BANGS", "442503593624AHLO", "442504315226ATCHE", "442505592599BADIE", "442506635706BARCE",
				"442507938204BUSHI", "442520106045BUTTE", "442520199891AMBRO", "442520297666BARRA", "442520298035BULLS",
				"442520355815AROST", "442520374375ACEVE", "442520391623ANTRO", "442520436256ARON", "442520436711BAGNE",
				"442520451555AMAYA", "442520479350ANTIS", "442520515820ACKIE", "442520611067ARAIZ", "442520611474ARMFI",
				"442520674963ABRAM", "442520738219BALD", "442520738282BUTKO", "442520875494BARBA", "442520916283BALLA",
				"442520997995ANGTO", "442521439258BARCK", "442521455208ALTY", "442521659788ALVIS", "442522418385BANGL",
				"442522490454ARMON", "442522519756BATEM", "442522620516ANDER", "442522718577ALVIS", "442522750724ALWAR",
				"442523410848BUDNE", "442523412616ANTRO", "442523455108ANGTO", "442523518726ALMEN", "442524399699ALARI",
				"442524439493BAUER", "442524450655BANGS", "442524557299BATUN", "442524575914AZER", "442524698941AGGAR",
				"442542712151ARCEM", "442552435713BARAB" };
		
		if ((missingPatientAcctList == null) || (missingPatientAcctList.size() != expectedMissingAr.length)) return false;
		
		for (int i = 0; i < expectedMissingAr.length; i++) {
			final ADFileRecord missingPatientAcct = missingPatientAcctList.get(i);
			
			if ((missingPatientAcct == null) || (! expectedMissingAr[i].equals(missingPatientAcct.getPatientAccount()))) return false;
			
		}
		
		return true;
	}

	private boolean ftpFileToServer(final String filepath) {
		// Transfer data file.
		final boolean dataFileResult = sftpService.ftpFileToServer(
				new File(this.getClass().getClassLoader().getResource(filepath).getFile()).getAbsolutePath(),
				updatePrintAckServerTargetDirectory);

		// Transfer don file.
		final String emptyFilename = "CCPC-to-CPSS_empty.txt";
		final String emptyFilepath = new File(
				this.getClass().getClassLoader().getResource("fps/" + emptyFilename).getFile()).getAbsolutePath();

		final String dataFilename = new File(this.getClass().getClassLoader().getResource(filepath).getFile())
				.getName();
		final String indicatorFilename = dataFilename.substring(0, dataFilename.lastIndexOf(".txt")) + ".don";

		final boolean indicatorFileResult = sftpService.ftpFileToServerWithName(emptyFilepath, indicatorFilename,
				updatePrintAckServerTargetDirectory);

		return (dataFileResult && indicatorFileResult);
	}

	private List<CBSStmt> createCBSSet1(CBSDAOTestWrapper wrapper) {
		final int stmt1Key = 1;
		final int stmt2Key = 2;

		final long stationNum1 = 442L;

		final String icn1 = CBSDAOImplUtils.getTestIcn(stmt1Key);
		final String icn2 = CBSDAOImplUtils.getTestIcn(stmt2Key);

		Date statementDate = null;
		try {
			statementDate = (new SimpleDateFormat("MMddyyyy")).parse("06012016");
		} catch (ParseException e) {
		}

		// Create an already sent statement for station 1/patient1
		final CBSStmt sentCBSStmt1 = cbsDAOImplUtils.createSimpleTestCBSStmt(stmt1Key, CBSDAOImplUtils.STATUS_SENT_ID,
				stationNum1, icn1, statementDate);
		cbsDAOImplUtils.saveCBSStmt(sentCBSStmt1);
		wrapper.getCbsStmtIdS().add(sentCBSStmt1.getId());
		wrapper.getAccountIdS().add(sentCBSStmt1.getAccountId());

		// Create an already sent statement for station 1/patient2
		final CBSStmt sentCBSStmt2 = cbsDAOImplUtils.createSimpleTestCBSStmt(stmt2Key, CBSDAOImplUtils.STATUS_SENT_ID,
				stationNum1, icn2, statementDate);
		cbsDAOImplUtils.saveCBSStmt(sentCBSStmt2);
		wrapper.getCbsStmtIdS().add(sentCBSStmt2.getId());
		wrapper.getAccountIdS().add(sentCBSStmt2.getAccountId());

		final List<CBSStmt> cbsStmtL = new ArrayList<CBSStmt>(2);
		cbsStmtL.add(sentCBSStmt1);
		cbsStmtL.add(sentCBSStmt2);
		return cbsStmtL;
	}

	private List<CBSStmt> createCBSSet2(CBSDAOTestWrapper wrapper) {
		final int stmt1Key = 1;
		final int stmt2Key = 2;
		final int stmt3Key = 3;
		final int stmt4Key = 4;

		final long stationNum1 = 442L;
		final long stationNum2 = 555L;

		final String icn3 = CBSDAOImplUtils.getTestIcn(stmt3Key);
		final String icn4 = CBSDAOImplUtils.getTestIcn(stmt4Key);

		Date statementDate = null;
		try {
			statementDate = (new SimpleDateFormat("MMddyyyy")).parse("06012016");
		} catch (ParseException e) {
		}

		// Create statement sent/patient1/primarysite1/nonprimarysite2
		final CBSStmt sentCBSStmt1 = cbsDAOImplUtils.createDoubleTestCBSStmt(stmt1Key, CBSDAOImplUtils.STATUS_SENT_ID,
				stationNum1, stationNum2);
		sentCBSStmt1.setStatementDate(statementDate);
		cbsDAOImplUtils.saveCBSStmt(sentCBSStmt1);
		wrapper.getCbsStmtIdS().add(sentCBSStmt1.getId());
		wrapper.getAccountIdS().add(sentCBSStmt1.getAccountId());

		// Create statement sent/patient2/primarysite2/nonprimarysite1
		final CBSStmt sentCBSStmt2 = cbsDAOImplUtils.createDoubleTestCBSStmt(stmt2Key, CBSDAOImplUtils.STATUS_SENT_ID,
				stationNum2, stationNum1);
		sentCBSStmt2.setStatementDate(statementDate);
		cbsDAOImplUtils.saveCBSStmt(sentCBSStmt2);
		wrapper.getCbsStmtIdS().add(sentCBSStmt2.getId());
		wrapper.getAccountIdS().add(sentCBSStmt2.getAccountId());

		// Create statement sent/patient3/primarysite1
		final CBSStmt sentCBSStmt3 = cbsDAOImplUtils.createSimpleTestCBSStmt(stmt3Key, CBSDAOImplUtils.STATUS_SENT_ID,
				stationNum1, icn3, statementDate);
		cbsDAOImplUtils.saveCBSStmt(sentCBSStmt3);
		wrapper.getCbsStmtIdS().add(sentCBSStmt3.getId());
		wrapper.getAccountIdS().add(sentCBSStmt3.getAccountId());

		// Create statement sent/patient4/primarysite2
		final CBSStmt sentCBSStmt4 = cbsDAOImplUtils.createSimpleTestCBSStmt(stmt4Key, CBSDAOImplUtils.STATUS_SENT_ID,
				stationNum2, icn4, statementDate);
		cbsDAOImplUtils.saveCBSStmt(sentCBSStmt4);
		wrapper.getCbsStmtIdS().add(sentCBSStmt4.getId());
		wrapper.getAccountIdS().add(sentCBSStmt4.getAccountId());

		final List<CBSStmt> cbsStmtL = new ArrayList<CBSStmt>(2);
		cbsStmtL.add(sentCBSStmt1);
		cbsStmtL.add(sentCBSStmt2);
		cbsStmtL.add(sentCBSStmt3);
		cbsStmtL.add(sentCBSStmt4);
		return cbsStmtL;
	}

	private abstract class FileTestWrapper {

		private List<String> expectSuccessFileNames;
		private List<String> expectFailFileNames;

		protected FileTestWrapper(final List<String> expectSuccessFileNames, final List<String> expectFailFileNames) {
			this.expectSuccessFileNames = expectSuccessFileNames;
			this.expectFailFileNames = expectFailFileNames;
		}

		private void checkSetup() {
			final List<List<String>> fileNameLists = new ArrayList<List<String>>(2);
			fileNameLists.add(expectSuccessFileNames);
			fileNameLists.add(expectFailFileNames);

			for (List<String> fileNameL : fileNameLists) {
				if (fileNameL == null)
					continue;

				for (String fileName : fileNameL) {
					assertFalse(sftpService.ftpFileExistsInDirectory(fileName, testUpdatePrintAckJob.getDataDirectory(),
							true));
					assertFalse(sftpService.ftpFileExistsInDirectory(fileName,
							testUpdatePrintAckJob.getArchiveDirectory(), true));
					assertFalse(sftpService.ftpFileExistsInDirectory(fileName,
							testUpdatePrintAckJob.getErrorDirectory(), true));
				}
			}
		}

		private void stageFiles() {
			final List<List<String>> fileNameLists = new ArrayList<List<String>>(2);
			fileNameLists.add(expectSuccessFileNames);
			fileNameLists.add(expectFailFileNames);

			for (List<String> fileNameL : fileNameLists) {
				if (fileNameL == null)
					continue;

				for (String fileName : fileNameL) {
					assertTrue(ftpFileToServer(TEST_FILE_DIR_PATH + fileName));
				}
			}
		}

		private void checkDirectories() {
			if (expectSuccessFileNames != null) {
				for (String fileName : expectSuccessFileNames) {
					// Verify file is not in the data directory.
					assertFalse(sftpService.ftpFileExistsInDirectory(fileName, testUpdatePrintAckJob.getDataDirectory(),
							true));
					// Verify file is in the archive directory.
					assertTrue(sftpService.ftpFileExistsInDirectory(fileName,
							testUpdatePrintAckJob.getArchiveDirectory(), true));
					// Verify file is not in the error directory.
					assertFalse(sftpService.ftpFileExistsInDirectory(fileName,
							testUpdatePrintAckJob.getErrorDirectory(), true));
				}
			}
			if (expectFailFileNames != null) {
				for (String fileName : expectFailFileNames) {
					// Verify file is not in the data directory.
					assertFalse(sftpService.ftpFileExistsInDirectory(fileName, testUpdatePrintAckJob.getDataDirectory(),
							true));
					// Verify file is not in the archive directory.
					assertFalse(sftpService.ftpFileExistsInDirectory(fileName,
							testUpdatePrintAckJob.getArchiveDirectory(), true));
					// Verify file is in the error directory.
					assertTrue(sftpService.ftpFileExistsInDirectory(fileName, testUpdatePrintAckJob.getErrorDirectory(),
							true));
				}
			}
		}

		private void cleanup() {
			final List<List<String>> fileNameLists = new ArrayList<List<String>>(2);
			fileNameLists.add(expectSuccessFileNames);
			fileNameLists.add(expectFailFileNames);

			for (List<String> fileNameL : fileNameLists) {
				if (fileNameL == null)
					continue;

				for (String fileName : fileNameL) {
					sftpService.ftpRemoveFileFromDirectory(fileName, testUpdatePrintAckJob.getDataDirectory());
					sftpService.ftpRemoveFileFromDirectory(fileName, testUpdatePrintAckJob.getArchiveDirectory());
					sftpService.ftpRemoveFileFromDirectory(fileName, testUpdatePrintAckJob.getErrorDirectory());
				}
			}
		}

		protected void run() {
			// Ensure test server is in expected initial state.
			checkSetup();

			try {
				// FTP test files to server.
				stageFiles();

				// Run test
				test();

				checkDirectories();

			} finally {

				// Cleanup.
				cleanup();
			}
		}

		protected abstract void test();
	}

}
