package gov.va.cpss.job;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assume.assumeTrue;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;

import javax.sql.DataSource;

import org.apache.log4j.Logger;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.dao.CBSMessageDAO;
import gov.va.cpss.dao.CBSStmtDAO;
import gov.va.cpss.dao.impl.CBSDAOImplUtils;
import gov.va.cpss.dao.impl.CBSDAOTestWrapper;
import gov.va.cpss.dao.impl.SendCBSServiceTestWrapper;
import gov.va.cpss.model.cbs.CBSMessage;
import gov.va.cpss.model.cbs.CBSStmt;
import gov.va.cpss.service.SendCBSService;
import gov.va.cpss.service.SftpService;

/**
 * Tests for SendCBSJob.
 * 
 * @author Brad Pickle
 *
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml", "/dao-test-context.xml",
		"/cpss-batch.xml", "/send-cbs-test-context.xml", "/cpss-email.xml" })
public class SendCBSJobTest {

	@SuppressWarnings("unused")
	private final Logger testLogger = Logger.getLogger(this.getClass().getCanonicalName());

	@Value("${run.integration.test:false}")
	private Boolean runIntegrationTest;

	private final static int MESSAGE_STATUS_SUCCESS_ID = 7;

	private final static int STMT_STATUS_NEW_ID = 2;

	private final static int STMT_STATUS_SENT_ID = 4;

	private final static String TEST_FILE_PATH = "sendcbs/";

	private final static String EXPECTED_CBS_FILE1 = "expected_cbs1.txt";

	private final static String EXPECTED_SBS_FILE1 = "expected_sbs1.txt";

	private final static String CBSS_ACCOUNT_NUM1 = "${CBSSACCOUNTNUM1}";

	private final static String CBSS_ACCOUNT_NUM2 = "${CBSSACCOUNTNUM2}";

	@Autowired
	private SendCBSJobTestWrapper sendCBSJobTestWrapper;

	@Autowired
	private SendCBSService sendCBSService;

	@Autowired
	private SftpService sftpService;

	@Autowired
	private CBSDAOImplUtils cbsDAOImplUtils;

	@Autowired
	private CBSMessageDAO cbsMessageDao;
	
	@Autowired
	private CBSStmtDAO cbsStmtDao;

	private JdbcTemplate jdbcTemplate;

	@Autowired
	public void setDataSource(DataSource dataSource) {
		jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/**
	 * Only run these tests if property is set to run integration test.
	 */
	@Before
	public final void beforeTest() {
		assumeTrue(runIntegrationTest);
	}

	/**
	 * Test SendCBSJob using test statements created with getCBSStmtList1().
	 * Expect successful run.
	 */
	@Test
	public void test() {
		(new FileTestWrapper() {

			@Override
			protected void test() {
				(new SendCBSServiceTestWrapper(sendCBSService) {

					@Override
					protected void testImpl() {
						final List<CBSMessage> cbsMessagesL = new ArrayList<CBSMessage>();
						final List<CBSStmt> cbsStmtL = getCBSStmtList1(this, cbsMessagesL);

						final boolean testResult = sendCBSJobTestWrapper.testJob();

						final CBSStmt newStmt1 = cbsStmtDao.get(cbsStmtL.get(1).getId());
						CBSMessage cbsMessage = null;

						// Save references to things we need to clean up, if
						// possible, before checking results
						if ((newStmt1 != null) && (newStmt1.getMessageId() > 0)) {
							getCbsMessageIdS().add(newStmt1.getMessageId());
							cbsMessage = cbsMessageDao.get(newStmt1.getMessageId());
							if (cbsMessage != null) {
								setStatementFiles(cbsMessage);
								setCBSFileName(cbsMessage.getCBSFileName());
								setSBSFileName(cbsMessage.getSBSFileName());
							}
						}

						assertTrue(testResult);
						assertNull(sendCBSJobTestWrapper.getErrorMessage());

						final int batchRunId = sendCBSJobTestWrapper.getBatchRunId();
						assertTrue(batchRunId > 0);

						assertNotNull(newStmt1);
						assertNotNull(cbsMessage);
						final long messageId = cbsMessage.getId();
						assertTrue(messageId > 0);
						assertEquals(batchRunId, cbsMessage.getBatchRunId());
						assertEquals(MESSAGE_STATUS_SUCCESS_ID, cbsMessage.getStatusId());
						assertNotNull(cbsMessage.getFileName());
						assertNotNull(cbsMessage.getCreatedBy());
						assertNotNull(cbsMessage.getCreatedDate());
						assertNotNull(cbsMessage.getModifiedBy());
						assertNotNull(cbsMessage.getModifiedDate());

						for (CBSStmt cbsStmt : cbsStmtL.subList(1, cbsStmtL.size())) {
							CBSStmt newStmt = cbsStmtDao.get(cbsStmt.getId());
							assertNotNull(newStmt);
							assertEquals(newStmt.getId(), cbsStmt.getId());
							assertEquals(messageId, newStmt.getMessageId());
							assertEquals(STMT_STATUS_SENT_ID, newStmt.getStatusId());
						}

						// Make sure local CBS file was deleted
						final String cbsFilePath = sendCBSService.getCBSOutputResource(cbsMessage.getCBSFileName());
						final File cbsFile = new File(cbsFilePath);
						assertFalse(cbsFile.exists());

						// Make sure local CBS file was deleted
						final String sbsFilePath = sendCBSService.getCBSOutputResource(cbsMessage.getSBSFileName());
						final File sbsFile = new File(sbsFilePath);
						assertFalse(sbsFile.exists());

						// Make sure remote temporary file was deleted
						List<String> fileNameL = sftpService
								.ftpGetFileListInDirectory(sendCBSJobTestWrapper.getServerTargetDirectory());
						if (fileNameL != null) {
							for (String fileName : fileNameL) {
								assertFalse(fileName.endsWith(sendCBSJobTestWrapper.getTempFilenamePostfix()));
							}
						}

						assertTrue(sftpService.ftpFileExistsInDirectory(cbsMessage.getCBSFileName(),
								sendCBSJobTestWrapper.getServerTargetDirectory(), true));
						assertTrue(sftpService.ftpFileExistsInDirectory(cbsMessage.getSBSFileName(),
								sendCBSJobTestWrapper.getServerTargetDirectory(), true));

						JobOutputRemoteFile cbsRemoteFile = new JobOutputRemoteFile(sftpService,
								cbsMessage.getCBSFileName(), sendCBSJobTestWrapper.getServerTargetDirectory());
						JobOutputRemoteFile sbsRemoteFile = new JobOutputRemoteFile(sftpService,
								cbsMessage.getSBSFileName(), sendCBSJobTestWrapper.getServerTargetDirectory());

						try {
							final String cbssAccountNum1 = cbsStmtL.get(2).getAcntNumDisp();
							final String cbssAccountNum2 = cbsStmtL.get(3).getAcntNumDisp();

							String expectedCbsFile1Content = getExpectedCBSFile1Content(cbssAccountNum1,
									cbssAccountNum2);
							assertTrue(cbsRemoteFile.fileContentsEqual(expectedCbsFile1Content));

							final String cbssAccountNumSBS = cbsStmtL.get(1).getAcntNumDisp();
							String expectedSbsFile1Content = getExpectedSBSFile1Content(cbssAccountNumSBS);
							assertTrue(sbsRemoteFile.fileContentsEqual(expectedSbsFile1Content));
						} catch (Exception e) {
							throw new RuntimeException(e);
						}
					}

				}).run(jdbcTemplate);
			}

		}).run();
	}

	private List<CBSStmt> getCBSStmtList1(CBSDAOTestWrapper wrapper, List<CBSMessage> cbsMessagesL) {
		
		final long stationNum1 = 1L;
		final long stationNum2 = 2L;
		final long stationNum3 = 3L;

		final int sentBatchRunId = 1;
		final String sentFileName = "cbsTestFileNameSent.txt";

		// Create an already sent statement for station 1
		final CBSStmt sentCBSStmt1 = cbsDAOImplUtils.createSimpleTestCBSStmt(1, STMT_STATUS_SENT_ID, stationNum1);
		cbsDAOImplUtils.saveCBSStmt(sentCBSStmt1);
		wrapper.getCbsStmtIdS().add(sentCBSStmt1.getId());
		wrapper.getAccountIdS().add(sentCBSStmt1.getAccountId());

		// Create the CBSMessage record for sent statement
		final long sentMessageId = cbsDAOImplUtils.insertTestCBSMessage(cbsMessagesL, MESSAGE_STATUS_SUCCESS_ID,
				sentBatchRunId, sentFileName);
		wrapper.getCbsMessageIdS().add(sentMessageId);
		cbsDAOImplUtils.updateMessageIdForStmt(sentCBSStmt1.getId(), sentMessageId);
		sentCBSStmt1.setMessageId(sentMessageId);

		final CBSStmt newCBSStmt1 = cbsDAOImplUtils.createSimpleTestCBSStmt(2, STMT_STATUS_NEW_ID, stationNum1);
		cbsDAOImplUtils.saveCBSStmt(newCBSStmt1);
		wrapper.getCbsStmtIdS().add(newCBSStmt1.getId());
		wrapper.getAccountIdS().add(newCBSStmt1.getAccountId());

		// Create a NEW statement for station 1 and station2.
		// station 1 is primary
		final CBSStmt newCBSStmt1and2 = cbsDAOImplUtils.createDoubleTestCBSStmt(3, STMT_STATUS_NEW_ID, stationNum1,
				stationNum2);
		cbsDAOImplUtils.saveCBSStmt(newCBSStmt1and2);
		wrapper.getCbsStmtIdS().add(newCBSStmt1and2.getId());
		wrapper.getAccountIdS().add(newCBSStmt1and2.getAccountId());

		// Create a NEW statement for station 2 and station3.
		// station 2 is primary
		final CBSStmt newCBSStmt2and3 = cbsDAOImplUtils.createDoubleTestCBSStmt(4, STMT_STATUS_NEW_ID, stationNum2,
				stationNum3);
		cbsDAOImplUtils.saveCBSStmt(newCBSStmt2and3);
		wrapper.getCbsStmtIdS().add(newCBSStmt2and3.getId());
		wrapper.getAccountIdS().add(newCBSStmt2and3.getAccountId());

		final List<CBSStmt> cbsStmtL = new ArrayList<CBSStmt>(4);
		cbsStmtL.add(sentCBSStmt1);
		cbsStmtL.add(newCBSStmt1);
		cbsStmtL.add(newCBSStmt1and2);
		cbsStmtL.add(newCBSStmt2and3);

		return cbsStmtL;
	}

	private String getExpectedCBSFile1Content(String cbssAccountNum1, String cbssAccountNum2) throws IOException {
		File expectedCbsFilePath = new File(
				this.getClass().getClassLoader().getResource(TEST_FILE_PATH + EXPECTED_CBS_FILE1).getFile());

		String content = readFile(expectedCbsFilePath.getAbsolutePath());
		content = content.replace(CBSS_ACCOUNT_NUM1, String.format("%-18s", cbssAccountNum1));
		content = content.replace(CBSS_ACCOUNT_NUM2, String.format("%-18s", cbssAccountNum2));

		return content;
	}

	private String getExpectedSBSFile1Content(String cbssAccountNum1) throws IOException {
		File expectedSbsFilePath = new File(
				this.getClass().getClassLoader().getResource(TEST_FILE_PATH + EXPECTED_SBS_FILE1).getFile());

		String content = readFile(expectedSbsFilePath.getAbsolutePath());
		content = content.replace(CBSS_ACCOUNT_NUM1, String.format("%-18s", cbssAccountNum1));

		return content;
	}

	private static String readFile(String path) throws IOException {
		byte[] encoded = Files.readAllBytes(Paths.get(path));
		return new String(encoded, StandardCharsets.UTF_8);
	}

	private abstract class FileTestWrapper {
		private String cbsFileName;
		private String sbsFileName;

		private void cleanup() {
			final String serverTargetDirectory = sendCBSJobTestWrapper.getServerTargetDirectory();

			if (cbsFileName != null) {
				sftpService.ftpRemoveFileFromDirectory(cbsFileName, serverTargetDirectory);
				final String donFileName = cbsFileName.substring(0, cbsFileName.lastIndexOf(".txt")) + ".don";
				sftpService.ftpRemoveFileFromDirectory(donFileName, serverTargetDirectory);
			}

			if (sbsFileName != null) {
				sftpService.ftpRemoveFileFromDirectory(sbsFileName, serverTargetDirectory);
				final String donFileName = sbsFileName.substring(0, sbsFileName.lastIndexOf(".txt")) + ".don";
				sftpService.ftpRemoveFileFromDirectory(donFileName, serverTargetDirectory);
			}

			for (String tempFileName : sftpService.ftpGetFileListInDirectory(serverTargetDirectory)) {
				if ((tempFileName != null) && (tempFileName.equals(sendCBSJobTestWrapper.getTempFilenamePostfix()))) {
					sftpService.ftpRemoveFileFromDirectory(tempFileName, serverTargetDirectory);
				}
			}

		}

		protected void run() {
			// Ensure test server is in expected initial state.
			// checkSetup();

			try {
				// Run test
				test();

				// checkDirectories();

			} finally {

				// Cleanup.
				cleanup();
			}
		}

		protected abstract void test();

		protected void setCBSFileName(String cbsFileName) {
			this.cbsFileName = cbsFileName;
		}

		protected void setSBSFileName(String sbsFileName) {
			this.sbsFileName = sbsFileName;
		}

	}
}
