package gov.va.cpss.email.impl;

import java.io.IOException;
import java.net.Socket;
import java.util.Iterator;
import java.util.Scanner;

import org.apache.log4j.Logger;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.springframework.stereotype.Service;

import com.dumbster.smtp.SimpleSmtpServer;
import com.dumbster.smtp.SmtpMessage;

import gov.va.cpss.email.MockEmailServer;
import gov.va.cpss.email.SmtpMailConfigurer;

/**
 * Implementation of MockEmailServer using Dumbster. Must be provided an
 * SmtpMailConfigurer before the server is started, which is used to specify the
 * port on localhost to bind the server.
 * 
 * If deployed within a Spring context, the start() and stop() methods will be
 * called through the InitializingBean and DisposableBean interfaces.
 * 
 * @author Brad Pickle
 *
 */
@Service
public class DumbsterMockEmailServer implements MockEmailServer<SmtpMessage>, InitializingBean, DisposableBean {

	private static final Logger logger = Logger.getLogger(DumbsterMockEmailServer.class.getCanonicalName());

	private SmtpMailConfigurer smtpMailConfigurer;

	private SimpleSmtpServer simpleSmtpServer;

	public SmtpMailConfigurer getSmtpMailConfigurer() {
		return smtpMailConfigurer;
	}

	public void setSmtpMailConfigurer(SmtpMailConfigurer smtpMailConfigurer) {
		this.smtpMailConfigurer = smtpMailConfigurer;
	}

	@Override
	public void afterPropertiesSet() throws Exception {
		start();
	}

	@Override
	public void destroy() throws Exception {
		stop();
	}

	@Override
	public void start() {
		stop();

		String mailSmtpPort = smtpMailConfigurer.getMailSmtpPort();
		logger.info("Starting SimpleSmtpServer on port:" + mailSmtpPort);

		DumbsterMockEmailServer.waitForPort(Integer.parseInt(mailSmtpPort), 5000L, 3);
		simpleSmtpServer = SimpleSmtpServer.start(Integer.parseInt(mailSmtpPort));
	}
	
	private static boolean waitForPort(int port, long waitMillis, int numTries) {
		boolean available = DumbsterMockEmailServer.portAvailable(port);
		int count = 0;
		
		while ((! available) && (count++ < numTries)) {
			try {
				Thread.sleep(waitMillis);
			} catch (InterruptedException e) {
				logger.error(e.getMessage());
			}
			available = DumbsterMockEmailServer.portAvailable(port);
		}
		
		return available;
	}
	private static boolean portAvailable(int port) {
	    try (Socket socket = new Socket("localhost", port)) {
	        return false;
	    } catch (IOException e) {
	        return true;
	    }
	}

	@Override
	public void stop() {
		if (simpleSmtpServer != null) {
			logger.info("Stopping SimpleSmtpServer");
			try {
				simpleSmtpServer.stop();
			} catch (Exception e) {
				logger.error("Error stopping SimpleSmtpServer:" + e.getMessage());
			}
			simpleSmtpServer = null;
		}
	}

	@Override
	public int getReceivedEmailSize() {
		return (simpleSmtpServer == null) ? 0 : simpleSmtpServer.getReceivedEmailSize();
	}

	@Override
	public Iterator<SmtpMessage> getReceivedEmail() {
		if (simpleSmtpServer == null)
			return null;

		final Iterator<?> receivedEmail = simpleSmtpServer.getReceivedEmail();
		return (receivedEmail == null) ? null : new Iterator<SmtpMessage>() {
			@Override
			public boolean hasNext() {
				return receivedEmail.hasNext();
			}

			@Override
			public SmtpMessage next() {
				return (SmtpMessage) receivedEmail.next();
			}

		};
	}

	/**
	 * Start an instance of the MockEmailServer with Spring context,
	 * test-context.xml and test-email-server.xml. Server will run until "1" is
	 * entered on the command line. Note that the mail server should not be
	 * running during the jUnit tests, since those tests expect the given port
	 * to be available.
	 * 
	 * @param args
	 */
	public static void main(String[] args) {

		ConfigurableApplicationContext context = new ClassPathXmlApplicationContext("test-context.xml",
				"test-email-server.xml");

		run();

		context.close();
	}

	private final static void run() {
		String input = "0";
		Scanner keyboard = new Scanner(System.in);
		while (!input.equals("1")) {
			System.out.println("input = " + input);
			System.out.println("Press 1 to stop");
			try {
				input = keyboard.next();
			} catch (Exception e) {
				System.err.println("Invalid Input Detected");
				input = "0";
			}
		}
		keyboard.close();
	}

}
