package gov.va.cpss.dao.impl;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assume.assumeTrue;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.sql.DataSource;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.PreparedStatementCreator;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.dao.CBSPatAcntBalMsgDAO;
import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.dao.SiteBalDAO;
import gov.va.cpss.dao.SiteBalRecDAO;
import gov.va.cpss.dao.VistaAccountDAO;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.bal.PatAcntBalMsg;
import gov.va.cpss.model.bal.SiteBalRec;
import gov.va.cpss.model.bal.SiteBalance;
import gov.va.cpss.model.icn.VistaAccount;

@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml"})
/**
 * Unit test for SiteBalDAOImpl.
 * 
 * @author Andrew Vance
 *
 */
public final class SiteBalDAOImplTest {
	
	@Value("${run.integration.test:false}")
	private Boolean runIntegrationTest;
	
	@Autowired
	private SiteBalRecDAO siteBalRecDao;
	
	@Autowired
	private SiteBalDAO siteBalDao;
	
	@Autowired
	private CBSPatAcntBalMsgDAO cbsPatAcntBalMsgDao;
	
	@Autowired
	private VistaAccountDAO vistaAccountDao;
	
	@Autowired
	private ProcessStatusDAO processStatusDao;
	
	private List<SiteBalance> siteBalanceL;
	
	private List<VistaAccount> vistaAccountL;
	
	private List<PatAcntBalMsg> patAcntBalMsgL;
	
	private List<SiteBalRec> siteBalRecL;
	
	private JdbcTemplate jdbcTemplate;

	@Autowired
	public void setDataSource(DataSource dataSource) {
		jdbcTemplate = new JdbcTemplate(dataSource);
	}
	
	@Before
	public final void beforeTest() {
		assumeTrue(runIntegrationTest);
		
		insertTestData();
	}
	
	@After
	public final void afterTest() {
		if(runIntegrationTest){
			cleanupTestData();
		}		
	}
	
	@Test
	public final void testSimpleGet() {
		final SiteBalance siteBalanceExpected = siteBalanceL.get(0);
		
		final SiteBalance siteBalanceRetrieved = siteBalDao.getById(siteBalanceExpected.getId());
		
		assertSiteBalanceEquals(siteBalanceExpected, siteBalanceRetrieved);
	}
	
	@Test
	public final void testGetByAcctNumAndStationNum() {
		final SiteBalance siteBalanceExpected = siteBalanceL.get(0);
		
		final SiteBalance siteBalanceRetrieved = siteBalDao.getByAcctNumAndStationNum(siteBalanceExpected.getPatientDfn(), siteBalanceExpected.getStationNum());
		
		assertSiteBalanceEquals(siteBalanceExpected, siteBalanceRetrieved);
	}
	
	@Test
	public final void testUpdateSiteBalPatAcntBalMsgId() {
		final SiteBalance siteBalanceExpected = siteBalanceL.get(0);
		siteBalanceExpected.setPatientAcctBalMsg(patAcntBalMsgL.get(1));
		
		siteBalDao.updateSiteBalPatAcntBalMsgId(siteBalanceExpected.getPatientAcctBalMsg().getId());
		
		final SiteBalance siteBalanceRetrieved = siteBalDao.getById(siteBalanceExpected.getId());
		
		assertSiteBalanceEquals(siteBalanceExpected, siteBalanceRetrieved);
	}
	
	@Test
	public final void testGetTotalRecordsFromSiteBalToProcess() {
		assertTrue(siteBalDao.getTotalRecordsFromSiteBalToProcess() >= siteBalanceL.size());
	}
	
	private final void insertTestData() {
		vistaAccountL = new ArrayList<VistaAccount>();
		
		final VistaAccount vistaAccount1 = new VistaAccount();
		vistaAccount1.setDfn(7148281);
		vistaAccount1.setStationNum("123");
		vistaAccount1.setIcn("a1b2c3d4");
		vistaAccount1.setCbssAcntId(1111);
		vistaAccount1.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.NEW));
		assertTrue(saveVistaAccount(vistaAccount1));
		vistaAccountL.add(vistaAccount1);
		
		patAcntBalMsgL = new ArrayList<PatAcntBalMsg>();
		
		final PatAcntBalMsg patAcntBalMsg1 = new PatAcntBalMsg();
		patAcntBalMsg1.setBatchRunId(1);
		patAcntBalMsg1.setFileName("fakefile.test");
		patAcntBalMsg1.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.NEW));
		patAcntBalMsg1.setTotNumPatient(11);
		patAcntBalMsg1.setId(cbsPatAcntBalMsgDao.insert(patAcntBalMsg1));	
		patAcntBalMsgL.add(patAcntBalMsg1);
		
		final PatAcntBalMsg patAcntBalMsg2 = new PatAcntBalMsg();
		patAcntBalMsg2.setBatchRunId(12);
		patAcntBalMsg2.setFileName("fakefi2le2.test");
		patAcntBalMsg2.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.SUCCESS));
		patAcntBalMsg2.setTotNumPatient(121);
		patAcntBalMsg2.setId(cbsPatAcntBalMsgDao.insert(patAcntBalMsg2));
		patAcntBalMsgL.add(patAcntBalMsg2);	
		
		siteBalRecL = new ArrayList<SiteBalRec>();
		
		final SiteBalRec siteBalRec1 = new SiteBalRec(0);
		siteBalRec1.setFileName("fakefile.test");
		siteBalRec1.setBatchRunId(0);
		siteBalRec1.setTotNumFacility(1);
		siteBalRec1.setTotNumPatient(1);
		siteBalRec1.setProcessStatus(processStatusDao.getStatusFromEnum(ProcessStatus.Status.INITIAL));
		siteBalRec1.setId(siteBalRecDao.save(siteBalRec1));		
		siteBalRecL.add(siteBalRec1);
		
		siteBalanceL = new ArrayList<SiteBalance>();
		
		final SiteBalance siteBalance1 = new SiteBalance(0);
		siteBalance1.setPatientDfn(vistaAccount1.getDfn());
		siteBalance1.setStationNum(vistaAccount1.getStationNum());
		siteBalance1.setOldAcctNum("testoldacntnum");
		siteBalance1.setBalance(12345);
		siteBalance1.setOldestBillDate(Date.from(Instant.now()));
		siteBalance1.setProcessDate(Date.from(Instant.now()));
		siteBalance1.setPatientAcctBalMsg(patAcntBalMsg1);
		siteBalance1.setSiteBalRec(siteBalRec1);
		siteBalance1.setId(siteBalDao.save(siteBalance1));
		siteBalanceL.add(siteBalance1);
	}
	
	private final void cleanupTestData() {
		siteBalanceL.forEach(siteBalance -> { 
			final String sql = "DELETE FROM SiteBal WHERE ID=?";
			
			jdbcTemplate.update(new PreparedStatementCreator() {

				@Override
				public PreparedStatement createPreparedStatement(Connection con) throws SQLException {
					final PreparedStatement ps = con.prepareStatement(sql);
					ps.setLong(1, siteBalance.getId());
					return ps;
				}				
			});
		});
		siteBalRecL.forEach(siteBalRec -> siteBalRecDao.deleteById(siteBalRec.getId()));
		patAcntBalMsgL.forEach(patAcntBalMsg -> {
			final String sql = "DELETE FROM PatAcntBalMsg WHERE ID=?";
			
			jdbcTemplate.update(new PreparedStatementCreator() {
				@Override
				public PreparedStatement createPreparedStatement(Connection con) throws SQLException {
					final PreparedStatement ps = con.prepareStatement(sql);
					ps.setLong(1, patAcntBalMsg.getId());
					return ps;
				}
			});
		});
		vistaAccountL.forEach(acnt -> {
			final String sql = "DELETE FROM VistaAccount WHERE DFN=? AND StationNum=?";
			
			jdbcTemplate.update(new PreparedStatementCreator() {
				@Override
				public PreparedStatement createPreparedStatement(Connection con) throws SQLException {
					final PreparedStatement ps = con.prepareStatement(sql);
					ps.setLong(1, acnt.getDfn());
					ps.setString(2, acnt.getStationNum());
					return ps;
				}							
			});
		});


	}
	
	private final boolean saveVistaAccount(final VistaAccount vistaAccount) {
		return vistaAccountDao.save(vistaAccount.getStatusId(),
				vistaAccount.getDfn(), 
				vistaAccount.getStationNum(), 
				vistaAccount.getIcn(), 
				vistaAccount.getCbssAcntId());
	}
	
	private static final void assertSiteBalanceEquals(final SiteBalance expected, final SiteBalance retrieved) {
		assertEquals(expected.getBalance(), retrieved.getBalance(), 0);
		assertEquals(expected.getCbsAccount(), retrieved.getCbsAccount());
		assertEquals(expected.getOldAcctNum(), retrieved.getOldAcctNum());
		assertEquals(expected.getOldestBillDate().getTime(), retrieved.getOldestBillDate().getTime(), 1000);
		//assertEquals(expected.getPatientAcctBalMsg().getId(), retrieved.getPatientAcctBalMsg().getId());
		assertEquals(expected.getPatientDfn(), retrieved.getPatientDfn());
		//assertEquals(expected.getProcessDate().getTime(), retrieved.getProcessDate().getTime(), 1000);
		assertEquals(expected.getSiteBalRec().getId(), retrieved.getSiteBalRec().getId());
		assertEquals(expected.getStationNum(), retrieved.getStationNum());
	}

}
