package gov.va.cpss.dao.impl;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Files;

import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.cbs.CBSMessage;
import gov.va.cpss.service.SendCBSService;

/**
 * Extends the test wrapper to provide support for a CBS file and status DAO
 * proxies
 * 
 * @author Brad Pickle
 *
 */
public abstract class SendCBSServiceTestWrapper extends CBSDAOTestWrapper {
	
	private File cbsFile;
	private File sbsFile;
	private ProcessStatusDAO processStatusDAO;
	
	private SendCBSService sendCBSService;

	public SendCBSServiceTestWrapper(SendCBSService sendCBSService) {
		this.sendCBSService = sendCBSService;
	}
	
	public File getCbsFile() {
		return cbsFile;
	}

	public File getSbsFile() {
		return sbsFile;
	}

	@Override
	protected final void test() {
		try {
			testImpl();
		} finally {
			if (cbsFile != null) {
				try {
					Files.deleteIfExists(cbsFile.toPath());
				} catch (IOException e) {
					logger.error(e.getMessage());
				}
			}

			if (sbsFile != null) {
				try {
					Files.deleteIfExists(sbsFile.toPath());
				} catch (IOException e) {
					logger.error(e.getMessage());
				}
			}

			if (processStatusDAO != null) {
				sendCBSService.setProcessStatusDAO(processStatusDAO);
			}
		}
	}

	protected abstract void testImpl();

	private void saveProcessStatusDAO() {
		if (processStatusDAO == null) {
			processStatusDAO = sendCBSService.getProcessStatusDAO();
		}
	}

	protected void failOnMessageStatus(final ProcessStatus.Status failEnumValue) {
		saveProcessStatusDAO();
		overrideProcessStatusDAO(new FailProcessStatusDAOImpl(failEnumValue, processStatusDAO));

	}

	protected void overrideProcessStatusDAO(final ProcessStatusDAO newDao) {
		saveProcessStatusDAO();
		sendCBSService.setProcessStatusDAO(newDao);
	}

	protected void failOnStmtStatus(final ProcessStatus.Status failEnumValue) {
		saveProcessStatusDAO();
		overrideCBSStmtStatusDAO(new FailCBSStmtStatusDAOImpl(failEnumValue, processStatusDAO));

	}

	protected void overrideCBSStmtStatusDAO(final ProcessStatusDAO newDao) {
		saveProcessStatusDAO();
		sendCBSService.setProcessStatusDAO(newDao);
	}

	protected void touchStatementFiles(final CBSMessage cbsMessage) {
		cbsFile = touchStatementFile(cbsMessage.getCBSFileName());
		sbsFile = touchStatementFile(cbsMessage.getSBSFileName());
	}
	
	private File touchStatementFile(final String statementFileName) {
		assertNotNull(statementFileName);
		assertTrue(statementFileName.length() > 0);

		final String statementFilePath = sendCBSService.getCBSOutputResource(statementFileName);
		assertNotNull(statementFilePath);
		assertFalse(statementFileName.equals(statementFilePath));
		assertTrue(statementFilePath.length() > statementFileName.length());
		assertEquals(statementFilePath.substring(statementFilePath.length() - statementFileName.length()), statementFileName);

		final File tempStatementFile = new File(statementFilePath);
		assertFalse(tempStatementFile.exists());

		final File statementFile = new File(statementFilePath);
		try {
			new FileOutputStream(statementFile).close();
			statementFile.setLastModified(System.currentTimeMillis());
		} catch (IOException e) {
			logger.error(e.getMessage());
		}

		assertTrue(statementFile.exists());
		
		return statementFile;
	}
	
	protected void setStatementFiles(final CBSMessage cbsMessage) {
		final String cbsFileName = cbsMessage.getCBSFileName();
		if ((cbsFileName != null) && (cbsFileName.length() > 0)) {
			final String cbsFilePath = sendCBSService.getCBSOutputResource(cbsFileName);
			
			cbsFile = new File(cbsFilePath);
		}
		
		final String sbsFileName = cbsMessage.getSBSFileName();
		if ((sbsFileName != null) && (sbsFileName.length() > 0)) {
			final String sbsFilePath = sendCBSService.getCBSOutputResource(sbsFileName);
			
			sbsFile = new File(sbsFilePath);
		}
	}

	/**
	 * A ProcessStatusDAO that will return null for a specified status.
	 * 
	 * @author Brad Pickle
	 *
	 */
	private class FailProcessStatusDAOImpl extends ProcessStatusDAOImpl {

		private ProcessStatus.Status failEnumValue;

		private ProcessStatusDAO processStatusDAO;

		/**
		 * Instantiate FailProcessStatusDAOImpl for the given status.
		 * 
		 * @param failEnumValue
		 *            The status which should fail.
		 * @param saveMessageStatusDAO
		 *            All status besides failEnumValue will be delegated to this
		 *            DAO.
		 */
		protected FailProcessStatusDAOImpl(ProcessStatus.Status failEnumValue,
				ProcessStatusDAO saveProcessStatusDAO) {
			this.failEnumValue = failEnumValue;
			this.processStatusDAO = saveProcessStatusDAO;
		}

		/**
		 * Returns null if enumValue is the failEnumValue specified in the
		 * constructor. Otherwise returns value from the delegated DAO.
		 * 
		 * @param enumValue
		 *            Status to get.
		 * @return The status id, or null if failEnumValue.
		 */
		@Override
		public Integer getStatusFromEnum(ProcessStatus.Status enumValue) {
			return (failEnumValue == enumValue) ? null : processStatusDAO.getStatusFromEnum(enumValue);
		}
	}

	/**
	 * A CBSStmtStatusDAO that will return null for a specified status.
	 * 
	 * @author Brad Pickle
	 *
	 */
	private class FailCBSStmtStatusDAOImpl extends ProcessStatusDAOImpl {

		private ProcessStatus.Status failEnumValue;

		private ProcessStatusDAO saveStmtStatusDAO;

		/**
		 * Instantiate FailCBSStmtStatusDAOImpl for the given status.
		 * 
		 * @param failEnumValue
		 *            The status which should fail.
		 * @param saveStmtStatusDAO
		 *            All status besides failEnumValue will be delegated to this
		 *            DAO.
		 */
		protected FailCBSStmtStatusDAOImpl(ProcessStatus.Status failEnumValue, ProcessStatusDAO saveStmtStatusDAO) {
			this.failEnumValue = failEnumValue;
			this.saveStmtStatusDAO = saveStmtStatusDAO;
		}

		/**
		 * Returns null if enumValue is the failEnumValue specified in the
		 * constructor. Otherwise returns value from the delegated DAO.
		 * 
		 * @param enumValue
		 *            Status to get.
		 * @return The status id, or null if failEnumValue.
		 */
		@Override
		public Integer getStatusFromEnum(ProcessStatus.Status enumValue) {
			return (failEnumValue == enumValue) ? null : saveStmtStatusDAO.getStatusFromEnum(enumValue);
		}
	}


}
