package gov.va.cpss.dao.impl;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assume.assumeTrue;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.sql.DataSource;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.dao.CBSSiteStmtDAO;
import gov.va.cpss.model.cbs.CBSMessage;
import gov.va.cpss.model.cbs.CBSSitePatient;
import gov.va.cpss.model.cbs.CBSSiteStmt;
import gov.va.cpss.model.cbs.CBSStmt;

/**
 * Unit tests the CBSSiteStmtDAOImpl class.
 * 
 * @author Brad Pickle
 *
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml", "/dao-test-context.xml" })
public class CBSSiteStmtDAOImplTest {

	@Value("${run.integration.test:false}")
	private Boolean runIntegrationTest;

	@Autowired
	private CBSDAOImplUtils cbsDAOImplUtils;

	@Autowired
	private CBSSiteStmtDAO cbsSiteStmtDao;

	private JdbcTemplate jdbcTemplate;

	@Autowired
	public void setDataSource(DataSource dataSource) {
		jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/**
	 * Only run these tests if property is set to run integration test.
	 */
	@Before
	public final void beforeTest() {
		assumeTrue(runIntegrationTest);
	}

	/**
	 * Tests the save() and getAllByCBSStmtID() methods of CBSSiteStmtDAOImpl
	 */
	@Test
	public final void testSaveGet() {
		(new CBSDAOTestWrapper() {

			@Override
			protected void test() {
				// Test Data
				final int cbsStmtKey = 1;
				final int statusId = 1;
				final long stationNumLong = 12345L;

				// Initialize the simple test CBSStmt object. The object graph
				// of
				// one
				// CBSStmt, one CBSSiteStmt, one CBSSitePatient and two
				// CBSSiteTrans
				// objects will be set up, but
				// the ids of all objects will not be initialized. A CBSAccount
				// is
				// created in the database.
				final CBSStmt cbsStmt = cbsDAOImplUtils.createSimpleTestCBSStmt(cbsStmtKey, statusId, stationNumLong);

				// Assume functioning CBSSiteStmt
				final CBSSiteStmt cbsSiteStmt = cbsStmt.getSiteStmtL().get(0);
				final CBSSitePatient cbsSitePatient = cbsSiteStmt.getSitePatient();

				final CBSSiteStmt originalCbsSiteStmt = cbsDAOImplUtils.copyCBSSiteStmt(cbsSiteStmt);
				final CBSSitePatient originalCbsSitePatient = originalCbsSiteStmt.getSitePatient();

				// Create the CBSStmt in the database. CBSSiteStmtDAOImpl.save()
				// will
				// be called.
				// Necessary to create dependent CBSAccount, CBSStmt and
				// CBSSiteStmt
				// records.
				cbsDAOImplUtils.saveCBSStmt(cbsStmt);
				getCbsStmtIdS().add(cbsStmt.getId());
				getAccountIdS().add(cbsStmt.getAccountId());

				final long cbsSiteStmtId = cbsSiteStmt.getId();
				final long stmtId = cbsSiteStmt.getStmtId();
				final String createdBy = cbsSiteStmt.getCreatedBy();
				final Date createdDate = cbsSiteStmt.getCreatedDate();
				final String modifiedBy = cbsSiteStmt.getModifiedBy();
				final Date modifiedDate = cbsSiteStmt.getModifiedDate();

				assertTrue(cbsSiteStmtId > 0);
				// assertNotNull(createdBy);
				assertNotNull(createdDate);
				// assertNotNull(modifiedBy);
				assertNotNull(modifiedDate);

				// Fields not set before save
				originalCbsSiteStmt.setId(cbsSiteStmtId);
				originalCbsSiteStmt.setStmtId(stmtId);
				originalCbsSiteStmt.setCreatedBy(createdBy);
				originalCbsSiteStmt.setCreatedDate(createdDate);
				originalCbsSiteStmt.setModifiedBy(modifiedBy);
				originalCbsSiteStmt.setModifiedDate(modifiedDate);

				originalCbsSitePatient.setId(cbsSitePatient.getId());
				originalCbsSitePatient.setSiteStmtId(cbsSitePatient.getSiteStmtId());
				originalCbsSitePatient.setCreatedBy(cbsSitePatient.getCreatedBy());
				originalCbsSitePatient.setCreatedDate(cbsSitePatient.getCreatedDate());
				originalCbsSitePatient.setModifiedBy(cbsSitePatient.getModifiedBy());
				originalCbsSitePatient.setModifiedDate(cbsSitePatient.getModifiedDate());

				// Assert relevant fields equal with original
				cbsDAOImplUtils.compareCBSSiteStmt(cbsSiteStmt, originalCbsSiteStmt);

				// Create a second statement that should be excluded from get
				final int cbsStmtKey2 = 2;
				final CBSStmt cbsStmt2 = cbsDAOImplUtils.createSimpleTestCBSStmt(cbsStmtKey2, statusId, stationNumLong);
				cbsDAOImplUtils.saveCBSStmt(cbsStmt2);
				getCbsStmtIdS().add(cbsStmt2.getId());
				getAccountIdS().add(cbsStmt2.getAccountId());

				// Test getAllByCBSStmtId with loadSitePatient = false;
				final List<CBSSiteStmt> newSiteStmtL = cbsSiteStmtDao.getAllByCBSStmtID(stmtId, false);

				assertNotNull(newSiteStmtL);
				assertEquals(newSiteStmtL.size(), 1);

				final CBSSiteStmt newCbsSiteStmt = newSiteStmtL.get(0);

				assertNotNull(newCbsSiteStmt);

				cbsDAOImplUtils.compareCBSSiteStmt(cbsSiteStmt, newCbsSiteStmt);
				assertNull(newCbsSiteStmt.getSitePatient());

				// Test getAllByCBSStmtId with loadSitePatient = true;
				final List<CBSSiteStmt> newSiteStmtWithPatientL = cbsSiteStmtDao.getAllByCBSStmtID(stmtId, true);

				assertNotNull(newSiteStmtWithPatientL);
				assertEquals(newSiteStmtWithPatientL.size(), 1);

				final CBSSiteStmt newCbsSiteStmtWithPatient = newSiteStmtWithPatientL.get(0);

				assertNotNull(newCbsSiteStmtWithPatient);

				cbsDAOImplUtils.compareCBSSiteStmt(cbsSiteStmt, newCbsSiteStmtWithPatient);

				final CBSSitePatient newCbsSitePatient = newCbsSiteStmtWithPatient.getSitePatient();
				assertNotNull(newCbsSitePatient);

				originalCbsSitePatient.setModifiedBy(newCbsSitePatient.getModifiedBy());
				originalCbsSitePatient.setModifiedDate(newCbsSitePatient.getModifiedDate());

				cbsDAOImplUtils.compareCBSSitePatients(originalCbsSitePatient, newCbsSitePatient);

				assertEquals(jdbcTemplate.update("DELETE FROM CBSStmt WHERE id=" + stmtId), 1);
				getCbsStmtIdS().remove(stmtId);

				// Test getAllByCBSStmtId with loadSitePatient = false;
				final List<CBSSiteStmt> emptySiteStmtL = cbsSiteStmtDao.getAllByCBSStmtID(stmtId, false);
				assertNotNull(emptySiteStmtL);
				assertEquals(emptySiteStmtL.size(), 0);
			}

		}).run(jdbcTemplate);
	}

	@Test
	public void testGetMessageStatementSitesForSite() {
		(new CBSDAOTestWrapper() {

			@Override
			protected void test() {
				final long stationNum1 = 1L;
				final long stationNum2 = 2L;
				final long stationNum3 = 3L;

				final int stmtStatusNewId = 2;
				final int stmtStatusSentId = 3;

				final int messageStatusInitialId = 1;
				final int messageStatusSuccessId = 2;

				final int sentBatchRunId = 1;
				final String sentFileName = "cbsTestFileNameSent.txt";

				final List<CBSMessage> cbsMessagesL = new ArrayList<CBSMessage>();

				// Create an already sent statement for station 1
				final CBSStmt sentCBSStmt1 = cbsDAOImplUtils.createSimpleTestCBSStmt(1, stmtStatusSentId, stationNum1);
				cbsDAOImplUtils.saveCBSStmt(sentCBSStmt1);
				getCbsStmtIdS().add(sentCBSStmt1.getId());
				getAccountIdS().add(sentCBSStmt1.getAccountId());

				// Create the CBSMessage record for sent statement
				final long sentMessageId = cbsDAOImplUtils.insertTestCBSMessage(cbsMessagesL, messageStatusSuccessId,
						sentBatchRunId, sentFileName);
				getCbsMessageIdS().add(sentMessageId);
				cbsDAOImplUtils.updateMessageIdForStmt(sentCBSStmt1.getId(), sentMessageId);

				final List<CBSSiteStmt> simpleResultsL = cbsSiteStmtDao.getMessageStatementSitesForSite(sentMessageId,
						"" + stationNum1);
				assertNotNull(simpleResultsL);
				assertEquals(simpleResultsL.size(), 1);
				assertNotNull(simpleResultsL.get(0));
				assertNotNull(simpleResultsL.get(0).getSitePatient());
				cbsDAOImplUtils.compareCBSSiteStmt(sentCBSStmt1.getSiteStmtL().get(0), simpleResultsL.get(0));
				cbsDAOImplUtils.compareCBSSitePatients(sentCBSStmt1.getSiteStmtL().get(0).getSitePatient(),
						simpleResultsL.get(0).getSitePatient());

				// Create a NEW statement for station 1
				final CBSStmt newCBSStmt1 = cbsDAOImplUtils.createSimpleTestCBSStmt(2, stmtStatusNewId, stationNum1);
				cbsDAOImplUtils.saveCBSStmt(newCBSStmt1);
				getCbsStmtIdS().add(newCBSStmt1.getId());
				getAccountIdS().add(newCBSStmt1.getAccountId());

				// Create a NEW statement for station 1 and station2. station 1
				// is primary
				final CBSStmt newCBSStmt1and2 = cbsDAOImplUtils.createDoubleTestCBSStmt(3, stmtStatusNewId, stationNum1,
						stationNum2);
				cbsDAOImplUtils.saveCBSStmt(newCBSStmt1and2);
				getCbsStmtIdS().add(newCBSStmt1and2.getId());
				getAccountIdS().add(newCBSStmt1and2.getAccountId());

				// Create a NEW statement for station 2 and station3. station 2
				// is primary
				final CBSStmt newCBSStmt2and3 = cbsDAOImplUtils.createDoubleTestCBSStmt(4, stmtStatusNewId, stationNum2,
						stationNum3);
				cbsDAOImplUtils.saveCBSStmt(newCBSStmt2and3);
				getCbsStmtIdS().add(newCBSStmt2and3.getId());
				getAccountIdS().add(newCBSStmt2and3.getAccountId());

				// Create the CBSMessage record for NEW statement
				final int newBatchRunId = 2;
				final String newFileName = "cbsTestFileNameNew.txt";
				final long newMessageId = cbsDAOImplUtils.insertTestCBSMessage(cbsMessagesL, messageStatusInitialId,
						newBatchRunId, newFileName);
				getCbsMessageIdS().add(newMessageId);
				cbsDAOImplUtils.updateMessageIdForStmt(newCBSStmt1.getId(), newMessageId);
				cbsDAOImplUtils.updateMessageIdForStmt(newCBSStmt1and2.getId(), newMessageId);
				cbsDAOImplUtils.updateMessageIdForStmt(newCBSStmt2and3.getId(), newMessageId);

				final List<CBSSiteStmt> doubleStat1ResultsL = cbsSiteStmtDao
						.getMessageStatementSitesForSite(newMessageId, "" + stationNum1);
				assertNotNull(doubleStat1ResultsL);
				assertEquals(doubleStat1ResultsL.size(), 3);
				final List<CBSStmt> newStmts = new ArrayList<CBSStmt>();
				newStmts.add(newCBSStmt1);
				newStmts.add(newCBSStmt1and2);
				for (CBSSiteStmt cbsSiteStmt : doubleStat1ResultsL) {
					assertNotNull(cbsSiteStmt);
					assertNotNull(cbsSiteStmt.getSitePatient());
					CBSSiteStmt expectedCbsSiteStmt = getStmtWithId(newStmts, cbsSiteStmt.getId());
					assertNotNull(expectedCbsSiteStmt);
					cbsDAOImplUtils.compareCBSSiteStmt(expectedCbsSiteStmt, cbsSiteStmt);
					cbsDAOImplUtils.compareCBSSitePatients(expectedCbsSiteStmt.getSitePatient(),
							cbsSiteStmt.getSitePatient());
				}

				final List<CBSSiteStmt> doubleStat2ResultsL = cbsSiteStmtDao
						.getMessageStatementSitesForSite(newMessageId, "" + stationNum2);
				assertNotNull(doubleStat2ResultsL);
				assertEquals(doubleStat2ResultsL.size(), 2);
				final List<CBSStmt> newStmtsStation2 = new ArrayList<CBSStmt>();
				newStmtsStation2.add(newCBSStmt2and3);
				for (CBSSiteStmt cbsSiteStmt : doubleStat2ResultsL) {
					assertNotNull(cbsSiteStmt);
					assertNotNull(cbsSiteStmt.getSitePatient());
					CBSSiteStmt expectedCbsSiteStmt = getStmtWithId(newStmtsStation2, cbsSiteStmt.getId());
					assertNotNull(expectedCbsSiteStmt);
					cbsDAOImplUtils.compareCBSSiteStmt(expectedCbsSiteStmt, cbsSiteStmt);
					cbsDAOImplUtils.compareCBSSitePatients(expectedCbsSiteStmt.getSitePatient(),
							cbsSiteStmt.getSitePatient());
				}

				final List<CBSSiteStmt> doubleStat3ResultsL = cbsSiteStmtDao
						.getMessageStatementSitesForSite(newMessageId, "" + stationNum3);
				assertNotNull(doubleStat3ResultsL);
				assertEquals(doubleStat3ResultsL.size(), 0);
			}

		}).run(jdbcTemplate);
	}

	private CBSSiteStmt getStmtWithId(List<CBSStmt> cbsStmtL, long cbsSiteStmtId) {
		CBSSiteStmt foundStmt = null;
		for (CBSStmt cbsStmt : cbsStmtL) {
			for (CBSSiteStmt cbsSiteStmt : cbsStmt.getSiteStmtL()) {
				if (cbsSiteStmt.getId() == cbsSiteStmtId) {
					assertNull(foundStmt); // Should only find 1
					foundStmt = cbsSiteStmt;
				}
			}
		}

		return foundStmt;
	}

}
