package gov.va.cpss.dao.impl;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.sql.DataSource;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.PreparedStatementCreator;
import org.springframework.jdbc.support.GeneratedKeyHolder;
import org.springframework.jdbc.support.KeyHolder;
import org.springframework.stereotype.Component;

import gov.va.cpss.cobol.Money;
import gov.va.cpss.dao.BatchJobDAO;
import gov.va.cpss.dao.BatchRunDAO;
import gov.va.cpss.dao.CBSAccountDAO;
import gov.va.cpss.dao.CBSMessageDAO;
import gov.va.cpss.dao.CBSSitePatientDAO;
import gov.va.cpss.dao.CBSSiteStmtDAO;
import gov.va.cpss.dao.CBSSiteTransDAO;
import gov.va.cpss.dao.CBSStmtDAO;
import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.BooleanChar;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.cbs.CBSMessage;
import gov.va.cpss.model.cbs.CBSSitePatient;
import gov.va.cpss.model.cbs.CBSSiteStmt;
import gov.va.cpss.model.cbs.CBSSiteTrans;
import gov.va.cpss.model.cbs.CBSStmt;
import gov.va.cpss.model.fps.PSDetails;
import gov.va.cpss.model.fps.PSPatient;
import gov.va.cpss.model.fps.PSReceived;
import gov.va.cpss.model.fps.PSSite;
import gov.va.cpss.model.printack.ADFileRecord;
import gov.va.cpss.model.printack.PrintAckRec;

/**
 * Utility class used for testing different DAO classes.
 * 
 */
@Component
public class CBSDAOImplUtils {
	public static final int STATUS_INITIAL_ID = 1;
	public static final int STATUS_NEW_ID = 2;
	public static final int STATUS_PROCESSED_ID = 3;
	public static final int STATUS_SENT_ID = 4;
	public static final int STATUS_ACK_ID = 5;
	public static final int STATUS_REPLACED_ID = 6;
	public static final int STATUS_SUCCESS_ID = 7;
	public static final int STATUS_ERROR_ID = 8;
	public static final int STATUS_OTHER_ID = 9;
	
	@Autowired
	private CBSAccountDAO cbsAccountDao;

	@Autowired
	private CBSMessageDAO cbsMessageDao;

	@Autowired
	private CBSStmtDAO cbsStmtDao;

	@Autowired
	private CBSSiteStmtDAO cbsSiteStmtDao;

	@Autowired
	private CBSSitePatientDAO cbsSitePatientDao;

	@Autowired
	private CBSSiteTransDAO cbsSiteTransDao;
	
	@Autowired
	private BatchJobDAO batchJobDao;
	
	@Autowired
	private BatchRunDAO batchRunDao;
	
	@Autowired
	private ProcessStatusDAO processStatusDao;

	private JdbcTemplate jdbcTemplate;

	@Autowired
	public void setDataSource(DataSource dataSource) {
		jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/**
	 * Creates a new test CBSMessage with the given statusId, batchRunId and
	 * fileName and inserts into database.
	 * 
	 * @param cbsMessagesL
	 *            The new CBSMessage will be added to the end of this list.
	 * @param statusId
	 *            The new CBSMessage will be given this status id.
	 * @param batchRunId
	 *            The new CBSMessage will be given this batchRunId.
	 * @param fileName
	 *            The new CBSMessage will be given this fileName.
	 * @return The id of the new CBSMessage record.
	 */
	public long insertTestCBSMessage(final List<CBSMessage> cbsMessagesL, final int statusId, final long batchRunId,
			final String fileName) {

		final CBSMessage cbsMessage = new CBSMessage();

		cbsMessage.setStatusId(statusId);
		cbsMessage.setBatchRunId(batchRunId);
		cbsMessage.setFileName(fileName);

		final long id = cbsMessageDao.insert(cbsMessage);
		cbsMessagesL.add(cbsMessage);

		return id;
	}

	/**
	 * Generates a test ICN using the provided key.
	 * 
	 * @param key
	 *            A value added to the key to make it unique among test cases.
	 * @return A test ICN.
	 */
	public static String getTestIcn(final int key) {
		return String.format("TESTICNNUMBER%02d", key);
	}

	/**
	 * Create and initialize the simple CBSStmt object. The object graph of one
	 * CBSStmt, one CBSSiteStmt, one CBSSitePatient and two CBSSiteTrans
	 * objects. will be set up. A CBSAccount record will be created in the
	 * database and its id will be associated with the CBSStmt object. No ids
	 * are initialized on the CBSStmt graph returned.
	 * 
	 * @param key
	 *            An int representing this CBSStmt object unique among
	 *            additional test objects created by calling this method
	 *            multiple times.
	 * @param statusId
	 *            The status id to set for the CBSStmt.
	 * @param stationNumLong
	 *            The station number to use for the CBSSiteStmt.
	 * @return The simple CBSStmt object that is created.
	 */
	public CBSStmt createSimpleTestCBSStmt(final int key, final int statusId, final long stationNumLong) {

		// Initialize the simple test CBSStmt object. The object graph of one
		// CBSStmt, one CBSSiteStmt, one CBSSitePatient and two CBSSiteTrans
		// objects will be set up, but
		// the ids of all objects will not be initialized. A CBSAccount is
		// created in the database.
		final String icn = getTestIcn(key);

		Date statementDate = null;
		try {
			statementDate = (new SimpleDateFormat("MMddyyyy")).parse("05252016");
		} catch (ParseException e) {
		}

		return createSimpleTestCBSStmt(key, statusId, stationNumLong, icn, statementDate);
	}

	/**
	 * Create and initialize the simple CBSStmt object. The object graph of one
	 * CBSStmt, one CBSSiteStmt, one CBSSitePatient and two CBSSiteTrans
	 * objects. will be set up. A CBSAccount record will be created in the
	 * database and its id will be associated with the CBSStmt object. No ids
	 * are initialized on the CBSStmt graph returned.
	 * 
	 * @param key
	 *            An int representing this CBSStmt object unique among
	 *            additional test objects created by calling this method
	 *            multiple times.
	 * @param statusId
	 *            The status id to set for the CBSStmt.
	 * @param stationNumLong
	 *            The station number to use for the CBSSiteStmt.
	 * @param icn
	 *            The icn to use for the CBSStmt.
	 * @param statementDate
	 *            The statementDate to use for the CBSStmt.
	 * @return The simple CBSStmt object that is created.
	 */
	public CBSStmt createSimpleTestCBSStmt(final int key, final int statusId, final long stationNumLong,
			final String icn, final Date statementDate) {

		// Create the CBSAccount record
		final long accountId = cbsAccountDao.getByICN(icn);


		return CBSDAOImplUtils.createSimpleTestCBSStmtObject(accountId, key, statusId, stationNumLong, icn, statementDate);
	}
	
	public static CBSStmt createSimpleTestCBSStmtObject(final long accountId, final int key, final int statusId, final long stationNumLong,
			final String icn, final Date statementDate) {

		// Initialize CBSSiteStmt
		final List<CBSSiteStmt> siteStmtL = new ArrayList<CBSSiteStmt>();

		CBSSiteStmt cbsSiteStmt = createTestCBSSiteStmt(icn, stationNumLong, key, BooleanChar.Y, BooleanChar.Y);
		siteStmtL.add(cbsSiteStmt);
		cbsSiteStmt.setStatementDate(statementDate);

		// Test Data
		final String lastName5 = cbsSiteStmt.getSitePatient().getLastName().substring(0, 5);
		final String acntNumDisp = cbsSiteStmt.getStationNum() + accountId + lastName5;
		final String lateStmtMsg = "N";
		final long batchRunId = 0L;
		final long replacedStmtId = 0L;
		final long messageId = 0L;

		// Initialize CBSStmt
		CBSStmt cbsStmt = new CBSStmt(accountId, statusId);

		cbsStmt.setAcntNumDisp(acntNumDisp);
		cbsStmt.setAmountDue(cbsSiteStmt.getAmountDue());
		cbsStmt.setPrevBalance(cbsSiteStmt.getPrevBalance());
		cbsStmt.setTotalCharges(cbsSiteStmt.getTotalCharges());
		cbsStmt.setTotalCredits(cbsSiteStmt.getTotalCredits());
		cbsStmt.setNewBalance(cbsSiteStmt.getNewBalance());
		cbsStmt.setLateStmtMsg(lateStmtMsg == "Y");
		cbsStmt.setStatementDate(cbsSiteStmt.getStatementDate());
		cbsStmt.setProcessDate(cbsSiteStmt.getProcessDate());
		cbsStmt.setBatchRunId(batchRunId);
		cbsStmt.setReplacedStmtId(replacedStmtId);
		cbsStmt.setMessageId(messageId);
		cbsStmt.setSiteStmtL(siteStmtL);

		return cbsStmt;
	}

	public CBSStmt createDoubleTestCBSStmt(final int key, final int statusId, final long station1NumLong,
			final long station2NumLong) {

		final String icn = getTestIcn(key);

		// Create the CBSAccount record
		final long icnId = cbsAccountDao.getByICN(icn);

		return CBSDAOImplUtils.createDoubleTestCBSStmtObject(icnId, icn, key, statusId, station1NumLong, station2NumLong);
	}

	public static CBSStmt createDoubleTestCBSStmtObject(final long icnId, final String icn, final int key, final int statusId, final long station1NumLong,
			final long station2NumLong) {

		// Initialize CBSSiteStmts
		final List<CBSSiteStmt> siteStmtL = new ArrayList<CBSSiteStmt>();

		CBSSiteStmt cbsSiteStmt1 = createTestCBSSiteStmt(icn, station1NumLong, key, BooleanChar.Y, BooleanChar.Y);
		siteStmtL.add(cbsSiteStmt1);

		CBSSiteStmt cbsSiteStmt2 = createTestCBSSiteStmt(icn, station2NumLong, key, BooleanChar.N, BooleanChar.N);
		siteStmtL.add(cbsSiteStmt2);

		// Test Data
		final String lastName5 = cbsSiteStmt1.getSitePatient().getLastName().substring(0, 5);
		final String acntNumDisp = cbsSiteStmt1.getStationNum() + icnId + lastName5;
		final String lateStmtMsg = "N";
		final long batchRunId = 0L;
		final long replacedStmtId = 0L;
		final long messageId = 0L;

		// Initialize CBSStmt
		CBSStmt cbsStmt = new CBSStmt(icnId, statusId);

		cbsStmt.setAcntNumDisp(acntNumDisp);
		cbsStmt.setAmountDue(addMoney(cbsSiteStmt1.getAmountDue(), cbsSiteStmt2.getAmountDue(), 9));
		cbsStmt.setPrevBalance(addMoney(cbsSiteStmt1.getPrevBalance(), cbsSiteStmt2.getPrevBalance(), 9));
		cbsStmt.setTotalCharges(addMoney(cbsSiteStmt1.getTotalCharges(), cbsSiteStmt2.getTotalCharges(), 9));
		cbsStmt.setTotalCredits(addMoney(cbsSiteStmt1.getTotalCredits(), cbsSiteStmt2.getTotalCredits(), 9));
		cbsStmt.setNewBalance(addMoney(cbsSiteStmt1.getNewBalance(), cbsSiteStmt2.getNewBalance(), 9));
		cbsStmt.setLateStmtMsg(lateStmtMsg == "Y");
		cbsStmt.setStatementDate(cbsSiteStmt1.getStatementDate());
		cbsStmt.setProcessDate(cbsSiteStmt1.getProcessDate());
		cbsStmt.setBatchRunId(batchRunId);
		cbsStmt.setReplacedStmtId(replacedStmtId);
		cbsStmt.setMessageId(messageId);
		cbsStmt.setSiteStmtL(siteStmtL);

		return cbsStmt;
	}

	private static Money addMoney(Money money1, Money money2, final int digitsLength) {
		return new Money(money1.getDouble() + money2.getDouble(), digitsLength);
	}

	private static CBSSiteStmt createTestCBSSiteStmt(final String icn, final long stationNumLong, final int key,
			final BooleanChar isPrimary, final BooleanChar isPrimaryAddress) {
		// Test Data
		String stationNum = "" + stationNumLong;

		final long dfn = stationNumLong * 100000000000L + key;
		final String firstName = "TestFirst";
		final String lastName = "TESTLast";
		final String middleName = "TestMiddle";
		final String address1 = "TestAddress1";
		final String address2 = "TestAddress2";
		final String address3 = "TestAddress3";
		final String city = "TestCity";
		final String state = "OH";
		final String zipCode = "12345";
		final String country = "TestCountry";

		final String lastName5 = lastName.substring(0, 5);
		final String patientId = String.format("%09d", key);
		final String oldAcntNum = stationNum + patientId + lastName5;

		final String stationPhoneNum = "987654321";

		SimpleDateFormat format = new SimpleDateFormat("MMddyyyy");

		Date processDate = null;
		Date statementDate = null;
		Date lastBillPrepDate = null;
		Date datePosted = null;
		try {
			processDate = format.parse("05202016");
			statementDate = format.parse("05252016");
			lastBillPrepDate = format.parse("05252016");
			datePosted = format.parse("05012016");
		} catch (ParseException e) {
		}

		final Money amountDue = new Money("00000270{");
		final Money prevBalance = new Money("00000400{");
		final Money totalCharges = new Money("00000270{");
		final Money totalCredits = new Money("00000400}");
		final Money newBalance = new Money("00000270{");
		final String specialNotes = "Test Special Notes";
		final String noParaCdes = "010203";
		final BooleanChar largeFontInd = BooleanChar.N;
		final BooleanChar arAddressFlag = BooleanChar.Y;

		final List<CBSSiteTrans> siteTransL = new ArrayList<CBSSiteTrans>();

		// Initialize CBSSiteStmt
		CBSSiteStmt cbsSiteStmt = new CBSSiteStmt();

		cbsSiteStmt.setStationNum(stationNum);
		cbsSiteStmt.setStationPhoneNum(stationPhoneNum);
		cbsSiteStmt.setProcessDate(processDate);
		cbsSiteStmt.setStatementDate(statementDate);
		cbsSiteStmt.setAmountDue(amountDue);
		cbsSiteStmt.setPrevBalance(prevBalance);
		cbsSiteStmt.setTotalCharges(totalCharges);
		cbsSiteStmt.setTotalCredits(totalCredits);
		cbsSiteStmt.setNewBalance(newBalance);
		cbsSiteStmt.setSpecialNotes(specialNotes);
		cbsSiteStmt.setNoParaCdes(noParaCdes);
		cbsSiteStmt.setLargeFontInd(largeFontInd);
		cbsSiteStmt.setArAddressFlag(arAddressFlag);
		cbsSiteStmt.setLastBillPrepDate(lastBillPrepDate);
		cbsSiteStmt.setIsPrimary(isPrimary);
		cbsSiteStmt.setIsPrimaryAddress(isPrimaryAddress);
		cbsSiteStmt.setSiteTransL(siteTransL);

		// Initialize CBSSitePatient
		CBSSitePatient cbsSitePatient = new CBSSitePatient();
		cbsSiteStmt.setSitePatient(cbsSitePatient);

		cbsSitePatient.setIcn(icn);
		cbsSitePatient.setDfn(dfn);
		cbsSitePatient.setOldAcntNum(oldAcntNum);
		cbsSitePatient.setFirstName(firstName);
		cbsSitePatient.setLastName(lastName);
		cbsSitePatient.setMiddleName(middleName);
		cbsSitePatient.setAddress1(address1);
		cbsSitePatient.setAddress2(address2);
		cbsSitePatient.setAddress3(address3);
		cbsSitePatient.setCity(city);
		cbsSitePatient.setState(state);
		cbsSitePatient.setZipCode(zipCode);
		cbsSitePatient.setCountry(country);

		siteTransL.add(createTestCBSSiteTrans(datePosted, "PAYMENT", totalCredits, "REF_1", 1));
		siteTransL.add(createTestCBSSiteTrans(datePosted, "CHARGE", totalCharges, "REF_2", 2));

		return cbsSiteStmt;
	}

	private static CBSSiteTrans createTestCBSSiteTrans(Date datePosted, String transDesc, Money transAmount,
			String referenceNum, int orderNum) {

		CBSSiteTrans cbsSiteTrans = new CBSSiteTrans();

		cbsSiteTrans.setDatePosted(datePosted);
		cbsSiteTrans.setTransDesc(transDesc);
		cbsSiteTrans.setTransAmount(transAmount);
		cbsSiteTrans.setReferenceNum(referenceNum);
		cbsSiteTrans.setOrderNum(orderNum);

		return cbsSiteTrans;
	}

	/**
	 * Saves the given CBSStmt object to the database and updates the assigned
	 * ids.
	 * 
	 * @param cbsStmt
	 *            the CBSStmt object to save.
	 */
	public void saveCBSStmt(CBSStmt cbsStmt) {

		// Create the CBSStmt record
		cbsStmt.setId(cbsStmtDao.save(cbsStmt));

		final long cbsStmtId = cbsStmt.getId();
		final List<CBSSiteStmt> cbsSiteStmtL = cbsStmt.getSiteStmtL();

		if (cbsSiteStmtL == null)
			return;

		for (CBSSiteStmt cbsSiteStmt : cbsSiteStmtL) {
			cbsSiteStmt.setStmtId(cbsStmtId);

			// Create the CBSSiteStmt record
			cbsSiteStmt.setId(cbsSiteStmtDao.save(cbsSiteStmt, true));

			final long cbsSiteStmtId = cbsSiteStmt.getId();
			final List<CBSSiteTrans> cbsSiteTransL = cbsSiteStmt.getSiteTransL();

			final CBSSitePatient cbsSitePatient = cbsSiteStmt.getSitePatient();
			cbsSitePatient.setSiteStmtId(cbsSiteStmtId);

			// Create the CBSSitePatient record
			cbsSitePatient.setId(cbsSitePatientDao.save(cbsSitePatient));

			if (cbsSiteTransL == null)
				continue;

			for (CBSSiteTrans cbsSiteTrans : cbsSiteTransL) {
				cbsSiteTrans.setSiteStmtId(cbsSiteStmtId);

				// Create the CBSSiteTrans record
				cbsSiteTrans.setId(cbsSiteTransDao.save(cbsSiteTrans));
			}
		}

	}

	/**
	 * Update the messageId to the given value for the CBSStmt with the given id
	 * 
	 * @param cbsStmtId
	 *            Id of the CBSStmt record to update
	 * @param newMessageId
	 *            New messageId
	 * @return Number of rows updated.
	 */
	public int updateMessageIdForStmt(long cbsStmtId, long newMessageId) {
		final String sql = "UPDATE CBSStmt SET messageId=" + newMessageId + " WHERE id=" + cbsStmtId;
		return jdbcTemplate.update(sql);
	}

	/**
	 * Creates a copy of given CBSStmt object.
	 * 
	 * @param cbsStmt
	 *            CBSStmt object to copy.
	 * @return Copy of given CBSStmt object.
	 */
	CBSStmt copyCBSStmt(CBSStmt cbsStmt) {
		final CBSStmt newCBSStmt = new CBSStmt(cbsStmt.getAccountId());

		newCBSStmt.setId(cbsStmt.getId());
		newCBSStmt.setReplacedStmtId(cbsStmt.getReplacedStmtId());
		newCBSStmt.setMessageId(cbsStmt.getMessageId());
		newCBSStmt.setAcntNumDisp(cbsStmt.getAcntNumDisp());
		newCBSStmt.setAmountDue(cbsStmt.getAmountDue());
		newCBSStmt.setPrevBalance(cbsStmt.getPrevBalance());
		newCBSStmt.setTotalCharges(cbsStmt.getTotalCharges());
		newCBSStmt.setTotalCredits(cbsStmt.getTotalCredits());
		newCBSStmt.setNewBalance(cbsStmt.getNewBalance());
		newCBSStmt.setLateStmtMsg(cbsStmt.getLateStmtMsg());
		newCBSStmt.setStatementDate(cbsStmt.getStatementDate());
		newCBSStmt.setProcessDate(cbsStmt.getProcessDate());
		newCBSStmt.setBatchRunId(cbsStmt.getBatchRunId());
		newCBSStmt.setPrintAckId(cbsStmt.getPrintAckId());
		newCBSStmt.setPrintDate(cbsStmt.getPrintDate());
		newCBSStmt.setStatusId(cbsStmt.getStatusId());
		newCBSStmt.setCreatedBy(cbsStmt.getCreatedBy());
		newCBSStmt.setCreatedDate(cbsStmt.getCreatedDate());
		newCBSStmt.setModifiedBy(cbsStmt.getModifiedBy());
		newCBSStmt.setModifiedDate(cbsStmt.getModifiedDate());

		List<CBSSiteStmt> siteStmtL = cbsStmt.getSiteStmtL();
		if (siteStmtL != null) {
			List<CBSSiteStmt> newSiteStmtL = new ArrayList<CBSSiteStmt>();
			newCBSStmt.setSiteStmtL(newSiteStmtL);

			for (CBSSiteStmt cbsSiteStmt : siteStmtL) {
				newSiteStmtL.add(copyCBSSiteStmt(cbsSiteStmt));
			}
		}

		return newCBSStmt;
	}

	/**
	 * Creates a copy of given CBSSiteStmt object.
	 * 
	 * @param cbsSiteStmt
	 *            CBSSiteStmt object to copy.
	 * @return Copy of given CBSSiteStmt object.
	 */
	CBSSiteStmt copyCBSSiteStmt(CBSSiteStmt cbsSiteStmt) {
		final CBSSiteStmt newCbsSiteStmt = new CBSSiteStmt();

		newCbsSiteStmt.setId(cbsSiteStmt.getId());
		newCbsSiteStmt.setStmtId(cbsSiteStmt.getStmtId());
		newCbsSiteStmt.setStationNum(cbsSiteStmt.getStationNum());
		newCbsSiteStmt.setStationPhoneNum(cbsSiteStmt.getStationPhoneNum());
		newCbsSiteStmt.setProcessDate(cbsSiteStmt.getProcessDate());
		newCbsSiteStmt.setStatementDate(cbsSiteStmt.getStatementDate());
		newCbsSiteStmt.setAmountDue(cbsSiteStmt.getAmountDue());
		newCbsSiteStmt.setPrevBalance(cbsSiteStmt.getPrevBalance());
		newCbsSiteStmt.setTotalCharges(cbsSiteStmt.getTotalCharges());
		newCbsSiteStmt.setTotalCredits(cbsSiteStmt.getTotalCredits());
		newCbsSiteStmt.setNewBalance(cbsSiteStmt.getNewBalance());
		newCbsSiteStmt.setSpecialNotes(cbsSiteStmt.getSpecialNotes());
		newCbsSiteStmt.setNoParaCdes(cbsSiteStmt.getNoParaCdes());
		newCbsSiteStmt.setLargeFontInd(cbsSiteStmt.getLargeFontInd());
		newCbsSiteStmt.setArAddressFlag(cbsSiteStmt.getArAddressFlag());
		newCbsSiteStmt.setLastBillPrepDate(cbsSiteStmt.getLastBillPrepDate());
		newCbsSiteStmt.setIsPrimary(cbsSiteStmt.getIsPrimary());
		newCbsSiteStmt.setIsPrimaryAddress(cbsSiteStmt.getIsPrimaryAddress());
		newCbsSiteStmt.setCreatedBy(cbsSiteStmt.getCreatedBy());
		newCbsSiteStmt.setCreatedDate(cbsSiteStmt.getCreatedDate());
		newCbsSiteStmt.setModifiedBy(cbsSiteStmt.getModifiedBy());
		newCbsSiteStmt.setModifiedDate(cbsSiteStmt.getModifiedDate());

		newCbsSiteStmt.setSitePatient(copyCBSSitePatient(cbsSiteStmt.getSitePatient()));

		List<CBSSiteTrans> siteTransL = cbsSiteStmt.getSiteTransL();
		if (siteTransL != null) {
			List<CBSSiteTrans> newSiteTransL = new ArrayList<CBSSiteTrans>();
			newCbsSiteStmt.setSiteTransL(newSiteTransL);

			for (CBSSiteTrans cbsSiteTrans : siteTransL) {
				newSiteTransL.add(copyCBSSiteTrans(cbsSiteTrans));
			}
		}

		return newCbsSiteStmt;
	}

	/**
	 * Creates a copy of given CBSSitePatient object.
	 * 
	 * @param cbsSitePatient
	 *            CBSSitePatient object to copy.
	 * @return Copy of given CBSSitePatient object.
	 */
	CBSSitePatient copyCBSSitePatient(CBSSitePatient cbsSitePatient) {
		final CBSSitePatient newCbsSitePatient = new CBSSitePatient();

		newCbsSitePatient.setId(cbsSitePatient.getId());
		newCbsSitePatient.setSiteStmtId(cbsSitePatient.getSiteStmtId());
		newCbsSitePatient.setIcn(cbsSitePatient.getIcn());
		newCbsSitePatient.setDfn(cbsSitePatient.getDfn());
		newCbsSitePatient.setOldAcntNum(cbsSitePatient.getOldAcntNum());
		newCbsSitePatient.setFirstName(cbsSitePatient.getFirstName());
		newCbsSitePatient.setLastName(cbsSitePatient.getLastName());
		newCbsSitePatient.setMiddleName(cbsSitePatient.getMiddleName());
		newCbsSitePatient.setAddress1(cbsSitePatient.getAddress1());
		newCbsSitePatient.setAddress2(cbsSitePatient.getAddress2());
		newCbsSitePatient.setAddress3(cbsSitePatient.getAddress3());
		newCbsSitePatient.setCity(cbsSitePatient.getCity());
		newCbsSitePatient.setState(cbsSitePatient.getState());
		newCbsSitePatient.setZipCode(cbsSitePatient.getZipCode());
		newCbsSitePatient.setCountry(cbsSitePatient.getCountry());
		newCbsSitePatient.setCreatedBy(cbsSitePatient.getCreatedBy());
		newCbsSitePatient.setCreatedDate(cbsSitePatient.getCreatedDate());
		newCbsSitePatient.setModifiedBy(cbsSitePatient.getModifiedBy());
		newCbsSitePatient.setModifiedDate(cbsSitePatient.getModifiedDate());

		return newCbsSitePatient;
	}

	/**
	 * Creates a copy of given CBSSiteTrans object.
	 * 
	 * @param cbsSiteTrans
	 *            CBSSiteTrans object to copy.
	 * @return Copy of given CBSSiteTrans object.
	 */
	CBSSiteTrans copyCBSSiteTrans(CBSSiteTrans cbsSiteTrans) {
		final CBSSiteTrans newCBSSiteTrans = new CBSSiteTrans();

		newCBSSiteTrans.setId(cbsSiteTrans.getId());
		newCBSSiteTrans.setSiteStmtId(cbsSiteTrans.getSiteStmtId());
		newCBSSiteTrans.setDatePosted(cbsSiteTrans.getDatePosted());
		newCBSSiteTrans.setTransDesc(cbsSiteTrans.getTransDesc());
		newCBSSiteTrans.setTransAmount(cbsSiteTrans.getTransAmount());
		newCBSSiteTrans.setReferenceNum(cbsSiteTrans.getReferenceNum());
		newCBSSiteTrans.setOrderNum(cbsSiteTrans.getOrderNum());
		newCBSSiteTrans.setCreatedBy(cbsSiteTrans.getCreatedBy());
		newCBSSiteTrans.setCreatedDate(cbsSiteTrans.getCreatedDate());
		newCBSSiteTrans.setModifiedBy(cbsSiteTrans.getModifiedBy());
		newCBSSiteTrans.setModifiedDate(cbsSiteTrans.getModifiedDate());

		return newCBSSiteTrans;
	}

	/**
	 * Creates a copy of given PrintAckRec object.
	 * 
	 * @param printAckRec
	 *            PrintAckRec object to copy.
	 * @return Copy of given PrintAckRec object.
	 */
	public PrintAckRec copyPrintAckRec(PrintAckRec printAckRec) {
		final PrintAckRec newPrintAckRec = new PrintAckRec();

		newPrintAckRec.setId(printAckRec.getId());
		newPrintAckRec.setDateReceived(printAckRec.getDateReceived());
		newPrintAckRec.setStationNum(printAckRec.getStationNum());
		newPrintAckRec.setStatusId(printAckRec.getStatusId());
		newPrintAckRec.setFileName(printAckRec.getFileName());
		newPrintAckRec.setBatchRunId(printAckRec.getBatchRunId());
		newPrintAckRec.setCreatedBy(printAckRec.getCreatedBy());
		newPrintAckRec.setCreatedDate(printAckRec.getCreatedDate());
		newPrintAckRec.setModifiedBy(printAckRec.getModifiedBy());
		newPrintAckRec.setModifiedDate(printAckRec.getModifiedDate());

		final List<ADFileRecord> printAckDetailL = printAckRec.getPrintAckDetailL();
		if (printAckDetailL != null) {
			final List<ADFileRecord> newPrintAckDetailL = new ArrayList<ADFileRecord>(printAckDetailL.size());
			newPrintAckRec.setPrintAckDetailL(newPrintAckDetailL);

			for (ADFileRecord adFileRecord : printAckDetailL) {
				ADFileRecord newADFileRecord = null;

				if (adFileRecord != null) {
					newADFileRecord = new ADFileRecord();

					newADFileRecord.setType(adFileRecord.getType());
					newADFileRecord.setPatientAccount(adFileRecord.getPatientAccount());
				}
				newPrintAckDetailL.add(newADFileRecord);
			}
		}

		return newPrintAckRec;
	}

	/**
	 * Compares properties of two CBSMessage objects and asserts equals.
	 * 
	 * @param cbsMessage1
	 * @param cbsMessage2
	 */
	public void compareCBSMessages(CBSMessage cbsMessage1, CBSMessage cbsMessage2) {
		assertEquals(cbsMessage1.getId(), cbsMessage2.getId());
		assertEquals(cbsMessage1.getStatusId(), cbsMessage2.getStatusId());
		assertEquals(cbsMessage1.getBatchRunId(), cbsMessage2.getBatchRunId());
		assertEquals(cbsMessage1.getFileName(), cbsMessage2.getFileName());
		assertEquals(cbsMessage1.getCreatedBy(), cbsMessage2.getCreatedBy());
		assertEquals(cbsMessage1.getCreatedDate(), cbsMessage2.getCreatedDate());
		assertEquals(cbsMessage1.getModifiedBy(), cbsMessage2.getModifiedBy());
		assertEquals(cbsMessage1.getModifiedDate(), cbsMessage2.getModifiedDate());
	}

	/**
	 * Compares two Lists of CBSStmt objects. Order is significant.
	 * 
	 * @param cbsStmtL1
	 * @param cbsStmtL2
	 * @param deepCompare
	 *            Boolean indicating whether the compare of CBSStmt objects
	 *            should be a deep compare.
	 */
	public void compareCBSStmtLists(List<CBSStmt> cbsStmtL1, List<CBSStmt> cbsStmtL2, boolean deepCompare) {
		if (cbsStmtL1 == null)
			assertNull(cbsStmtL2);
		else {
			assertNotNull(cbsStmtL2);
			assertEquals(cbsStmtL1.size(), cbsStmtL2.size());

			int index = 0;
			for (CBSStmt cbsStmt1 : cbsStmtL1) {
				if (deepCompare)
					compareCBSStmtDeep(cbsStmt1, cbsStmtL2.get(index));
				else
					compareCBSStmt(cbsStmt1, cbsStmtL2.get(index));
				index++;
			}
		}
	}

	/**
	 * Compares properties of two CBSStmt objects and nested objects and asserts
	 * equals. Order of List properties is not required.
	 * 
	 * @param cbsStmt1
	 * @param cbsStmt2
	 */
	public void compareCBSStmtDeep(CBSStmt cbsStmt1, CBSStmt cbsStmt2) {
		compareCBSStmt(cbsStmt1, cbsStmt2);

		final List<CBSSiteStmt> cbsSiteStmtL1 = cbsStmt1.getSiteStmtL();
		final List<CBSSiteStmt> cbsSiteStmtL2 = cbsStmt2.getSiteStmtL();

		if (cbsSiteStmtL1 == null) {
			assertNull(cbsSiteStmtL2);
		} else {
			assertNotNull(cbsSiteStmtL2);
			assertEquals(cbsSiteStmtL1.size(), cbsSiteStmtL2.size());

			int numNulls2 = 0;
			final Map<Long, CBSSiteStmt> stmtM2 = new HashMap<Long, CBSSiteStmt>();
			for (CBSSiteStmt cbsSiteStmt : cbsSiteStmtL2) {
				if (cbsSiteStmt == null) {
					numNulls2++;
				} else {
					stmtM2.put(cbsSiteStmt.getId(), cbsSiteStmt);
				}
			}

			int numNulls1 = 0;
			for (CBSSiteStmt cbsSiteStmt1 : cbsSiteStmtL1) {
				if (cbsSiteStmt1 == null) {
					numNulls1++;
				} else {
					CBSSiteStmt cbsSiteStmt2 = stmtM2.get(cbsSiteStmt1.getId());
					assertNotNull(cbsSiteStmt2);
					compareCBSSiteStmtDeep(cbsSiteStmt1, cbsSiteStmt2);
				}
			}
			assertEquals(numNulls1, numNulls2);

		}
	}

	/**
	 * Compares properties of two CBSStmt objects and asserts equals.
	 * 
	 * @param cbsStmt1
	 * @param cbsStmt2
	 */
	public void compareCBSStmt(CBSStmt cbsStmt1, CBSStmt cbsStmt2) {
		if (cbsStmt1 == null)
			assertNull(cbsStmt2);
		else {
			assertNotNull(cbsStmt2);
			assertEquals(cbsStmt1.getId(), cbsStmt2.getId());
			assertEquals(cbsStmt1.getAccountId(), cbsStmt2.getAccountId());
			assertEquals(cbsStmt1.getReplacedStmtId(), cbsStmt2.getReplacedStmtId());
			assertEquals(cbsStmt1.getMessageId(), cbsStmt2.getMessageId());
			assertEquals(cbsStmt1.getAcntNumDisp(), cbsStmt2.getAcntNumDisp());
			assertEquals(cbsStmt1.getAmountDue(), cbsStmt2.getAmountDue());
			assertEquals(cbsStmt1.getPrevBalance(), cbsStmt2.getPrevBalance());
			assertEquals(cbsStmt1.getTotalCharges(), cbsStmt2.getTotalCharges());
			assertEquals(cbsStmt1.getTotalCredits(), cbsStmt2.getTotalCredits());
			assertEquals(cbsStmt1.getNewBalance(), cbsStmt2.getNewBalance());
			assertEquals(cbsStmt1.getLateStmtMsg(), cbsStmt2.getLateStmtMsg());
			assertEquals(cbsStmt1.getStatementDate(), cbsStmt2.getStatementDate());
			assertEquals(cbsStmt1.getProcessDate(), cbsStmt2.getProcessDate());
			assertEquals(cbsStmt1.getBatchRunId(), cbsStmt2.getBatchRunId());
			assertEquals(cbsStmt1.getPrintAckId(), cbsStmt2.getPrintAckId());
			assertEquals(cbsStmt1.getPrintDate(), cbsStmt2.getPrintDate());
			assertEquals(cbsStmt1.getStatusId(), cbsStmt2.getStatusId());
			assertEquals(cbsStmt1.getCreatedBy(), cbsStmt2.getCreatedBy());
			assertEquals(cbsStmt1.getCreatedDate(), cbsStmt2.getCreatedDate());
			assertEquals(cbsStmt1.getModifiedBy(), cbsStmt2.getModifiedBy());
			assertEquals(cbsStmt1.getModifiedDate(), cbsStmt2.getModifiedDate());
		}
	}

	/**
	 * Compares properties of two CBSSiteStmt objects and nested objects and
	 * asserts equals. Order of List properties is not required.
	 * 
	 * @param cbsSiteStmt1
	 * @param cbsSiteStmt2
	 */
	void compareCBSSiteStmtDeep(CBSSiteStmt cbsSiteStmt1, CBSSiteStmt cbsSiteStmt2) {
		compareCBSSiteStmt(cbsSiteStmt1, cbsSiteStmt2);

		CBSSitePatient cbsSitePatient1 = cbsSiteStmt1.getSitePatient();
		CBSSitePatient cbsSitePatient2 = cbsSiteStmt2.getSitePatient();
		if (cbsSitePatient1 == null) {
			assertNull(cbsSitePatient2);
		} else {
			assertNotNull(cbsSitePatient2);
			compareCBSSitePatients(cbsSitePatient1, cbsSitePatient2);
		}

		final List<CBSSiteTrans> cbsSiteTransL1 = cbsSiteStmt1.getSiteTransL();
		final List<CBSSiteTrans> cbsSiteTransL2 = cbsSiteStmt2.getSiteTransL();

		if (cbsSiteTransL1 == null) {
			assertNull(cbsSiteTransL2);
		} else {
			assertNotNull(cbsSiteTransL2);
			assertEquals(cbsSiteTransL1.size(), cbsSiteTransL2.size());

			int numNulls2 = 0;
			final Map<Long, CBSSiteTrans> transM2 = new HashMap<Long, CBSSiteTrans>();
			for (CBSSiteTrans cbsSiteTrans : cbsSiteTransL2) {
				if (cbsSiteTrans == null) {
					numNulls2++;
				} else {
					transM2.put(cbsSiteTrans.getId(), cbsSiteTrans);
				}
			}

			int numNulls1 = 0;
			for (CBSSiteTrans cbsSiteTrans1 : cbsSiteTransL1) {
				if (cbsSiteTrans1 == null) {
					numNulls1++;
				} else {
					CBSSiteTrans cbsSiteTrans2 = transM2.get(cbsSiteTrans1.getId());
					assertNotNull(cbsSiteTrans2);
					compareCBSSiteTrans(cbsSiteTrans1, cbsSiteTrans2);
				}
			}
			assertEquals(numNulls1, numNulls2);

		}
	}

	/**
	 * Compares properties of two CBSSiteStmt objects and asserts equals.
	 * 
	 * @param cbsSiteStmt1
	 * @param cbsSiteStmt2
	 */
	void compareCBSSiteStmt(CBSSiteStmt cbsSiteStmt1, CBSSiteStmt cbsSiteStmt2) {
		assertEquals(cbsSiteStmt1.getId(), cbsSiteStmt2.getId());
		assertEquals(cbsSiteStmt1.getStmtId(), cbsSiteStmt2.getStmtId());
		assertEquals(cbsSiteStmt1.getStationNum(), cbsSiteStmt2.getStationNum());
		assertEquals(cbsSiteStmt1.getStationPhoneNum(), cbsSiteStmt2.getStationPhoneNum());
		assertEquals(cbsSiteStmt1.getProcessDate(), cbsSiteStmt2.getProcessDate());
		assertEquals(cbsSiteStmt1.getStatementDate(), cbsSiteStmt2.getStatementDate());
		assertEquals(cbsSiteStmt1.getAmountDue(), cbsSiteStmt2.getAmountDue());
		assertEquals(cbsSiteStmt1.getPrevBalance(), cbsSiteStmt2.getPrevBalance());
		assertEquals(cbsSiteStmt1.getTotalCharges(), cbsSiteStmt2.getTotalCharges());
		assertEquals(cbsSiteStmt1.getTotalCredits(), cbsSiteStmt2.getTotalCredits());
		assertEquals(cbsSiteStmt1.getNewBalance(), cbsSiteStmt2.getNewBalance());
		assertEquals(cbsSiteStmt1.getSpecialNotes(), cbsSiteStmt2.getSpecialNotes());
		assertEquals(cbsSiteStmt1.getNoParaCdes(), cbsSiteStmt2.getNoParaCdes());
		assertEquals(cbsSiteStmt1.getLargeFontInd(), cbsSiteStmt2.getLargeFontInd());
		assertEquals(cbsSiteStmt1.getArAddressFlag(), cbsSiteStmt2.getArAddressFlag());
		assertEquals(cbsSiteStmt1.getLastBillPrepDate(), cbsSiteStmt2.getLastBillPrepDate());
		assertEquals(cbsSiteStmt1.getIsPrimary(), cbsSiteStmt2.getIsPrimary());
		assertEquals(cbsSiteStmt1.getIsPrimaryAddress(), cbsSiteStmt2.getIsPrimaryAddress());
		assertEquals(cbsSiteStmt1.getCreatedBy(), cbsSiteStmt2.getCreatedBy());
		assertEquals(cbsSiteStmt1.getCreatedDate(), cbsSiteStmt2.getCreatedDate());
		assertEquals(cbsSiteStmt1.getModifiedBy(), cbsSiteStmt2.getModifiedBy());
		assertEquals(cbsSiteStmt1.getModifiedDate(), cbsSiteStmt2.getModifiedDate());
	}

	/**
	 * Compares properties of two CBSSitePatient objects and asserts equals.
	 * 
	 * @param cbsSitePatient1
	 * @param cbsSitePatient2
	 */
	void compareCBSSitePatients(CBSSitePatient cbsSitePatient1, CBSSitePatient cbsSitePatient2) {
		assertEquals(cbsSitePatient1.getId(), cbsSitePatient2.getId());
		assertEquals(cbsSitePatient1.getSiteStmtId(), cbsSitePatient2.getSiteStmtId());
		assertEquals(cbsSitePatient1.getIcn(), cbsSitePatient2.getIcn());
		assertEquals(cbsSitePatient1.getDfn(), cbsSitePatient2.getDfn());
		assertEquals(cbsSitePatient1.getOldAcntNum(), cbsSitePatient2.getOldAcntNum());
		assertEquals(cbsSitePatient1.getFirstName(), cbsSitePatient2.getFirstName());
		assertEquals(cbsSitePatient1.getLastName(), cbsSitePatient2.getLastName());
		assertEquals(cbsSitePatient1.getMiddleName(), cbsSitePatient2.getMiddleName());
		assertEquals(cbsSitePatient1.getAddress1(), cbsSitePatient2.getAddress1());
		assertEquals(cbsSitePatient1.getAddress2(), cbsSitePatient2.getAddress2());
		assertEquals(cbsSitePatient1.getAddress3(), cbsSitePatient2.getAddress3());
		assertEquals(cbsSitePatient1.getCity(), cbsSitePatient2.getCity());
		assertEquals(cbsSitePatient1.getState(), cbsSitePatient2.getState());
		assertEquals(cbsSitePatient1.getZipCode(), cbsSitePatient2.getZipCode());
		assertEquals(cbsSitePatient1.getCountry(), cbsSitePatient2.getCountry());
		assertEquals(cbsSitePatient1.getCreatedBy(), cbsSitePatient2.getCreatedBy());
		assertEquals(cbsSitePatient1.getCreatedDate(), cbsSitePatient2.getCreatedDate());
		assertEquals(cbsSitePatient1.getModifiedBy(), cbsSitePatient2.getModifiedBy());
		assertEquals(cbsSitePatient1.getModifiedDate(), cbsSitePatient2.getModifiedDate());
	}

	/**
	 * Compares properties of two CBSSiteTrans objects and asserts equals.
	 * 
	 * @param cbsSiteTrans1
	 * @param cbsSiteTrans2
	 */
	void compareCBSSiteTrans(CBSSiteTrans cbsSiteTrans1, CBSSiteTrans cbsSiteTrans2) {

		assertEquals(cbsSiteTrans1.getId(), cbsSiteTrans2.getId());
		assertEquals(cbsSiteTrans1.getSiteStmtId(), cbsSiteTrans2.getSiteStmtId());
		assertEquals(cbsSiteTrans1.getDatePosted(), cbsSiteTrans2.getDatePosted());
		assertEquals(cbsSiteTrans1.getTransDesc(), cbsSiteTrans2.getTransDesc());
		assertEquals(cbsSiteTrans1.getTransAmount(), cbsSiteTrans2.getTransAmount());
		assertEquals(cbsSiteTrans1.getReferenceNum(), cbsSiteTrans2.getReferenceNum());
		assertEquals(cbsSiteTrans1.getOrderNum(), cbsSiteTrans2.getOrderNum());
		assertEquals(cbsSiteTrans1.getCreatedBy(), cbsSiteTrans2.getCreatedBy());
		assertEquals(cbsSiteTrans1.getCreatedDate(), cbsSiteTrans2.getCreatedDate());
		assertEquals(cbsSiteTrans1.getModifiedBy(), cbsSiteTrans2.getModifiedBy());
		assertEquals(cbsSiteTrans1.getModifiedDate(), cbsSiteTrans2.getModifiedDate());
	}

	/**
	 * Compares properties of two PrintAckRec objects and asserts equals.
	 * 
	 * @param printAckRec1
	 * @param cbsSiteTrans2
	 */
	public void comparePrintAckRec(PrintAckRec printAckRec1, PrintAckRec printAckRec2) {

		assertEquals(printAckRec1.getId(), printAckRec2.getId());
		assertEquals(printAckRec1.getDateReceived(), printAckRec2.getDateReceived());
		assertEquals(printAckRec1.getStationNum(), printAckRec2.getStationNum());
		assertEquals(printAckRec1.getStatusId(), printAckRec2.getStatusId());
		assertEquals(printAckRec1.getFileName(), printAckRec2.getFileName());
		assertEquals(printAckRec1.getBatchRunId(), printAckRec2.getBatchRunId());
		assertEquals(printAckRec1.getCreatedBy(), printAckRec2.getCreatedBy());
		assertEquals(printAckRec1.getCreatedDate(), printAckRec2.getCreatedDate());
		assertEquals(printAckRec1.getModifiedBy(), printAckRec2.getModifiedBy());
		assertEquals(printAckRec1.getModifiedDate(), printAckRec2.getModifiedDate());

		final List<ADFileRecord> printAckDetailL1 = printAckRec1.getPrintAckDetailL();
		final List<ADFileRecord> printAckDetailL2 = printAckRec2.getPrintAckDetailL();
		if (printAckDetailL1 == null) {
			assertNull(printAckDetailL2);
		} else {
			assertNotNull(printAckDetailL2);
			assertEquals(printAckDetailL1.size(), printAckDetailL2.size());

			int index = 0;
			for (ADFileRecord adFileRecord1 : printAckDetailL1) {
				final ADFileRecord adFileRecord2 = printAckDetailL2.get(index++);

				if (adFileRecord1 == null) {
					assertNull(adFileRecord2);
				} else {
					assertEquals(adFileRecord1.getType(), adFileRecord2.getType());
					assertEquals(adFileRecord1.getPatientAccount(), adFileRecord2.getPatientAccount());
				}
			}
		}
	}

	public static CBSStmt getStmtWithId(List<CBSStmt> cbsStmtL, long cbsStmtId) {
		CBSStmt foundStmt = null;
		for (CBSStmt cbsStmt : cbsStmtL) {
			if (cbsStmt.getId() == cbsStmtId) {
				assertNull(foundStmt); // Should only find 1
				foundStmt = cbsStmt;
			}
		}

		return foundStmt;
	}

	public static String cbsStmtListToString(List<CBSStmt> cbsStmtL, boolean deep) {
		final StringBuffer out = new StringBuffer();
		out.append("List<CBSStmt> : ");

		if (cbsStmtL == null) {
			out.append("null\n");
		} else {
			out.append(" size()=").append(cbsStmtL.size()).append(", {\n");

			String delim = "  ";

			for (CBSStmt cbsStmt : cbsStmtL) {
				out.append(delim);
				if (cbsStmt == null) {
					out.append("CBSStmt [ null ]");
				} else {
					out.append(cbsStmt);
				}
				if (deep) {
					appendSiteStmtString(out, cbsStmt);
				}
				delim = ",\n  ";
			}

			out.append("\n}\n");
		}

		return out.toString();
	}
	
	private static void appendSiteStmtString(StringBuffer out, CBSStmt cbsStmt) {
		final List<CBSSiteStmt> cbsSiteStmtL = cbsStmt.getSiteStmtL();
		if ((cbsSiteStmtL == null) || (cbsSiteStmtL.size() == 0)) {
			out.append("\n  - No CBSSiteStmts");
		} else {
			for (CBSSiteStmt cbsSiteStmt : cbsSiteStmtL) {
				if (cbsSiteStmt == null) {
					out.append("\n  - CBSSiteStmt : null");
				} else {
					out.append("\n  - ").append(cbsSiteStmt);
					
					final CBSSitePatient cbsSitePatient = cbsSiteStmt.getSitePatient();
					if (cbsSitePatient == null) {
						out.append("\n    - CBSSitePatient : null");
					} else {
						out.append("\n    - ").append(cbsSitePatient);
					}
					
					final List<CBSSiteTrans> cbsSiteTransL = cbsSiteStmt.getSiteTransL();
					if ((cbsSiteTransL == null) || (cbsSiteTransL.size() == 0)) {
						out.append("\n    - No CBSSiteTrans");
					} else {
						for (CBSSiteTrans cbsSiteTrans : cbsSiteTransL) {
							if (cbsSiteTrans == null) {
								out.append("\n    - CBSSiteTrans : null");
							} else {
								out.append("\n    - ").append(cbsSiteTrans);
							}
						}
					}
				}
			}
		}
	}

	public BatchRun createTestBatchRun() {
		final BatchRun batchRun = new BatchRun();
		batchRun.setJobId(batchJobDao.batchList().get(0).getId());
		batchRun.setStartDate(Timestamp.from(Instant.now()));
		batchRun.setEndDate(Timestamp.from(Instant.now()));
		batchRun.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.NEW));
		batchRun.setMessage("testmessage");
		return batchRun;
	}
	
	public PSReceived createTestPSReceived(final BatchRun batchRun) {
		final PSReceived psReceived = new PSReceived(batchRun.getId(), batchRun.getStatusId(), batchRun.getStartDate(), "psReceived.filename");
		psReceived.setNumOfSite(2);
		psReceived.setNumOfStatement(1);
		return psReceived;
	}
	
	public PSSite createTestPSSite(final PSReceived psReceived) {
		final PSSite psSite = new PSSite();
		psSite.setPsReceivedId(psReceived.getId());
		psSite.setSeqNum(1);
		psSite.setTotSeqNum(1);
		psSite.setFacilityNum("123");
		psSite.setFacilityPhoneNum("555-5555");
		psSite.setTotStatement(1);
		psSite.setStatementVal(new Money(12345, 10));
		psSite.setProcessDate(Date.from(Instant.now()));
		psSite.setStatementDate(Date.from(Instant.now()));
		return psSite;
	}
	
	public PSPatient createTestPSPatient(final PSSite psSite) {
		final PSPatient psPatient = new PSPatient();
		psPatient.setPsSite(psSite);
		psPatient.setPatientAccount("patientAccount");
		psPatient.setPatientLastName("pslastname");
		psPatient.setPatientFirstName("ps1stname");
		psPatient.setPatientMiddleName("psmidname");
		psPatient.setAddress1("add1");
		psPatient.setAddress2("add2");
		psPatient.setAddress3("add3");
		psPatient.setCity("chandigarh");
		psPatient.setState("pj");
		psPatient.setZipCode("122222");
		psPatient.setCountryName("india");
		psPatient.setAmountDue(new Money(2222, 10));
		psPatient.setPrevBalance(new Money(11111, 10));
		psPatient.setTotalCharges(new Money(1234, 10));
		psPatient.setTotalCredits(new Money(22222, 10));
		psPatient.setNewBalance(new Money(1738, 10));
		psPatient.setSpecialNotes("special notes");
		psPatient.setSpecialNotesForLateSites1("last 1 specialnotes");
		psPatient.setSpecialNotesForLateSites2("last 2 special notes");
		psPatient.setRightsObligationParagraphCodes("paracode");
		psPatient.setNumOfLines(10);
		psPatient.setDfnNumber(12345678);
		psPatient.setLargeFontInd(BooleanChar.from(true));
		psPatient.setIcnNumber("testicn12345");
		psPatient.setAddressFlag(BooleanChar.from(true));
		psPatient.setLastBillPrepDate(Date.from(Instant.now()));
		return psPatient;
	}
	
	public PSDetails createTestPSDetails(final PSPatient psPatient) {
		final PSDetails psDetails = new PSDetails();
		psDetails.setPsPatientId(psPatient.getId());
		psDetails.setDatePosted(Date.from(Instant.now()));
		psDetails.setTransDesc("transdesc");
		psDetails.setTransAmount(new Money(1923, 10));
		psDetails.setReferenceNum("333333");
		psDetails.setIcnNumber("testIcnAgain");
		psDetails.setSeqNum(2);
		return psDetails;
	}
	
	public long savePSDetails(final PSDetails psDetails) {

		KeyHolder keyHolder = new GeneratedKeyHolder();
		jdbcTemplate.update(new PreparedStatementCreator(){

			@Override
			public PreparedStatement createPreparedStatement(Connection con) throws SQLException {
				// For reference: (3/22/2017)
				// "INSERT INTO PSDetails(psPatient_Id, datePosted, transDesc, transAmount, transAmountRaw, referenceNum, icnNumber, seqNum) 
				// values(?, ?, ?, ?, ?, ?, ?, ?)";
				final PreparedStatement ps = con.prepareStatement(PSDetailsDAOImpl.getInsertPSDetailsSql(), new String[]{"id"});
				ps.setLong(1, psDetails.getPsPatientId());
				ps.setDate(2, new java.sql.Date(psDetails.getDatePosted().getTime()));
				ps.setString(3, psDetails.getTransDesc());
				ps.setLong(4, psDetails.getTransAmount().getLong());
				ps.setString(5, psDetails.getTransAmount().getCobol());
				ps.setString(6, psDetails.getReferenceNum());
				ps.setString(7, psDetails.getIcnNumber());
				ps.setLong(8, psDetails.getSeqNum());
				return ps;
			}
			
		}, keyHolder);
		
		return keyHolder.getKey().longValue();
	}
}
