package gov.va.cpss.dao.impl;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.time.Instant;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import org.junit.Test;

import gov.va.cpss.model.AITCDollar;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.apps.APSSiteStmt;
import gov.va.cpss.model.apps.APSStmt;
import gov.va.cpss.model.cbs.CBSAccount;

/**
 * APSSiteStmtDAOImplTest class
 * Unit test for APSSiteStmtDAOImpl
 *
 */
public final class APSSiteStmtDAOImplTest extends AbstractApsDaoImplTest {

	@Test
	public final void testGetExistingSiteStatements() {
		final Long[] apsSiteStmtStatementIds = new Long[testApsSiteStmtL.size()];
		for(int idx = 0; idx < apsSiteStmtStatementIds.length; idx++) {
			apsSiteStmtStatementIds[idx] = testApsSiteStmtL.get(idx).getStatementId();
		}
		
		final List<APSSiteStmt> apsSiteStmtRetrievedL = apsSiteStmtDao.getExistingSiteStatements(apsSiteStmtStatementIds);
		
		assertNotNull(apsSiteStmtRetrievedL);
		assertTrue(apsSiteStmtRetrievedL.stream().map(APSSiteStmt::getId).collect(Collectors.toList())
				.containsAll(testApsSiteStmtL.stream().map(APSSiteStmt::getId).collect(Collectors.toList())));
	}
	
	@Test
	public final void testGetStmtSitesForStmt() {
		final APSSiteStmt apsSiteStmtExpected = testApsSiteStmtL.get(0);
		
		final List<APSSiteStmt> apsSiteStmtRetrievedL = apsSiteStmtDao.getStmtSitesForStmt(apsSiteStmtExpected.getStatementId());
		
		assertTrue(apsSiteStmtRetrievedL.stream().map(APSSiteStmt::getId).collect(Collectors.toList())
				.contains(apsSiteStmtExpected.getId()));
	}
	
	@Test
	public final void testBatchUpdate() {
		testApsSiteStmtL.forEach(apsSiteStmt -> {
			apsSiteStmt.setPrimary(false);
			apsSiteStmt.setPrimaryAddress(false);
		});
		
		final int[] batchUpdate = apsSiteStmtDao.batchUpdate(testApsSiteStmtL);
		assertEquals(testApsSiteStmtL.size(), batchUpdate.length);
		
		testApsSiteStmtL.forEach(apsSiteStmt -> {
			final APSSiteStmt retrieved = (APSSiteStmt)apsSiteStmtDao.select("WHERE ID = " + apsSiteStmt.getId(), "");
			assertNotNull(retrieved);
			assertEquals(apsSiteStmt.isPrimary(), retrieved.isPrimary());
			assertEquals(apsSiteStmt.isPrimaryAddress(), retrieved.isPrimaryAddress());
		});
	}
	
	@Override
	protected void insertTestData() {
		// Create CBSAccount
		final String icn1 = "icnForJUnitTest1";
		final CBSAccount cbsAccount = new CBSAccount();
		cbsAccount.setIcn(icn1);
		cbsAccount.setId(cbsAccountDao.save(icn1));
		testCbsAccountL.add(cbsAccount);
		
		// Create APSStmt
		final APSStmt apsStmt1 = new APSStmt();
		apsStmt1.setAccountId(cbsAccount.getId());
		apsStmt1.setStatementDate(Date.from(Instant.now()));
		apsStmt1.setTotalAmountReceived(new AITCDollar(19999));
		apsStmt1.setProcessDate(Date.from(Instant.now()));
		apsStmt1.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.NEW));
		apsStmt1.setReplacedStmtId(123);
		apsStmt1.setPrintDate(Date.from(Instant.now()));
		apsStmt1.setId(apsStmtDao.save(apsStmt1));
		testApsStmtL.add(apsStmt1);	
		
		final APSStmt apsStmt2 = new APSStmt();
		apsStmt2.setAccountId(cbsAccount.getId());
		apsStmt2.setStatementDate(Date.from(Instant.now()));
		apsStmt2.setTotalAmountReceived(new AITCDollar(19999));
		apsStmt2.setProcessDate(Date.from(Instant.now()));
		apsStmt2.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.NEW));
		apsStmt2.setReplacedStmtId(123);
		apsStmt2.setPrintDate(Date.from(Instant.now()));
		apsStmt2.setId(apsStmtDao.save(apsStmt2));
		testApsStmtL.add(apsStmt2);	
		
		// Create APSSiteStmt
		final APSSiteStmt apsSiteStmt1 = new APSSiteStmt();
		apsSiteStmt1.setStatementId(apsStmt1.getId());
		apsSiteStmt1.setFacilityNum("123");
		apsSiteStmt1.setFacilityPhoneNum("555-5555");
		apsSiteStmt1.setStatementDate(Date.from(Instant.now()));
		apsSiteStmt1.setProcessDate(Date.from(Instant.now()));
		apsSiteStmt1.setTotalAmountReceived(new AITCDollar(19282));
		apsSiteStmt1.setArAddress(true);
		apsSiteStmt1.setLastBillPrepDate(Date.from(Instant.now()));
		apsSiteStmt1.setPrimary(true);
		apsSiteStmt1.setPrimaryAddress(true);
		apsSiteStmt1.setId(apsSiteStmtDao.save(apsSiteStmt1));
		testApsSiteStmtL.add(apsSiteStmt1);
		
		final APSSiteStmt apsSiteStmt2 = new APSSiteStmt();
		apsSiteStmt2.setStatementId(apsStmt2.getId());
		apsSiteStmt2.setFacilityNum("123");
		apsSiteStmt2.setFacilityPhoneNum("511-1251");
		apsSiteStmt2.setStatementDate(Date.from(Instant.now()));
		apsSiteStmt2.setProcessDate(Date.from(Instant.now()));
		apsSiteStmt2.setTotalAmountReceived(new AITCDollar(19282));
		apsSiteStmt2.setArAddress(true);
		apsSiteStmt2.setLastBillPrepDate(Date.from(Instant.now()));
		apsSiteStmt2.setPrimary(true);
		apsSiteStmt2.setPrimaryAddress(true);
		apsSiteStmt2.setId(apsSiteStmtDao.save(apsSiteStmt2));
		testApsSiteStmtL.add(apsSiteStmt2);
	}
	
	private static void assertAPSSiteStmtEquals(final APSSiteStmt expected, final APSSiteStmt retrieved) {
		assertEquals(expected.getId(), retrieved.getId());
		assertEquals(expected.getAccountId(), retrieved.getAccountId());
		assertEquals(expected.getFacilityNum(), retrieved.getFacilityNum());
		assertEquals(expected.getFacilityPhoneNum(), retrieved.getFacilityPhoneNum());
		assertEquals(expected.getLastBillPrepDate(), retrieved.getLastBillPrepDate());
		assertEquals(expected.getPrintDate().getTime(), retrieved.getPrintDate().getTime(), 1000);
		assertEquals(expected.getProcessDate().getTime(), retrieved.getProcessDate().getTime(), 1000);
		assertEquals(expected.getReplacedStmtId(), retrieved.getReplacedStmtId());
		assertEquals(expected.getStatementDate().getTime(), retrieved.getStatementDate().getTime(), 1000);
		assertEquals(expected.getStatusId(), retrieved.getStatusId());
		assertEquals(expected.getTotalAmountReceived().getDouble(), retrieved.getTotalAmountReceived().getDouble(), 0);
		//TODO: getPatient
	}
}
