package gov.va.cpss.dao.impl;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.Month;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;

import javax.sql.DataSource;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.PreparedStatementCreator;
import org.springframework.jdbc.support.GeneratedKeyHolder;
import org.springframework.jdbc.support.KeyHolder;

import gov.va.cpss.dao.APSPaymentDAO;
import gov.va.cpss.dao.APSSitePatientDAO;
import gov.va.cpss.dao.APSSiteStmtDAO;
import gov.va.cpss.dao.APSStmtDAO;
import gov.va.cpss.dao.BatchJobDAO;
import gov.va.cpss.dao.BatchRunDAO;
import gov.va.cpss.dao.BatchRunProcessDAO;
import gov.va.cpss.dao.CBSAccountDAO;
import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.model.AITCDollar;
import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.BatchRunProcess;
import gov.va.cpss.model.BooleanChar;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.apps.APSDetails;
import gov.va.cpss.model.apps.APSPatient;
import gov.va.cpss.model.apps.APSPayment;
import gov.va.cpss.model.apps.APSReceived;
import gov.va.cpss.model.apps.APSSite;
import gov.va.cpss.model.apps.APSSitePatient;
import gov.va.cpss.model.apps.APSSiteStmt;
import gov.va.cpss.model.apps.APSStmt;
import gov.va.cpss.model.cbs.CBSAccount;

/** 
 * APPSDAOImplUtils class
 * This class is used as utility class and used for testing cbss account
 *
 */
public class APPSDAOImplUtils {

	@Autowired
	private APSStmtDAO apsStmtDao;

	@Autowired
	private APSSiteStmtDAO apsSiteStmtDao;

	@Autowired
	private APSSitePatientDAO apsSitePatientDao;

	@Autowired
	private APSPaymentDAO apsPaymentDao;

	@Autowired
	private CBSAccountDAO cbsAccountDao;

	@Autowired
	private BatchRunProcessDAO batchRunProcessDao;

	@Autowired
	private BatchRunDAO batchRunDao;

	@Autowired
	private BatchJobDAO batchJobDao;

	@Autowired
	private ProcessStatusDAO processStatusDao;

	private JdbcTemplate jdbcTemplate;

	@Autowired
	public void setDataSource(DataSource dataSource) {
		jdbcTemplate = new JdbcTemplate(dataSource);
	}

	public void saveAPSStmt(APSStmt apsStmt) {

		apsStmt.setId(apsStmtDao.save(apsStmt));

		final long apsStmtId = apsStmt.getId();
		final List<APSSiteStmt> apsSiteStmtL = apsStmt.getSiteStmts();

		if (apsSiteStmtL == null) {
			return;
		}

		for (APSSiteStmt apsSiteStmt : apsSiteStmtL) {
			apsSiteStmt.setStatementId(apsStmtId);

			apsSiteStmt.setId(apsSiteStmtDao.save(apsSiteStmt));

			final long apsSiteStmtId = apsSiteStmt.getId();
			final List<APSPayment> appsPaymentL = apsSiteStmt.getPayments();

			final APSSitePatient apsSitePatient = apsSiteStmt.getPatient();
			apsSitePatient.setSiteStmtId(apsSiteStmtId);

			apsSitePatient.setId(apsSitePatientDao.save(apsSitePatient));

			if (appsPaymentL == null) {
				continue;
			}

			for (APSPayment apsPayment : appsPaymentL) {
				apsPayment.setSiteStmtId(apsSiteStmtId);

				apsPayment.setId(apsPaymentDao.save(apsPayment));
			}
		}

	}

	public long insertTestCbsAccount(final String icn) {
		final String sql = "INSERT INTO CBSAccount (icn) values (?)";
		KeyHolder keyHolder = new GeneratedKeyHolder();

		jdbcTemplate.update(sql, new PreparedStatementCreator() {

			@Override
			public PreparedStatement createPreparedStatement(Connection conn) throws SQLException {
				PreparedStatement ps = conn.prepareStatement(sql, new String[] { "id" });
				ps.setString(1, icn);
				return ps;
			}

		}, keyHolder);

		return keyHolder.getKey().longValue();
	}

	public void deleteTestCbsAccount(final String icn) {
		final String sql = "DELETE FROM CBSAccount where icn='" + icn + "'";

		jdbcTemplate.update(sql);
	}

	public APSStmt createSimpleTestAPSStmt(final int key, final int statusid, final long stationNumLong) {
		final String icn = getTestIcn(key);

		final long accountId = cbsAccountDao.getByICN(icn);

		return createTestAPSStmtObject(key, accountId, statusid, icn, stationNumLong, 2016);
	}

	public APSStmt createDoubleTestAPSStmt(final int key, final int statusId, final long stationNum1,
			final long stationNum2) {
		final String icn = getTestIcn(key);

		final long accountId = cbsAccountDao.getByICN(icn);

		return createTestDoubleAPSStmtObject(accountId, icn, key, statusId, stationNum1, stationNum2);
	}

	public static BatchRunProcess createTestBatchRunProcessObject(final long batchRunId, final String fileName,
			final Date processDate, final String other) {
		final BatchRunProcess batchRunProcess = new BatchRunProcess();

		batchRunProcess.setBatchRunId(batchRunId);
		batchRunProcess.setFileName(fileName);
		batchRunProcess.setProcessDate(processDate);
		batchRunProcess.setOther(other);

		return batchRunProcess;
	}

	private static APSStmt createTestAPSStmtObject(final int key, final long accountId, final int statusId,
			final String icn, final long stationNumLong, final int stmtYear) {

		// Initialize APSSiteStmt
		final List<APSSiteStmt> siteStmtL = new ArrayList<APSSiteStmt>();
		APSSiteStmt apsSiteStmt = createTestAPSSiteStmt(icn, stationNumLong, key, BooleanChar.Y, BooleanChar.Y);
		// TODO:
		// apsSiteStmt.setStmtYear(stmtYear);
		siteStmtL.add(apsSiteStmt);

		// Test data
		final String lastName5 = apsSiteStmt.getPatient().getLastName().substring(0, 5);
		final String acntNumDisp = apsSiteStmt.getFacilityNum() + accountId + lastName5;
		final long batchRunId = 0L;
		final long replacedStmtId = 0L;
		final long messageId = 0L;

		// Initialize APSStmt
		APSStmt apsStmt = new APSStmt(accountId, statusId);

		// TODO: stmtyear
		// apsStmt.setStmtYear(stmtYear);
		apsStmt.setStatementDate(getPreviousYear()); // TODO: find out if this
														// is deprecated
		apsStmt.setTotalAmountReceived(apsSiteStmt.getTotalAmountReceived());
		apsStmt.setProcessDate(apsSiteStmt.getProcessDate());
		apsStmt.setAccountNumDisp(acntNumDisp);
		apsStmt.setPrintDate(apsSiteStmt.getPrintDate());
		apsStmt.setStatusId(statusId);
		apsStmt.setSiteStmts(siteStmtL);

		return apsStmt;
	}

	private static APSStmt createTestDoubleAPSStmtObject(final long icnId, final String icn, final int key,
			final int statusId, final long station1NumLong, final long station2NumLong) {
		// Build APSSiteStmts
		final List<APSSiteStmt> apsSiteStmtL = new ArrayList<APSSiteStmt>();

		final APSSiteStmt apsSiteStmt1 = createTestAPSSiteStmt(icn, station1NumLong, key, BooleanChar.Y, BooleanChar.Y);
		apsSiteStmtL.add(apsSiteStmt1);

		final APSSiteStmt apsSiteStmt2 = createTestAPSSiteStmt(icn, station2NumLong, key, BooleanChar.N, BooleanChar.N);
		apsSiteStmtL.add(apsSiteStmt2);

		// Test Data
		final String lastName5 = apsSiteStmt1.getPatient().getLastName().substring(0, 5);
		final String accountNumDisp = apsSiteStmt1.getFacilityNum() + icnId + lastName5;
		final long batchRunProcessId = 0L;
		final long replacedStmtId = 0L;

		// Initialize CBSStmt
		APSStmt apsStmt = new APSStmt(icnId, statusId);

		apsStmt.setAccountNumDisp(accountNumDisp);
		apsStmt.setTotalAmountReceived(new AITCDollar(
				apsSiteStmt1.getTotalAmountReceived().getDouble() + apsSiteStmt2.getTotalAmountReceived().getDouble()));
		apsStmt.setStatementDate(apsSiteStmt1.getStatementDate());
		apsStmt.setProcessDate(apsSiteStmt1.getProcessDate());
		apsStmt.setAckBatchRunId(batchRunProcessId);
		apsStmt.setGenBatchRunId(batchRunProcessId);
		apsStmt.setSendBatchRunId(batchRunProcessId);
		apsStmt.setReplacedStmtId(replacedStmtId);
		apsStmt.setSiteStmts(apsSiteStmtL);

		return apsStmt;
	}

	private static APSSiteStmt createTestAPSSiteStmt(final String icn, final long stationNumLong, final int key,
			final BooleanChar isPrimary, final BooleanChar isPrimaryAddress) {
		// Test data
		String stationNum = Long.toString(stationNumLong);

		final long dfn = stationNumLong * 100000000000L + key;
		final String firstName = "TestFirst";
		final String lastName = "TESTLast";
		final String middleName = "TestMiddle";
		final String address1 = "TestAddress1";
		final String address2 = "TestAddress2";
		final String address3 = "TestAddress3";
		final String city = "TestCity";
		final String state = "TT";
		final String zipCode = "12345";
		final String country = "TestCountry";

		final String lastName5 = lastName.substring(0, 5);
		final String patientId = String.format("%09d", key);
		final String oldAcntNum = stationNum + patientId + lastName5;

		final String stationPhoneNum = "987654321";
		final int stmtYear = 2016;

		SimpleDateFormat format = new SimpleDateFormat("MMddyyyy");

		Date processDate = null;
		Date lastBillPrepDate = null;
		Date datePosted = null;
		try {
			processDate = format.parse("05202016");
			lastBillPrepDate = format.parse("05252016");
			datePosted = format.parse("05122016");
		} catch (ParseException e) {
		}

		final AITCDollar totalAmountReceived = new AITCDollar(2700);
		final BooleanChar arAddressFlag = BooleanChar.Y;

		final List<APSPayment> appsPaymentL = new ArrayList<APSPayment>();

		// Initialize APSSiteStmt
		APSSiteStmt apsSiteStmt = new APSSiteStmt();

		apsSiteStmt.setFacilityNum(stationNum);
		apsSiteStmt.setFacilityPhoneNum(stationPhoneNum);
		// TODO: apsSiteStmt.setStmtYear(stmtYear);
		apsSiteStmt.setProcessDate(processDate);
		apsSiteStmt.setTotalAmountReceived(totalAmountReceived);
		apsSiteStmt.setArAddress(arAddressFlag.isTrue());
		apsSiteStmt.setLastBillPrepDate(lastBillPrepDate);
		apsSiteStmt.setPrimary(isPrimary.isTrue());
		apsSiteStmt.setPrimaryAddress(isPrimaryAddress.isTrue());
		apsSiteStmt.setPayments(appsPaymentL);

		// Initialize APSSitePatient
		APSSitePatient apsSitePatient = new APSSitePatient();
		apsSiteStmt.setPatient(apsSitePatient);

		apsSitePatient.setIcn(icn);
		apsSitePatient.setDfn(dfn);
		apsSitePatient.setAccountNumber(oldAcntNum);
		apsSitePatient.setFirstName(firstName);
		apsSitePatient.setLastName(lastName);
		apsSitePatient.setMiddleName(middleName);
		apsSitePatient.setAddress1(address1);
		apsSitePatient.setAddress2(address2);
		apsSitePatient.setAddress3(address3);
		apsSitePatient.setCity(city);
		apsSitePatient.setState(state);
		apsSitePatient.setZipCode(zipCode);
		apsSitePatient.setCountry(country);

		appsPaymentL.add(createTestAPSPayment(datePosted, new AITCDollar(100), Month.MAY));
		appsPaymentL.add(createTestAPSPayment(datePosted, new AITCDollar(300), Month.JANUARY));

		return apsSiteStmt;
	}

	private static APSPayment createTestAPSPayment(final Date datePosted, final AITCDollar transactionAmount,
			final Month month) {

		final APSPayment apsPayment = new APSPayment(0L);

		apsPayment.setDatePosted(datePosted);
		apsPayment.setTransactionAmount(transactionAmount);
		apsPayment.setMonth(month);

		return apsPayment;
	}

	/**
	 * Creates a simple APSStmt for use in unit testing. None of the batch run
	 * process fields are initialized.
	 * 
	 * @param cbsAccount
	 * @return test APSStmt
	 */
	public APSStmt createTestAPSStmt(final CBSAccount cbsAccount) {
		final APSStmt apsStmt1 = new APSStmt();
		apsStmt1.setAccountId(cbsAccount.getId());
		apsStmt1.setStatementDate(Date.from(Instant.now()));
		apsStmt1.setTotalAmountReceived(new AITCDollar(19999));
		apsStmt1.setProcessDate(Date.from(Instant.now()));
		apsStmt1.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.NEW));
		apsStmt1.setReplacedStmtId(123);
		apsStmt1.setPrintDate(Date.from(Instant.now()));
		return apsStmt1;
	}

	public APSSiteStmt createTestAPSSiteStmt(final APSStmt apsStmt) {
		final APSSiteStmt apsSiteStmt1 = new APSSiteStmt();
		apsSiteStmt1.setStatementId(apsStmt.getId());
		apsSiteStmt1.setFacilityNum("123");
		apsSiteStmt1.setFacilityPhoneNum("555-5555");
		apsSiteStmt1.setStatementDate(Date.from(Instant.now()));
		apsSiteStmt1.setProcessDate(Date.from(Instant.now()));
		apsSiteStmt1.setTotalAmountReceived(new AITCDollar(19282));
		apsSiteStmt1.setArAddress(true);
		apsSiteStmt1.setLastBillPrepDate(Date.from(Instant.now()));
		apsSiteStmt1.setPrimary(true);
		apsSiteStmt1.setPrimaryAddress(true);
		return apsSiteStmt1;
	}

	public APSReceived createTestAPSReceived() {
		final int batchRunId = 0;
		final int statusId = processStatusDao.getStatusFromEnum(ProcessStatus.Status.NEW);
		final Date dateReceived = Timestamp.from(Instant.now());
		final String filename = "testFilename.txt";
		final int numOfSite = 1;
		final int numOfPatient = 1;

		final APSReceived apsReceived = new APSReceived(batchRunId, statusId, dateReceived, filename);
		apsReceived.setNumOfSite(numOfSite);
		apsReceived.setNumOfPatient(numOfPatient);
		return apsReceived;
	}

	public APSSite createTestApsSite(final APSReceived apsReceived) {
		final APSSite apsSite = new APSSite();
		apsSite.setApsReceivedId(apsReceived.getId());
		apsSite.setSeqNum(1);
		apsSite.setTotSeqNum(1);
		apsSite.setFacilityNum("1");
		apsSite.setFacilityPhoneNum("111-111-1111");
		apsSite.setTotalPatient(1);
		apsSite.setStatementDate(Timestamp.from(Instant.EPOCH.plusSeconds(1000000)));
		apsSite.setProcessDate(Timestamp.from(Instant.now()));
		return apsSite;
	}

	public APSPatient createTestApsPatient(final APSSite apsSite) {
		final APSPatient apsPatient = new APSPatient();
		apsPatient.setApsSite(apsSite);
		apsPatient.setIcn("testicn54321");
		apsPatient.setDfn(19181716);
		apsPatient.setAccountNumber("oldaccntnum12");
		apsPatient.setFirstName("testfirst");
		apsPatient.setLastName("testlast");
		apsPatient.setMiddleName("testmid");
		apsPatient.setAddress1("add1");
		apsPatient.setAddress2("add2");
		apsPatient.setAddress3("add3");
		apsPatient.setCity("orlaaandooo");
		apsPatient.setState("fl");
		apsPatient.setZipCode("00000");
		apsPatient.setCountry("usa");
		apsPatient.setTotalAmountReceived(new AITCDollar(10000));
		apsPatient.setArAddress(true);
		apsPatient.setLastBillPrepDate(Date.from(Instant.now()));
		apsPatient.setNumOfPD(1);
		return apsPatient;
	}

	public APSDetails createTestApsDetails(final APSPatient apsPatient) {
		final APSDetails apsDetails = new APSDetails();
		apsDetails.setApsPatientId(apsPatient.getId());
		apsDetails.setDatePosted(Date.from(Instant.now()));
		apsDetails.setTransactionAmount(new AITCDollar(2222));
		return apsDetails;
	}

	public APSSitePatient createTestAPSSitePatient(final APSSiteStmt apsSiteStmt) {
		final APSSitePatient apsSitePatient = new APSSitePatient();
		apsSitePatient.setSiteStmtId(apsSiteStmt.getId());
		apsSitePatient.setIcn("apsSitePatient");
		apsSitePatient.setDfn(1010101);
		apsSitePatient.setAccountNumber("oldaccountnum");
		apsSitePatient.setFirstName("firstname");
		apsSitePatient.setMiddleName("middley");
		apsSitePatient.setLastName("lastly");
		apsSitePatient.setAddress1("addre1");
		apsSitePatient.setAddress2("addr2");
		apsSitePatient.setAddress3("adres3");
		apsSitePatient.setCity("chicago");
		apsSitePatient.setState("IL");
		apsSitePatient.setZipCode("12345");
		apsSitePatient.setCountry("usa");
		return apsSitePatient;
	}

	public APSPayment createTestAPSPayment(final APSSiteStmt apsSiteStmt) {
		final APSPayment apsPayment = new APSPayment();
		apsPayment.setSiteStmtId(apsSiteStmt.getId());
		apsPayment.setMonth(getMonth(apsSiteStmt.getStatementDate()));
		apsPayment.setDatePosted(Date.from(Instant.now()));
		apsPayment.setTransactionAmount(new AITCDollar(919191));
		apsPayment.setTransactionAmountStr(apsPayment.getTransactionAmount().toString().replace("+", ""));
		return apsPayment;
	}

	public BatchRun createTestBatchRun() {
		final int jobId = batchJobDao.batchList().get(0).getId();
		final Timestamp startDate = Timestamp.from(Instant.now());
		final Timestamp endDate = Timestamp.from(Instant.now().plusSeconds(100000));
		final int statusId = 2;
		final String message = "Test message.:;<>\"'&*()\\/$%@!";

		final BatchRun batchRun = new BatchRun();
		batchRun.setJobId(jobId);
		batchRun.setStartDate(startDate);
		batchRun.setEndDate(endDate);
		batchRun.setStatusId(statusId);
		batchRun.setMessage(message);

		return batchRun;
	}

	public BatchRunProcess createTestBatchRunProcess(final BatchRun batchRun) {
		final BatchRunProcess batchRunProcess = new BatchRunProcess();
		batchRunProcess.setBatchRunId(batchRun.getId());
		batchRunProcess.setFileName("genfilename.test");
		batchRunProcess.setProcessDate(Date.from(Instant.now()));
		batchRunProcess.setStatusId(processStatusDao.getStatusFromEnum(ProcessStatus.Status.NEW));
		return batchRunProcess;
	}

	public CBSAccount createTestCbsAccount(final String icn) {
		final CBSAccount cbsAccount = new CBSAccount();
		cbsAccount.setIcn(icn);
		return cbsAccount;
	}

	public CBSAccount createTestCbsAccount() {
		return createTestCbsAccount("testIcn" + ThreadLocalRandom.current().nextInt(100000, 999999));
	}

	private static String getTestIcn(final int key) {
		return String.format("TESTAPPSICN%02d", key);
	}

	public void deleteBatchRun(final BatchRun batchRun) {
		jdbcTemplate.update("DELETE FROM BatchRun WHERE ID=" + batchRun.getId());
	}

	public void deleteCbsAccount(final long cbsAccountId) {
		jdbcTemplate.update("DELETE FROM CBSAccount WHERE ID =" + cbsAccountId);
	}

	public static Month getMonth(final Date date) {
		final Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		return Month.of(calendar.get(Calendar.MONTH));
	}

	public static Date getPreviousYear() {
		final Calendar calendar = Calendar.getInstance();
		calendar.setTime(Date.from(Instant.now()));
		calendar.roll(Calendar.YEAR, -1);
		return calendar.getTime();
	}
}
