package gov.va.cpss.service.impl.vfs;

import java.io.File;
import java.lang.reflect.Constructor;
import java.net.URLStreamHandlerFactory;
import java.util.Collection;

import org.apache.commons.logging.Log;
import org.apache.commons.vfs2.CacheStrategy;
import org.apache.commons.vfs2.Capability;
import org.apache.commons.vfs2.FileContentInfoFactory;
import org.apache.commons.vfs2.FileName;
import org.apache.commons.vfs2.FileObject;
import org.apache.commons.vfs2.FileSystem;
import org.apache.commons.vfs2.FileSystemConfigBuilder;
import org.apache.commons.vfs2.FileSystemException;
import org.apache.commons.vfs2.FileSystemManager;
import org.apache.commons.vfs2.FileSystemOptions;
import org.apache.commons.vfs2.FilesCache;
import org.apache.commons.vfs2.NameScope;
import org.apache.commons.vfs2.impl.DefaultFileSystemManager;
import org.apache.commons.vfs2.operations.FileOperationProvider;

/**
 * Provides a proxy for a DefaultFileSystemManager which allows the reference to
 * be removed when the file system is closed and also ensures the file system is
 * closed when finalized.
 * 
 * @author Brad Pickle
 *
 * @param <T>
 *            A DefaultFileSystemManager Type
 */
public class ProxyFileSystemManager<T extends DefaultFileSystemManager> implements FileSystemManager {

	/**
	 * The proxied DefaultFileSystemManager.
	 */
	private T fsm;

	/**
	 * Create a proxy for the given DefaultFileSystemManager.
	 * 
	 * @param defaultFileSystemManager
	 *            DefaultFileSystemManager to proxy.
	 */
	public ProxyFileSystemManager(T defaultFileSystemManager) {
		fsm = defaultFileSystemManager;
	}

	/**
	 * Close the proxied DefaultFileSystemManager.
	 */
	public void close() {
		if (fsm != null) {
			fsm.close();
			fsm = null;
		}
	}

	/**
	 * Get the proxied DefaultFileSystemManager.
	 * 
	 * @return The proxied DefaultFileSystemManager.
	 */
	T getFileSystemManager() {
		return fsm;
	}

	/**
	 * Set the proxied DefaultFileSystemManager.
	 * 
	 * @param fsm
	 *            The DefaultFileSystemManager to proxy.
	 */
	void setFileSystemManager(T fsm) {
		this.fsm = fsm;
	}

	@Override
	protected void finalize() throws Throwable {
		close();
	}

	@Override
	public FileObject getBaseFile() throws FileSystemException {
		return fsm.getBaseFile();
	}

	@Override
	public FileObject resolveFile(String name) throws FileSystemException {
		return fsm.resolveFile(name);
	}

	@Override
	public FileObject resolveFile(String name, FileSystemOptions fileSystemOptions) throws FileSystemException {
		return fsm.resolveFile(name, fileSystemOptions);
	}

	@Override
	public FileObject resolveFile(FileObject baseFile, String name) throws FileSystemException {
		return fsm.resolveFile(baseFile, name);
	}

	@Override
	public FileObject resolveFile(File baseFile, String name) throws FileSystemException {
		return fsm.resolveFile(baseFile, name);
	}

	@Override
	public FileName resolveName(FileName root, String name) throws FileSystemException {
		return fsm.resolveName(root, name);
	}

	@Override
	public FileName resolveName(FileName root, String name, NameScope scope) throws FileSystemException {
		return fsm.resolveName(root, name, scope);
	}

	@Override
	public FileObject toFileObject(File file) throws FileSystemException {
		return fsm.toFileObject(file);
	}

	@Override
	public FileObject createFileSystem(String provider, FileObject file) throws FileSystemException {
		return fsm.createFileSystem(provider, file);
	}

	@Override
	public void closeFileSystem(FileSystem filesystem) {
		fsm.closeFileSystem(filesystem);
	}

	@Override
	public FileObject createFileSystem(FileObject file) throws FileSystemException {
		return fsm.createFileSystem(file);
	}

	@Override
	public FileObject createVirtualFileSystem(String rootUri) throws FileSystemException {
		return fsm.createVirtualFileSystem(rootUri);
	}

	@Override
	public FileObject createVirtualFileSystem(FileObject rootFile) throws FileSystemException {
		return fsm.createVirtualFileSystem(rootFile);
	}

	@Override
	public URLStreamHandlerFactory getURLStreamHandlerFactory() {
		return fsm.getURLStreamHandlerFactory();
	}

	@Override
	public boolean canCreateFileSystem(FileObject file) throws FileSystemException {
		return fsm.canCreateFileSystem(file);
	}

	@Override
	public FilesCache getFilesCache() {
		return fsm.getFilesCache();
	}

	@Override
	public CacheStrategy getCacheStrategy() {
		return fsm.getCacheStrategy();
	}

	@Override
	public Class<?> getFileObjectDecorator() {
		return fsm.getFileObjectDecorator();
	}

	@Override
	public Constructor<?> getFileObjectDecoratorConst() {
		return fsm.getFileObjectDecoratorConst();
	}

	@Override
	public FileContentInfoFactory getFileContentInfoFactory() {
		return fsm.getFileContentInfoFactory();
	}

	@Override
	public boolean hasProvider(String scheme) {
		return fsm.hasProvider(scheme);
	}

	@Override
	public String[] getSchemes() {
		return fsm.getSchemes();
	}

	@Override
	public Collection<Capability> getProviderCapabilities(String scheme) throws FileSystemException {
		return fsm.getProviderCapabilities(scheme);
	}

	@Override
	public void setLogger(Log log) {
		fsm.setLogger(log);
	}

	@Override
	public FileSystemConfigBuilder getFileSystemConfigBuilder(String scheme) throws FileSystemException {
		return fsm.getFileSystemConfigBuilder(scheme);
	}

	@Override
	public FileName resolveURI(String uri) throws FileSystemException {
		return fsm.resolveURI(uri);
	}

	@Override
	public void addOperationProvider(String scheme, FileOperationProvider operationProvider)
			throws FileSystemException {
		fsm.addOperationProvider(scheme, operationProvider);
	}

	@Override
	public void addOperationProvider(String[] schemes, FileOperationProvider operationProvider)
			throws FileSystemException {
		fsm.addOperationProvider(schemes, operationProvider);
	}

	@Override
	public FileOperationProvider[] getOperationProviders(String scheme) throws FileSystemException {
		return fsm.getOperationProviders(scheme);
	}

}
