package gov.va.cpss.service;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.springframework.stereotype.Service;

import gov.va.cpss.dao.CBSStmtDAO;
import gov.va.cpss.dao.PrintAckRecDAO;
import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.cbs.CBSStmt;
import gov.va.cpss.model.printack.ADFileRecord;
import gov.va.cpss.model.printack.PrintAckRec;

/**
 * Service class for handling activities relating to processing jobs.
 * 
 * @author Brad Pickle
 *
 */
@Service
public class UpdatePrintAckService {

	private static final Logger logger = Logger.getLogger(UpdatePrintAckService.class.getCanonicalName());

	private ProcessStatusDAO processStatusDAO;

	private PrintAckRecDAO printAckRecDAO;

	private CBSStmtDAO cbsStmtDAO;

	public PrintAckRecDAO getPrintAckRecDAO() {
		return printAckRecDAO;
	}

	public void setPrintAckRecDAO(PrintAckRecDAO printAckRecDAO) {
		this.printAckRecDAO = printAckRecDAO;
	}

	public CBSStmtDAO getCbsStmtDAO() {
		return cbsStmtDAO;
	}

	public void setCbsStmtDAO(CBSStmtDAO cbsStmtDAO) {
		this.cbsStmtDAO = cbsStmtDAO;
	}

	public ProcessStatusDAO getProcessStatusDAO() {
		return processStatusDAO;
	}

	public void setProcessStatusDAO(ProcessStatusDAO processStatusDAO) {
		this.processStatusDAO = processStatusDAO;
	}

	/**
	 * Initialize the given PrintAckRec. Status will be set to INITIAL and a
	 * record created in the database.
	 * 
	 * @param printAckRec
	 *            PrintAckRec to initialize.
	 * @return The id of the PrintAckRec that is created
	 */
	public long initializePrintAck(final PrintAckRec printAckRec) {

		final Integer initialStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.INITIAL);
		if (initialStatusId == null) {
			final StringBuilder error = new StringBuilder();
			error.append("Unrecoverable data error while initializing PrintAckRec. ")
					.append("Unable to obtain status mapping for: ").append(ProcessStatus.Status.INITIAL);
			logger.error(error.toString());
			throw new RuntimeException(error.toString());
		}

		printAckRec.setStatusId(initialStatusId);

		return printAckRecDAO.insert(printAckRec);
	}

	/**
	 * Updates all of the CBSStmt records acknowledged by the given PrintAckRec
	 * for the given statementDate and sets the printAckId and printDate fields
	 * to the id of the given PrintAckRec and the given printDate, respectively,
	 * and sets the status to ACK.
	 * 
	 * @param printAckRec
	 *            CBSStmt records acknowledged by this PrintAckRec will be
	 *            updated.
	 * @param statementDate
	 *            CBSStmt records with this statementDate will be updated.
	 * @param printDate
	 *            The printDate to update CBSStmt records with.
	 * @return The number of print acknowledgments for the given PrintAckRec.
	 */
	public int processPrintAcks(final PrintAckRec printAckRec, final Date statementDate, final Date printDate,
			final List<ADFileRecord> missingPatientAccts) {
		final Integer sentStmtStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.SENT);
		if (sentStmtStatusId == null) {
			final StringBuilder error = new StringBuilder();
			error.append("Unrecoverable data error while initializing PrintAckRec. ")
					.append("Unable to obtain status mapping for: ").append(ProcessStatus.Status.SENT);
			logger.error(error.toString());
			throw new RuntimeException(error.toString());
		}

		final Integer ackStmtStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.ACK);
		if (ackStmtStatusId == null) {
			final StringBuilder error = new StringBuilder();
			error.append("Unrecoverable data error while initializing PrintAckRec. ")
					.append("Unable to obtain status mapping for: ").append(ProcessStatus.Status.ACK);
			logger.error(error.toString());
			throw new RuntimeException(error.toString());
		}

		// Get the CBSStmts for all of the patients for the PrintAckRec for the
		// given statementDate.
		// Only statements currently in SENT or ACK status will be returned.
		final Map<String, CBSStmt> cbsStmtByAccountM = cbsStmtDAO.getStatementsForPrintAck(printAckRec, statementDate,
				sentStmtStatusId, ackStmtStatusId);
		logger.debug("cbsStmtDAO.getStatementsForPrintAck cbsStmtByAccountM.size():" + cbsStmtByAccountM.size());

		// Build the list of CBSStmts that need to be updated. If the statement
		// is not in ACK status or doesn't have the given statement date or
		// printAckRecId, then it will be updated. Also check for patient
		// accounts that we don't have statements for. These will be accumulated
		// in missingPatientAccts if there are any missing statements, then
		// nothing will be updated.
		final List<CBSStmt> cbsStmtsToUpdateL = prepareStatementsForUpdate(printAckRec, cbsStmtByAccountM,
				missingPatientAccts, printDate, ackStmtStatusId);
		if (missingPatientAccts.size() > 0)
			return 0;

		// Update the status=ACK, printAckId and printDate of the CBSStmts.
		cbsStmtDAO.updatePrintAckFields(cbsStmtsToUpdateL, printAckRec.getId(), printDate, ackStmtStatusId,
				printAckRec.getStationNum());

		// The number of Acks is the number of patient accounts associated with
		// the PrintAckRec.
		return numberOfAcks(printAckRec);

	}

	/**
	 * Builds and returns the list of CBSStmts that need to be updated. Each
	 * CBSStmt in the given cbsStmtL list will be checked and if it needs to be
	 * updated, will be added to the return list. A CBSStmt needs to be updated
	 * if its printAckId or printDate fields are not set, it's status is not
	 * ACK, or the statements primary site matches the PrintAckRec's stationNum
	 * but its printAckId does not match the printAckRec's id.
	 * 
	 * This method also determines if statements could not be found for any of
	 * the patient accounts in the PrintAckDetail list of the given printAckRec.
	 * If a given ADFileRecord in the printAckRec.printAckDetailL does not have
	 * a corresponding statement in the given CBSStmt list, then it will be
	 * added to the missingPatientAccts list. This list should be passed in by
	 * the calling routine.
	 * 
	 * @param printAckRec
	 *            The PrintAckRec to prepare the statements for.
	 * @param cbsStmtByAccount
	 *            Map of CBSStmts for each ADFileRecord in the given
	 *            printAckRec indexed by oldAcntNum.
	 * @param missingPatientAccts
	 *            List provided by the calling routine. This method will add
	 *            ADFileRecords to this list if it doesn't have a CBSStmt.
	 * @param printDate
	 *            The print date that the CBSStmts should have.
	 * @param ackStmtStatusId
	 *            The status id that the CBSStmts should have.
	 * @return The List of CBSStmts that need to be updated in the database.
	 */
	private List<CBSStmt> prepareStatementsForUpdate(final PrintAckRec printAckRec, final Map<String, CBSStmt> cbsStmtByAccount,
			final List<ADFileRecord> missingPatientAccts, final Date printDate, final int ackStmtStatusId) {
		final List<ADFileRecord> patientAccts = (printAckRec.getPrintAckDetailL() == null)
				? new ArrayList<ADFileRecord>(0) : printAckRec.getPrintAckDetailL();
		final List<CBSStmt> cbsStmtsToUpdateL = new ArrayList<CBSStmt>(patientAccts.size());
		
		for (ADFileRecord adFileRecord : patientAccts) {
			if (adFileRecord != null) {
				CBSStmt cbsStmt = cbsStmtByAccount.get(adFileRecord.getPatientAccount());
				if (cbsStmt == null) {
					missingPatientAccts.add(adFileRecord);
				} else if (shouldUpdateCBSStmt(cbsStmt, printAckRec, printDate, ackStmtStatusId)) {
//					logger.debug("Updating CBSStmt:" + cbsStmt);
					cbsStmtsToUpdateL.add(cbsStmt);
				} else {
//					logger.debug("Not Updating CBSStmt:" + cbsStmt);

				}
			}
		}

		return cbsStmtsToUpdateL;
	}

	/**
	 * Determines if the given CBSStmt should be updated.
	 * 
	 * @param cbsStmt
	 *            The CBSStmt to test.
	 * @param printAckRec
	 *            The CBSStmt.printAckId property should match the id of this
	 *            PrintAckRec. However, if the statement has already been
	 *            acknowledged then it will only be updated to the printAckRec
	 *            id if the statements primary site is the same as the
	 *            printAckRec stationNum.
	 * @param printDate
	 *            The CBSStmt.printDate property should match this print date.
	 * @param ackStmtStatusId
	 *            The CBSStmt.statusId property should match this status id.
	 * @return True if the given CBSStmt object's printAckId, printDate or
	 *         statusId do not match the given values, indicating that these
	 *         values should be updated in the database for this CBSStmt.
	 */
	private boolean shouldUpdateCBSStmt(final CBSStmt cbsStmt, final PrintAckRec printAckRec, final Date printDate,
			final int ackStmtStatusId) {

		return ((cbsStmt.getStatusId() != ackStmtStatusId) || !printDate.equals(cbsStmt.getPrintDate())
				|| (cbsStmt.getPrintAckId() != ackStmtStatusId));
	}

	/**
	 * Calculate the number of acknowledgments for the PrintAckRec. This is just
	 * the number of ADFileRecords, since any missing statements would cause the
	 * process to abort.
	 * 
	 * @param printAckRec
	 *            Calculates number of acknowledgments for this PrintAckRec
	 * @return Number of acknowledgments for the given PrintAckRec.
	 */
	private int numberOfAcks(final PrintAckRec printAckRec) {
		final List<ADFileRecord> patientAccts = printAckRec.getPrintAckDetailL();

		int numberOfAcks = 0;

		if (patientAccts == null)
			return numberOfAcks;

		for (ADFileRecord adFileRecord : patientAccts) {
			if (adFileRecord != null)
				numberOfAcks++;
		}

		return numberOfAcks;
	}

	/**
	 * Updates all CBSStmt rows in SENT status for the given PrintAckRec to ACK
	 * status.
	 * 
	 * @param printAckRec
	 *            Update CBSStmt rows with printAckId equal to this records Id.
	 * @return The number of CBSStmt rows updated.
	 */
	public int updateStmtStatusToAck(final PrintAckRec printAckRec) {
		final Integer sentStmtStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.SENT);
		if (sentStmtStatusId == null) {
			final StringBuilder error = new StringBuilder();
			error.append("Unrecoverable data error while initializing PrintAckRec. ")
					.append("Unable to obtain status mapping for: ").append(ProcessStatus.Status.SENT);
			logger.error(error.toString());
			throw new RuntimeException(error.toString());
		}

		final Integer ackStmtStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.ACK);
		if (ackStmtStatusId == null) {
			final StringBuilder error = new StringBuilder();
			error.append("Unrecoverable data error while initializing PrintAckRec. ")
					.append("Unable to obtain status mapping for: ").append(ProcessStatus.Status.ACK);
			logger.error(error.toString());
			throw new RuntimeException(error.toString());
		}

		return cbsStmtDAO.updateStatusFromToForPrintAckId(sentStmtStatusId, ackStmtStatusId, printAckRec.getId());
	}

	/**
	 * Update the status of the given PrintAckRec to SUCCESS.
	 * 
	 * @param printAckRec
	 *            PrintAckRec to update.
	 */
	public void setPrintAckSuccess(final PrintAckRec printAckRec) {
		final Integer successStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.SUCCESS);
		if (successStatusId == null) {
			final StringBuilder error = new StringBuilder();
			error.append("Unrecoverable data error while initializing PrintAckRec. ")
					.append("Unable to obtain status mapping for: ").append(ProcessStatus.Status.SUCCESS);
			logger.error(error.toString());
			throw new RuntimeException(error.toString());
		}

		printAckRec.setStatusId(successStatusId);

		printAckRecDAO.updateStatus(printAckRec);
	}

	/**
	 * Undo the print acknowledgments represented by the given List of
	 * PrintAckRec record ids. The status of each PrintAckRec record will be set
	 * to ERROR and all associated CBSStmt records will be reverted back to SENT
	 * and the printDate and printAckId nulled.
	 * 
	 * @param printAckRecIdList
	 *            List of PrintAckRec record ids to undo.
	 */
	public Result<Object[]> undoPrintAcknowledgements(final List<Long> printAckRecIdList) {
		final Integer errorStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.ERROR);
		if (errorStatusId == null) {
			final StringBuilder error = new StringBuilder();
			error.append("Unrecoverable data error while undoing print acknowledgements. ")
					.append("Unable to obtain status mapping for: ").append(ProcessStatus.Status.ERROR);
			logger.error(error.toString());
			return new Result<>(false, error.toString(), null);
		}

		// Set the status of the PrintAckRecs to ERROR
		final int numPrintAckRecUpdated = printAckRecDAO.updateStatus(printAckRecIdList, errorStatusId);

		final Integer sentStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.SENT);
		if (sentStatusId == null) {
			final StringBuilder error = new StringBuilder();
			error.append("Unrecoverable data error while undoing print acknowledgements. ")
					.append("Unable to obtain status mapping for: ").append(ProcessStatus.Status.SENT);
			logger.error(error.toString());
			return new Result<>(false, error.toString(), null);
		}

		// Revert statements for each PrintAckRec back to SENT and null the
		// printDate and printAckId
		cbsStmtDAO.undoPrintAcknowledgements(printAckRecIdList, sentStatusId);

		Result<Object[]> result;

		if (numPrintAckRecUpdated != printAckRecIdList.size()) {
			final StringBuilder error = new StringBuilder();
			error.append("ERROR: Unexpected number of PrintAckRec records set to ");
			error.append(ProcessStatus.Status.ERROR);
			error.append(". Expected: ");
			error.append(printAckRecIdList.size());
			error.append(", Actual: ");
			error.append(numPrintAckRecUpdated);
			result = new Result<>(false, error.toString(), null);
		} else {
			result = new Result<>(true, "Success", null);
		}

		return result;
	}

}
