package gov.va.cpss.service;

import java.util.Date;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import org.apache.log4j.Logger;
import org.springframework.stereotype.Service;

import gov.va.cpss.dao.APSStmtDAO;
import gov.va.cpss.dao.BatchRunProcessDAO;
import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.model.BatchRunProcess;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.appsprintack.APPSADFileRecord;

/**
 * Service class for the Update APPS Print Ack batch job.
 * 
 * @author Andrew Vance
 */
@Service
public class UpdateAPPSPrintAckService {
	private static final Logger logger = Logger.getLogger(UpdateAPPSPrintAckService.class.getCanonicalName());
	
	private APSStmtDAO apsStmtDAO;
	
	private ProcessStatusDAO processStatusDAO;
	
	private BatchRunProcessDAO batchRunProcessDAO;

	public APSStmtDAO getApsStmtDAO() {
		return apsStmtDAO;
	}

	public void setApsStmtDAO(APSStmtDAO apsStmtDAO) {
		this.apsStmtDAO = apsStmtDAO;
	}
	
	public ProcessStatusDAO getProcessStatusDAO() {
		return processStatusDAO;
	}

	public void setProcessStatusDAO(ProcessStatusDAO processStatusDAO) {
		this.processStatusDAO = processStatusDAO;
	}

	public BatchRunProcessDAO getBatchRunProcessDAO() {
		return batchRunProcessDAO;
	}

	public void setBatchRunProcessDAO(BatchRunProcessDAO batchRunProcessDAO) {
		this.batchRunProcessDAO = batchRunProcessDAO;
	}

	/**
	 * Initialize the given BatchRunProcess. Status will be set to INITIAL and a 
	 * record created in the database.
	 * @param batchRunProcess BatchRunProcess to initialize
	 * @return the id of the BatchRunProcess created
	 */
	public long initializeBatchRunProcess(final BatchRunProcess batchRunProcess) {
		
		final Integer initialStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.INITIAL);
		if (initialStatusId == null) {
			final StringBuilder error = new StringBuilder();
			error.append("Unrecoverable data error while initializing BatchRunProcess. ")
			.append("Unable to obtain status mapping for: ").append(ProcessStatus.Status.INITIAL);
			logger.error(error.toString());
			throw new RuntimeException(error.toString());
		}
		
		batchRunProcess.setStatusId(initialStatusId);
		
		return batchRunProcessDAO.insert(batchRunProcess);
	
	}
	
	/**
	 * Updates all of the APSStmt records acknowledged by the given BatchRunProcess
	 * for the given statementDate and sets the statusId and printDate fields
	 * to the id of the given BatchRunProcess and the given printDate, respectively,
	 * and sets the status to ACK.
	 * 
	 * @param batchRunProcess
	 * @param statementYear
	 * @param printDate
	 * @param missingPatientAccts
	 * @return
	 */
	public int processPrintAckDetails(final BatchRunProcess batchRunProcess, final List<APPSADFileRecord> printAckDetailL, final int statementYear, final Date printDate, 
			final List<APPSADFileRecord> missingPatientAccts) {
		final Integer sentStmtStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.SENT);
		if(sentStmtStatusId == null) {
			final StringBuilder error = new StringBuilder();
			error.append("Unrecoverable data error while initializing BatchRunProcess. ")
			.append("Unable to obtain status mapping for: ").append(ProcessStatus.Status.SENT);
			logger.error(error.toString());
			throw new RuntimeException(error.toString());			
		}
		
		final Integer ackStmtStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.ACK);
		if (ackStmtStatusId == null) {
			final StringBuilder error = new StringBuilder();
			error.append("Unrecoverable data error while initializing BatchRunProcess. ")
					.append("Unable to obtain status mapping for: ").append(ProcessStatus.Status.ACK);
			logger.error(error.toString());
			throw new RuntimeException(error.toString());
		}
		
		final List<Long> apsStmtIdL = apsStmtDAO.getStatementIdsForPrintAck(printAckDetailL, statementYear, 
				sentStmtStatusId.intValue(), ackStmtStatusId.intValue(), printDate, batchRunProcess.getId());
		if(apsStmtIdL == null || apsStmtIdL.isEmpty() ) {
			final StringBuilder error = new StringBuilder();
			error.append("No Data");
			logger.error(error.toString());
			throw new RuntimeException(error.toString());
		}
		
		logger.debug("apsStmtDAO.getStatementsForBatchRunProcess apsStmtL.size():" + apsStmtIdL.size());
		for(Long apsStmtId : apsStmtIdL) {
			logger.debug(apsStmtId);
		}
		
		final List<Long> apsStmtIdsToUpdateL = prepareStatementsForUpdate(printAckDetailL, apsStmtIdL, missingPatientAccts, 
				printDate, ackStmtStatusId);
		if(missingPatientAccts.size() > 0)
			return 0;
		
		apsStmtDAO.updatePrintAckFields(apsStmtIdsToUpdateL, batchRunProcess.getId(), printDate, ackStmtStatusId);
		
		return numberOfAcks(printAckDetailL);
	}
	
	/**
	 * Update the status of the given BatchRunProcess to SUCCESS.
	 * 
	 * @param batchRunProcess
	 */
	public void setPrintAckSuccess(final BatchRunProcess batchRunProcess) {
		final Integer successStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.SUCCESS);
		if(successStatusId == null) {
			final StringBuilder error = new StringBuilder();
			error.append("Unrecoverable data error while initializing BatchRunProcess. ")
					.append("Unable to obtain status mapping for: ").append(ProcessStatus.Status.SUCCESS);
			logger.error(error.toString());
			throw new RuntimeException();
		}
		
		batchRunProcess.setStatusId(successStatusId);
		
		batchRunProcessDAO.updateStatus(batchRunProcess);
	}

	//TODO: flesh out javadoc
	/**
	 * Builds and returns the list of APSStmts that need to be updated. 
	 * 
	 * @param batchRunProcess
	 * @param apsStmtIdL
	 * @param missingPatientAccts
	 * @param printDate
	 * @param ackStmtStatusId
	 * @return
	 */
	private List<Long> prepareStatementsForUpdate(final List<APPSADFileRecord> printAckDetailL, final List<Long> apsStmtIdL,
			final List<APPSADFileRecord> missingPatientAccts, final Date printDate, final int ackStmtStatusId) {

		final List<APPSADFileRecord> nonNullPatientAccountsL = Objects.nonNull(printAckDetailL) 
				? printAckDetailL.stream().filter(Objects::nonNull).collect(Collectors.toList()) 
						: new ArrayList<APPSADFileRecord>(0);
		
		final List<Long> absStmtIdsToUpdateL = new ArrayList<Long>((Objects.isNull(apsStmtIdL)) ? 0 : apsStmtIdL.size());
		
		int idx = 0;
		for(APPSADFileRecord record : nonNullPatientAccountsL) {		
			if(Objects.isNull(apsStmtIdL)) {
				logger.debug("APSStmt list is null, no possible action for patient account: " + record.getPatientAccount());
				missingPatientAccts.add(record);
			} else if (apsStmtIdL.size() <= idx || Objects.isNull(apsStmtIdL.get(idx))) {
				logger.debug("Missing patient account: " + record.getPatientAccount());
				missingPatientAccts.add(record);
			} else {
				logger.debug("Updating APSStmt:" + apsStmtIdL.get(idx));
				absStmtIdsToUpdateL.add(apsStmtIdL.get(idx));					
			}
			++idx;
		}
		
		return absStmtIdsToUpdateL;
	}
	
	/**
	 * Calculate the number of acknowledgments for the BatchRunProcess. This is just
	 * the number of APPSADFileRecords, since any missing statements would cause the
	 * process to abort.
	 * 
	 * @param batchRunProcess
	 *            Calculates number of acknowledgments for this BatchRunProcess
	 * @return Number of acknowledgments for the given BatchRunProcess.
	 */
	private int numberOfAcks(final List<APPSADFileRecord> patientAccts) {
		int numberOfAcks = 0;
		
		if(patientAccts == null)
			return numberOfAcks;
		
		for (APPSADFileRecord record : patientAccts) {
			if(record != null)
				numberOfAcks++;
		}
		
		return numberOfAcks;
	}
	
	/**
	 * Undo the print acknowledgments represented by the given List of
	 * PrintAckRec record ids. The status of each PrintAckRec record will be set
	 * to ERROR and all associated CBSStmt records will be reverted back to SENT
	 * and the printDate and printAckId nulled.
	 * 
	 * @param batchRunProcessIdList
	 *            List of PrintAckRec record ids to undo.
	 */
	public Result<Object[]> undoPrintAcknowledgements(final List<Long> batchRunProcessIdList) {
		final Integer errorStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.ERROR);
		if (errorStatusId == null) {
			final StringBuilder error = new StringBuilder();
			error.append("Unrecoverable data error while undoing print acknowledgements. ")
					.append("Unable to obtain status mapping for: ").append(ProcessStatus.Status.ERROR);
			logger.error(error.toString());
			return new Result<>(false, error.toString(), null);
		}

		// Set the status of the PrintAckRecs to ERROR
		final int numPrintAckRecUpdated = batchRunProcessDAO.updateStatus(batchRunProcessIdList, errorStatusId);

		final Integer sentStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.SENT);
		if (sentStatusId == null) {
			final StringBuilder error = new StringBuilder();
			error.append("Unrecoverable data error while undoing print acknowledgements. ")
					.append("Unable to obtain status mapping for: ").append(ProcessStatus.Status.SENT);
			logger.error(error.toString());
			return new Result<>(false, error.toString(), null);
		}

		// Revert statements for each PrintAckRec back to SENT and null the
		// printDate and printAckId
		apsStmtDAO.undoPrintAcknowledgements(batchRunProcessIdList, sentStatusId);

		Result<Object[]> result;

		if (numPrintAckRecUpdated != batchRunProcessIdList.size()) {
			final StringBuilder error = new StringBuilder();
			error.append("ERROR: Unexpected number of BatchRunProcess records set to ");
			error.append(ProcessStatus.Status.ERROR);
			error.append(". Expected: ");
			error.append(batchRunProcessIdList.size());
			error.append(", Actual: ");
			error.append(numPrintAckRecUpdated);
			result = new Result<>(false, error.toString(), null);
		} else {
			result = new Result<>(true, "Success", null);
		}

		return result;
	}
}