package gov.va.cpss.service;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Stream;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.log4j.Logger;

import gov.va.cpss.model.updatestation.Address;
import gov.va.cpss.model.updatestation.AddressEditor;
import gov.va.cpss.model.updatestation.StationInfo;
import gov.va.cpss.model.updatestation.ZipCode;
import gov.va.cpss.model.updatestation.ZipCodeEditor;

@SuppressWarnings("nls")
public class StationInfoService {

	private Logger logger = Logger.getLogger(StationInfoService.class);

	// This is to alleviate the "race condition" raised by the Fortify scan.
	private AtomicReference<Map<String, StationInfo>> stationInfoMapRef = new AtomicReference<>();

	private Map<String, Integer[]> stationInfoRangeMap;

	private String stationInfoDir;

	private String stationInfoFileName;

	public String getStationInfoDir() {
		return this.stationInfoDir;
	}

	public void setStationInfoDir(String inStationInfoDir) {
		this.stationInfoDir = inStationInfoDir;
	}

	public String getStationInfoFileName() {
		return this.stationInfoFileName;
	}

	public void setStationInfoFileName(String inStationInfoFileName) {
		this.stationInfoFileName = inStationInfoFileName;
	}

	public Map<String, Integer[]> getStationInfoRangeMap() {
		return this.stationInfoRangeMap;
	}

	public void setStationInfoRangeMap(Map<String, Integer[]> inStationInfoRangeMap) {
		this.stationInfoRangeMap = inStationInfoRangeMap;
	}

	/**
	 * Returns the StationInfo object for the respective station number.
	 * 
	 * @param stationNum
	 *            The station (facility) number
	 */
	public StationInfo getStation(String stationNum) {
		return getStationMap().get(stationNum);
	}

	public void setStationMap(Map<String, StationInfo> map) {
		this.stationInfoMapRef.set(map);
	}

    public synchronized Map<String, StationInfo> getStationMap()
    {
		if (this.stationInfoMapRef.get() == null)
		{
		    Map<String, StationInfo> stationInfoMap = new HashMap<>();

			try (Stream<String> stream = Files.lines(Paths.get(this.stationInfoDir + File.separator + this.stationInfoFileName)))
			{
				stream.forEach(line ->
				{
					final StationInfo si = buildStationInfo(line);
					stationInfoMap.put(si.getStationNum(), si);
				});
			}
			catch (IOException e)
			{
				this.logger.error(String.format("Unable to read station info file: ", this.stationInfoFileName) + ". Error: " + e.getMessage());
			}

			this.stationInfoMapRef.set(stationInfoMap);
		}

		return this.stationInfoMapRef.get();
	}

	/**
	 * Returns the size of the StationInfo map
	 */
	public int getTotalNumberOfStations()
	{
		return (this.stationInfoMapRef.get() == null) ? 0 : this.stationInfoMapRef.get().size();
	}

	protected StationInfo buildStationInfo(final String line) {

		final StationInfo stationInfo = new StationInfo();
		final ZipCodeEditor zipCode = new ZipCodeEditor();
		final AddressEditor address = new AddressEditor();

		for (String key : this.stationInfoRangeMap.keySet()) {

			final Integer[] range = this.stationInfoRangeMap.get(key);

			String value = "";

			if (range.length == 2) {

				value = line.substring(range[0].intValue() - 1, range[1].intValue()).trim();

			} else {

				value = line.substring(range[0].intValue() - 1, range[0].intValue()).trim();

			}

			try {
				final Class<?> propertyClass = PropertyUtils.getPropertyType(stationInfo, key);

				if (propertyClass.equals(String.class)) {
					PropertyUtils.setSimpleProperty(stationInfo, key, value);
				} else if (propertyClass.equals(Address.class)) {
					// Correct Address not breaking up as it's supposed to be.
					// Yiping Yao - 02/22/2018
					address.setAsText(value);
					PropertyUtils.setSimpleProperty(stationInfo, key, address.getValue());
					//final Address address = new Address();
					//address.setAddress1(value);
					//PropertyUtils.setSimpleProperty(si, key, address);
				} else if (propertyClass.equals(ZipCode.class)) {
					// Correct ZIP Code not breaking up as it's supposed to be.
					// Yiping Yao - 02/21/2018
					zipCode.setAsText(value);
					PropertyUtils.setSimpleProperty(stationInfo, key, zipCode.getValue());
					//final ZipCode zipCode = new ZipCode();
					//zipCode.setZipCode1(value);
					//PropertyUtils.setSimpleProperty(si, key, zipCode);
				}
			} catch (Exception e) {
				this.logger.error(String.format("Unable to set field [%s] with value [%s] due to: " + e.getMessage(), key, value));
			}
		}

		return stationInfo;
	}
}
