package gov.va.cpss.service;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.text.SimpleDateFormat;
import java.util.Formatter;
import java.util.List;

import org.apache.log4j.Logger;

import gov.va.cpss.dao.CBSPatAcntBalMsgDAO;
import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.dao.SiteBalDAO;
import gov.va.cpss.job.sendbill.SendBillStringDTO;
import gov.va.cpss.model.AITCDollar;
import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.bal.PatAcntBalMsg;
import gov.va.cpss.model.bal.SiteBalance;

public class SendBillDataService {

	private final String OUTPUT_RECORD_TYPE = "LP";
	private final String OUTPUT_DATE_FORMAT = "MMddyyyy";
	private SiteBalDAO siteBalDAO;
	private ProcessStatusDAO processStatusDAO;
	private CBSPatAcntBalMsgDAO cbsPatAcntBalMsgDAO;
	private String stagingDirectory;
	private static Logger serviceLogger = Logger.getLogger(SendBillDataService.class.getCanonicalName());

	/**
	 * @return the processStatusDAO
	 */
	public ProcessStatusDAO getProcessStatussDAO() {
		return processStatusDAO;
	}

	/**
	 * @param processStatusDAO
	 *            the processStatusDAO to set
	 */
	public void setProcessStatusDAO(ProcessStatusDAO processStatusDAO) {
		this.processStatusDAO = processStatusDAO;
	}

	/**
	 * @return the cbsPatAcntBalMsgDAO
	 */
	public CBSPatAcntBalMsgDAO getCbsPatAcntBalMsgDAO() {
		return cbsPatAcntBalMsgDAO;
	}

	/**
	 * @param cbsPatAcntBalMsgDAO
	 *            the cbsPatAcntBalMsgDAO to set
	 */
	public void setCbsPatAcntBalMsgDAO(CBSPatAcntBalMsgDAO cbsPatAcntBalMsgDAO) {
		this.cbsPatAcntBalMsgDAO = cbsPatAcntBalMsgDAO;
	}

	public PatAcntBalMsg startIsolatedSendBillJob(final BatchRun batchRun) {

		final Integer initialStatusId = getProcessStatussDAO().getStatusFromEnum(ProcessStatus.Status.INITIAL);
		if (initialStatusId == null) {
			final StringBuilder error = new StringBuilder();
			error.append("Unrecoverable data error while initializing job. ")
					.append("Unable to obtain status mapping for: ").append(ProcessStatus.Status.INITIAL);
			serviceLogger.error(error.toString());
			throw new RuntimeException(error.toString());
		}
		serviceLogger.debug("initialStatusId:" + initialStatusId);

		final Integer errorStatusId = getProcessStatussDAO().getStatusFromEnum(ProcessStatus.Status.ERROR);
		if (errorStatusId == null) {
			serviceLogger.error("Unable to obtain status mapping for: " + ProcessStatus.Status.ERROR);
			return null;
		}
		serviceLogger.debug("errorStatusId:" + errorStatusId);

		final List<PatAcntBalMsg> cbsMessagesInInitialL = getCbsPatAcntBalMsgDAO()
				.getCBSPatAcntBalMsgInStatus(initialStatusId);
		for (PatAcntBalMsg cbsMessageInInitial : cbsMessagesInInitialL) {
			cbsMessageInInitial.setStatusId(errorStatusId);
			getCbsPatAcntBalMsgDAO().update(cbsMessageInInitial);
		}

		PatAcntBalMsg patAcntBalMessage = new PatAcntBalMsg();
		patAcntBalMessage.setId(initialStatusId);
		patAcntBalMessage.setBatchRunId(batchRun.getId());
		patAcntBalMessage.setFileName(generateFileName(batchRun));
		patAcntBalMessage.setStatusId(batchRun.getStatusId());
		patAcntBalMessage.setTotNumPatient(0L);

		getCbsPatAcntBalMsgDAO().insert(patAcntBalMessage);

		return patAcntBalMessage;
	}

	private String generateFileName(BatchRun batchRun) {
		return String.format("SENDBILLDATA-%tm%<td%<tY.txt", batchRun.getStartDate());
	}

	public SendBillStringDTO processNewBillRecord(SiteBalance patientData) {

		SendBillStringDTO patientDTO = new SendBillStringDTO();
		patientDTO.setRecordType(OUTPUT_RECORD_TYPE);
		patientDTO.setCbssId(patientData.getCbsAccount().getId());
		patientDTO.setNumOfFacilitys(1);
		patientDTO.setPatientBillInfo(buildPatientBillInfo(patientData));

		return patientDTO;
	}

	public SendBillStringDTO processExistingBillRecord(SiteBalance patientData,
			SendBillStringDTO existingPatientBillData) {
		existingPatientBillData.setNumOfFacilitys(existingPatientBillData.getNumOfFacilitys() + 1);
		existingPatientBillData
				.setPatientBillInfo(existingPatientBillData.getPatientBillInfo() + buildPatientBillInfo(patientData));
		return existingPatientBillData;
	}

	/**
	 * Build a formatted string representing the bill info for a patient site
	 * balance.
	 * 
	 * @param patientData
	 *            The site balance for which to create a string representation.
	 * @return A formatted output string for the patient site balance.
	 */
	private String buildPatientBillInfo(final SiteBalance patientData) {

		// Format of fields defined in 'CPSS to LBX record layout_v4.1.xls'
		// These format values are similar to that found in
		// cpss-batch-send-cbs.xml
		// A formatter is used directly for send bill.
		// stationNumber: %-5.5s
		// oldAcctNum: %-19.19s
		// balance: Money(double, 9).getCobol(): %9.9s
		// oldestBillDate: %8.8s

		Formatter formatter = new Formatter();
		SimpleDateFormat dateFormat = new SimpleDateFormat(OUTPUT_DATE_FORMAT);
		final String output = formatter.format("%-5.5s%-19.19s%9.9s%8.8s", patientData.getStationNum(),
				patientData.getOldAcctNum(), new AITCDollar(patientData.getBalance()).toPaddedString(9),
				dateFormat.format(patientData.getOldestBillDate())).toString();
		formatter.close();

		return output;
	}

	public String getBillDataOutputResource(String sendBillOutputFileName) {
		return getStagingDirectory() + "/" + sendBillOutputFileName;
	}

	public boolean deleteBillDataOutputResource(String cbsFileName) {

		File file = new File(cbsFileName);

		try {
			return Files.deleteIfExists(file.toPath());
		} catch (IOException e) {
			final StringBuilder error = new StringBuilder();
			error.append("An error occurred while attempting to delete the local CBS file, ");
			error.append(cbsFileName);
			error.append(", error: ");
			error.append(e.getClass().getSimpleName());
			error.append("\nMessage: ");
			error.append(e.getMessage());

			serviceLogger.error(error.toString());
			return false;
		}

	}

	public boolean setMessageStatus(long patAcntBalMsgId, ProcessStatus.Status status) {

		final Integer statusId = processStatusDAO.getStatusFromEnum(status);
		if (statusId == null) {
			serviceLogger.error("Unable to obtain status mapping for: " + status);
			return false;
		}

		// Look up the PatAcntBalMsg record to make sure it has not been set to
		PatAcntBalMsg cbsPataAccountBalMSG = cbsPatAcntBalMsgDAO.get(patAcntBalMsgId);
		if (cbsPataAccountBalMSG == null) {
			serviceLogger.error("Unable to obtain CBSMessage record for id: " + patAcntBalMsgId);
			return false;
		}

		// Update PatAcntBalMsg status to SUCCESS
		cbsPataAccountBalMSG.setStatusId(statusId);
		cbsPatAcntBalMsgDAO.update(cbsPataAccountBalMSG);

		return true;
	}

	/**
	 * @return the stagingDirectory
	 */
	public String getStagingDirectory() {
		return stagingDirectory;
	}

	/**
	 * @param stagingDirectory
	 *            the stagingDirectory to set
	 */
	public void setStagingDirectory(String stagingDirectory) {
		this.stagingDirectory = stagingDirectory;
	}

	/**
	 * @return the servicelogger
	 */
	public static Logger getServicelogger() {
		return serviceLogger;
	}

	/**
	 * @param servicelogger
	 *            the servicelogger to set
	 */
	public static void setServicelogger(Logger servicelogger) {
		serviceLogger = servicelogger;
	}

	public void updateTotalPatientProcessed(Long totalNumberOfPatients, Long msgId) {
		cbsPatAcntBalMsgDAO.updateTotalPatients(totalNumberOfPatients, msgId);

	}

	public void updateSiteBalPatAcntBalMsgId(Long msgId) {
		serviceLogger.info("Updating SiteBal Acnt Msg ID with: " + msgId);
		siteBalDAO.updateSiteBalPatAcntBalMsgId(msgId);

	}
	
	public Integer countSiteBalPatients()
	{
		return siteBalDAO.getTotalRecordsFromSiteBalToProcess();
	}

	public SiteBalDAO getSiteBalDAO() {
		return siteBalDAO;
	}

	public void setSiteBalDAO(SiteBalDAO siteBalDAO) {
		this.siteBalDAO = siteBalDAO;
	}
}
