package gov.va.cpss.service;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.List;
import java.util.Objects;

import org.springframework.batch.core.JobExecution;
import org.springframework.stereotype.Service;

import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.BatchRunProcess;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.ProcessStatus.Status;
import gov.va.cpss.model.apps.APSStmt;

@Service
@SuppressWarnings({"nls", "static-method"})
public class SendAPPSService extends APPSBaseService<BatchRunProcess> {

	private String stagingDirectory;

	public int getPreviousYear() {
		return this.previousYear;
	}

	public String getStagingDirectory() {
		return this.stagingDirectory;
	}

	public void setStagingDirectory(String inStagingDirectory) {
		this.stagingDirectory = inStagingDirectory;
	}

	public String getAPPSOutputResource(String appsOutputFileName) {
		return this.stagingDirectory + "/" + appsOutputFileName;
	}

	public List<APSStmt> getApsStmtsWithPaging(int page, int pageSize) {
		return this.apsStmtDAO.getApsStmtsWithPaging(page, pageSize, this.previousYear, ProcessStatus.Status.NEW);
	}

	/**
	 * Update all statement status to sent status for the specified message ID.
	 * 
	 * @param batchRunProcessId
	 *            The batch run process ID that will be used to update the
	 *            batchRunProcessIdSend column.
	 * @return Long value that is the number of statement status updated.
	 */
	public long updateAllStatementStatusToSent(final long batchRunProcessId) {

		long totalPatientsIncluded = -1;

		final List<APSStmt> apsStmtL = this.apsStmtDAO.getAllStatementsForStatus(Status.NEW);

		if (Objects.isNull(apsStmtL) || apsStmtL.isEmpty()) {
			totalPatientsIncluded = 0;
		} else {
			/*
			 * TODO: Validate apsStmtL only has the statements we want to
			 * update, perhaps check batchRunProcessIdGen e.g.
			 * apsStmtL.removeif(stmt -> { stmt.getGenBatchRunProcessId != ...
			 * }) Andrew Vance 3/1/2017
			 */

			totalPatientsIncluded = this.apsStmtDAO.updateStatusToSent(apsStmtL, batchRunProcessId).length;
		}

		return totalPatientsIncluded;
	}

	/**
	 * @param appsTempFilePath
	 *            The path to the file that needs to be deleted.
	 * @return True if successful, false if not.
	 */
	public boolean deleteAPPSOutputResource(String appsTempFilePath) {
		File file = new File(appsTempFilePath);

		try {
			return Files.deleteIfExists(file.toPath());
		} catch (IOException e) {
			final StringBuilder error = new StringBuilder();
			error.append("An error occurred while attempting to delete the local APPS file, ");
			error.append(appsTempFilePath);
			error.append(", error: ");
			error.append(e.getClass().getSimpleName());
			error.append("\nMessage: ");
			error.append(e.getMessage());

			this.logger.error(error.toString());
			return false;
		}
	}

    /**
     * Start Send APPS batch job.
     * 
     * @param batchRun
     * @return
     */
    @Override
    public Result<BatchRunProcess> startJob(final BatchRun batchRun, final String fileName)
    {
//    public BatchRunProcess startSendAPPSJob(final BatchRun batchRun) {
/*	
		final Integer initialStatusId = this.processStatusDAO.getStatusFromEnum(ProcessStatus.Status.INITIAL);
		if (initialStatusId == null) {
			final StringBuilder error = new StringBuilder();
			error.append("Unrecoverable data error while initializing job. ")
					.append("Unable to obtain status mapping for: ").append(ProcessStatus.Status.INITIAL);
			this.logger.error(error.toString());
			throw new RuntimeException(error.toString());
		}

		this.logger.debug("initialStatusId:" + initialStatusId);

		final Integer errorStatusId = this.processStatusDAO.getStatusFromEnum(ProcessStatus.Status.ERROR);
		if (errorStatusId == null) {
			this.logger.error("Unable to obtain status mapping for: " + ProcessStatus.Status.ERROR);
			return null;
		}

		this.logger.debug("errorStatusId:" + errorStatusId);
*/

    	Result<BatchRunProcess> result = super.startJob(batchRun, fileName);

        if (result == null)
        {
            result = new Result<>();
        }

		final List<BatchRunProcess> batchRunProcessInitialL = this.batchRunProcessDAO
				.getBatchRunProcessesInStatus(this.statusIdInitial);
		for (BatchRunProcess brp : batchRunProcessInitialL) {
			brp.setStatusId(this.statusIdError);
			this.batchRunProcessDAO.update(brp);
		}

		final BatchRunProcess batchRunProcess = new BatchRunProcess();

		batchRunProcess.setBatchRunId(batchRun.getId());
		batchRunProcess.setStatusId(this.statusIdInitial);
		batchRunProcess.setFileName(generateFileName(batchRun));
		batchRunProcess.setProcessDate(batchRun.getStartDate());

        try
        {
    		batchRunProcess.setId(this.batchRunProcessDAO.insert(batchRunProcess));
 
            result.set(batchRunProcess);
            result.setSuccessful(true);
        }
        catch (Exception ex)
        {
            result.setMessage(ex.getMessage());
            result.setSuccessful(false);
        }

        return result;

		//return batchRunProcess;
	}

    /**
     * End Send APPS batch job.
     * 
     * @param execution
     * @param process
     * @return
     */
    @Override
    public String endJob(JobExecution execution, BatchRunProcess process)
    {
        // Nothing specific for Send APPS job.
        return super.endJob(execution, process);
    }
/*
	public boolean endSendAPPSJob(JobExecution execution, BatchRunProcess batchRunProcess) {

		boolean jobSuccess = false;

		if (execution == null) {

			this.logger.error("Job ended with null execution");
			errorSendAPPSJob(batchRunProcess);

		} else if (execution.getExitStatus() != ExitStatus.COMPLETED) {

			if (execution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {
				this.logger.error("Job ended with failure: " + execution.getExecutionContext().getString(JOB_FAILURE_KEY));
			} else {
				this.logger.error("Job ended with unknown error: " + execution.getExitStatus());
			}
			errorSendAPPSJob(batchRunProcess);
		} else {

			// Job was successful!
			jobSuccess = true;
		}

		this.logger.info("End Send APPS Job");

		return jobSuccess;
	}
*/

	private String generateFileName(BatchRun batchRun) {
		return String.format("APPS-%tm%<td%<tY.txt", batchRun.getStartDate());
	}

	// This method is replaced by override super class method processJobError().
/*
	private void errorSendAPPSJob(BatchRunProcess batchRunProcess) {

		// Change BatchRunProcess status to ERROR
		final Integer errorStatusId = this.processStatusDAO.getStatusFromEnum(ProcessStatus.Status.ERROR);
		if (errorStatusId == null) {
			this.logger.error("Unable to obtain status mapping for: " + ProcessStatus.Status.ERROR);
		} else {
			batchRunProcess.setStatusId(errorStatusId.intValue());
			this.batchRunProcessDAO.update(batchRunProcess);
		}
	}

	public Integer getStatusFromEnum(final ProcessStatus.Status status) {
		return this.processStatusDAO.getStatusFromEnum(status);
	}
*/

	@Override
	protected String processJobError(BatchRunProcess process)
	{
        this.logger.info("Service: Process Send APPS Job Error.");

        String message = null;

        try
        {
            process.setStatusId(this.statusIdError);
            this.batchRunProcessDAO.updateStatus(process);
        }
        catch (Exception ex)
        {
            message = ex.getMessage();
        }
 
        return message;
	}

	@Override
	protected String completeJob(JobExecution execution, BatchRunProcess t)
	{
        this.logger.info("Service: Complete Send APPS Job.");

        String message = null;

        return message;
	}
}
