package gov.va.cpss.service;

import static gov.va.cpss.ESAPI.EsapiValidationType.LOG_FORGING;
import static gov.va.cpss.ESAPI.EsapiValidator.validateStringInput;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Set;

import org.apache.log4j.Logger;
import org.quartz.CronExpression;
import org.quartz.CronTrigger;
import org.quartz.Job;
import org.quartz.JobBuilder;
import org.quartz.JobDataMap;
import org.quartz.JobDetail;
import org.quartz.JobKey;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.SimpleTrigger;
import org.quartz.Trigger;
import org.quartz.TriggerKey;
import org.quartz.impl.matchers.GroupMatcher;
import org.quartz.impl.triggers.CronTriggerImpl;
import org.quartz.impl.triggers.SimpleTriggerImpl;
import org.springframework.batch.core.launch.JobLauncher;
import org.springframework.scheduling.quartz.QuartzJobBean;
import org.springframework.stereotype.Service;

import gov.va.cpss.cron.CronUtils;
import gov.va.cpss.job.InternalJob;
import gov.va.cpss.model.BatchJob;
import gov.va.cpss.model.BatchType;

/**
 * Service class for scheduling activities relating to batch jobs.
 * 
 * @author DNS  
 */
@Service
public class SchedulingService {

	private static final Logger logger = Logger.getLogger(SchedulingService.class.getCanonicalName());

	/**
	 * The job group for Quartz Jobs.
	 */
	private String jobGroup;

	/**
	 * The trigger group for Quartz Jobs.
	 */
	private String triggerGroup;

	/**
	 * The internal job group for Quartz jobs that are not open to the UI.
	 */
	private List<InternalJob> internalGroup;

	/**
	 * The Quartz scheduler.
	 */
	private Scheduler scheduler;

	/**
	 * The service for managing batch jobs.
	 */
	private BatchService batchService;

	/**
	 * Spring Batch JobLauncher for starting FlowJobs
	 */
	private JobLauncher jobLauncher;

	public SchedulingService() {
		internalGroup = new ArrayList<InternalJob>();
	}

	public String getJobGroup() {
		return jobGroup;
	}

	public void setJobGroup(String jobGroup) {
		this.jobGroup = jobGroup;
	}

	public String getTriggerGroup() {
		return triggerGroup;
	}

	public void setTriggerGroup(String triggerGroup) {
		this.triggerGroup = triggerGroup;
	}

	public List<InternalJob> getInternalGroup() {
		return internalGroup;
	}

	public void setInternalGroup(List<InternalJob> internalGroup) {
		this.internalGroup = internalGroup;
	}

	public Scheduler getScheduler() {
		return scheduler;
	}

	public void setScheduler(Scheduler scheduler) {
		this.scheduler = scheduler;
	}

	public BatchService getBatchService() {
		return batchService;
	}

	public void setBatchService(BatchService batchService) {
		this.batchService = batchService;
	}

	public JobLauncher getJobLauncher() {
		return jobLauncher;
	}

	public void setJobLauncher(JobLauncher jobLauncher) {
		this.jobLauncher = jobLauncher;
	}

	/**
	 * Build the trigger name for the batch job. The name is simply the batch
	 * job name with Trigger postfix.
	 * 
	 * @param batchJob
	 *            The batch job for which to create a trigger name.
	 * @return String of the name of the trigger.
	 */
	private String buildTriggerNameForJob(BatchJob batchJob) {
		return batchJob.getName() + "Trigger";
	}

	/**
	 * Save the specified batch job to the database and update the Quartz
	 * Schedule.
	 * 
	 * @param batchJob
	 *            The batch job to save.
	 */
	public void saveBatchJob(BatchJob batchJob) {
		BatchType bt = batchService.getBatchTypeForJob(batchJob);

		// Handle scheduling based on job type.
		switch (bt.getJobType()) {
		case DAILY: {
			logger.info("Saving " + validateStringInput(batchJob.getName(), LOG_FORGING) + ": "
					+ validateStringInput(batchJob.getSchedule(), LOG_FORGING) + " "
					+ validateStringInput(Integer.toString(batchJob.getTypeId()), LOG_FORGING));
			saveBatchCron(batchJob);
			break;
		}
		case MANUAL: {
			logger.info("Saving " + validateStringInput(batchJob.getName(), LOG_FORGING) + ": " + "Manual Run" + " "
					+ validateStringInput(Integer.toString(batchJob.getTypeId()), LOG_FORGING));
			batchJob.setSchedule(BatchType.JobType.MANUAL.getName());
			saveBatchManual(batchJob);
			break;
		}
		case SCHEDULED: {
			logger.info("Saving " + validateStringInput(batchJob.getName(), LOG_FORGING) + ": " + "Scheduled Run" + " "
					+ validateStringInput(batchJob.getSchedule(), LOG_FORGING) + " "
					+ validateStringInput(Integer.toString(batchJob.getTypeId()), LOG_FORGING));

			saveBatchScheduled(batchJob);
			break;
		}
		case UNDEFINED: {
			logger.error("Unable to handle schedule for UNDEFINED job type: " + bt.getName());
			break;
		}
		default: {
			logger.error("Unable to handle schedule for unrecognized job type: " + bt.getName());
			break;
		}
		}

	}

	/**
	 * Save the batch job in the database and update the Quartz Scheduler as a
	 * cron triggered job.
	 * 
	 * @param batchJob
	 *            The batch job from which to update database and scheduler.
	 */
	public void saveBatchCron(BatchJob batchJob) {

		// Save in DB!
		batchService.updateBatchJob(batchJob);

		// Set the schedule!
		setBatchCron(batchJob);
	}

	/**
	 * Set the scheduler with the updated batch cron triggered job.
	 * 
	 * @param batchJob
	 *            The batch job from which to update scheduler.
	 */
	public void setBatchCron(BatchJob batchJob) {

		try {
			JobKey jk = new JobKey(batchJob.getName(), jobGroup);

			@SuppressWarnings("unchecked")
			List<Trigger> tL = (List<Trigger>) scheduler.getTriggersOfJob(jk);

			// If the trigger list is empty then create a new cron trigger for
			// the job.
			if (tL.isEmpty()) {

				if (!scheduler.checkExists(jk)) {
					// This is an unrecoverable error. The job key should
					// always exist.
					logger.error("setBatchCron - job key does not exist! - exiting");
					return;
				}

				CronTriggerImpl trigger = new CronTriggerImpl();

				TriggerKey tK = new TriggerKey(buildTriggerNameForJob(batchJob), triggerGroup);

				trigger.setKey(tK);

				try {
					checkJobSchedule(batchJob);

					CronExpression cronExp = new CronExpression(CronUtils.getCronExpression(batchJob.getSchedule()));

					trigger.setCronExpression(cronExp);
					trigger.setStartTime(new Date());

					trigger.setJobKey(jk);
					trigger.setJobName(batchJob.getName());

					logger.info(trigger.toString());

					Date nextFireDate = scheduler.scheduleJob(trigger);

					if (nextFireDate == null) {
						logger.error("Schedule of job was unsuccessful.");
					}
				} catch (ParseException e) {
					logger.error("Parse error with Cron Expression: " + e.getMessage());
				}

				return;
			}

			// The trigger for the job exists.
			Trigger t = tL.get(0);

			// Update the trigger depending on the type.
			if (t instanceof CronTriggerImpl) {

				CronTriggerImpl trigger = (CronTriggerImpl) t;
				TriggerKey tK = trigger.getKey();

				scheduler.pauseTrigger(tK);

				try {
					CronExpression cronExp = new CronExpression(CronUtils.getCronExpression(batchJob.getSchedule()));
					trigger.setCronExpression(cronExp);
					trigger.setStartTime(new Date());

					Date srjDate = scheduler.rescheduleJob(tK, trigger);
					if (srjDate == null) {
						logger.error("Problem scheduling job: " + validateStringInput(batchJob.getName(), LOG_FORGING));
					}
				} catch (ParseException e) {
					logger.error(
							"Parse error with Cron Expression: " + validateStringInput(e.getMessage(), LOG_FORGING));
				}

			} else if (t instanceof SimpleTriggerImpl) {

				SimpleTriggerImpl trigger = (SimpleTriggerImpl) t;
				TriggerKey tK = trigger.getKey();

				scheduler.pauseTrigger(tK);

				CronTriggerImpl newTrigger = new CronTriggerImpl();
				newTrigger.setKey(tK);

				try {
					CronExpression cronExp = new CronExpression(CronUtils.getCronExpression(batchJob.getSchedule()));

					newTrigger.setCronExpression(cronExp);
					newTrigger.setStartTime(new Date());

					Date srjDate = scheduler.rescheduleJob(tK, newTrigger);
					if (srjDate == null) {
						logger.error("Problem scheduling job: " + validateStringInput(batchJob.getName(), LOG_FORGING));
					}
				} catch (ParseException e) {
					logger.error("Parse error with Cron Expression: " + e.getMessage());
				}

			} else {
				logger.error("Unknown trigger type! - Exiting!");
				return;
			}

		} catch (SchedulerException e) {
			logger.error("Caught Exception: " + e.getMessage());
		}

	}

	public void setInternalCron(InternalJob internalJob) {
		try {
			JobKey jk = new JobKey(internalJob.getFlowJob().getName(), jobGroup);

			@SuppressWarnings("unchecked")
			List<Trigger> tL = (List<Trigger>) scheduler.getTriggersOfJob(jk);

			// If the trigger list is empty then create a new cron trigger for
			// the job.
			if (tL.isEmpty()) {

				if (!scheduler.checkExists(jk)) {
					// This is an unrecoverable error. The job key should
					// always exist.
					logger.error("setBatchCron - job key does not exist! - exiting");
					return;
				}

				if (internalJob.getTrigger() instanceof CronTrigger) {

					CronTrigger oldTrigger = (CronTrigger) internalJob.getTrigger();
					CronTriggerImpl newTrigger = new CronTriggerImpl();
					TriggerKey tK = new TriggerKey(internalJob.getFlowJob().getName() + "Trigger", triggerGroup);

					newTrigger.setKey(tK);

					try {

						newTrigger.setCronExpression(oldTrigger.getCronExpression());
						newTrigger.setStartTime(new Date());

						newTrigger.setJobKey(jk);
						newTrigger.setJobName(internalJob.getFlowJob().getName());
						newTrigger.setMisfireInstruction(CronTriggerImpl.MISFIRE_INSTRUCTION_IGNORE_MISFIRE_POLICY);

						logger.info(newTrigger.toString());

						Date nextFireDate = scheduler.scheduleJob(newTrigger);

						if (nextFireDate == null) {
							logger.error("Schedule of job was unsuccessful.");
						}
					} catch (ParseException e) {
						logger.error("Parse error with Cron Expression: " + e.getMessage());
					}
				} else if (internalJob.getTrigger() instanceof SimpleTrigger) {

					SimpleTrigger oldTrigger = (SimpleTrigger) internalJob.getTrigger();
					SimpleTriggerImpl newTrigger = new SimpleTriggerImpl();
					TriggerKey tK = new TriggerKey(internalJob.getFlowJob().getName() + "Trigger", triggerGroup);

					newTrigger.setKey(tK);

					try {
						newTrigger.setRepeatCount(oldTrigger.getRepeatCount());
						newTrigger.setRepeatInterval(oldTrigger.getRepeatInterval());
						newTrigger.setStartTime(new Date());

						newTrigger.setJobKey(jk);
						newTrigger.setJobName(internalJob.getFlowJob().getName());
						newTrigger.setMisfireInstruction(CronTriggerImpl.MISFIRE_INSTRUCTION_IGNORE_MISFIRE_POLICY);

						logger.info(newTrigger.toString());

						Date nextFireDate = scheduler.scheduleJob(newTrigger);

						if (nextFireDate == null) {
							logger.error("Schedule of job was unsuccessful.");
						}
					} catch (Exception e) {
						logger.error("Error with building Simple Trigger: " + e.getMessage());
					}
				}

				return;
			}

			// The trigger for the job exists.
			Trigger t = tL.get(0);
			if (!(t instanceof CronTriggerImpl) && !(t instanceof SimpleTriggerImpl)) {
				logger.error("Unknown trigger type! - Exiting!");
				return;
			}
			// Update to a new trigger type if necessary.
			if (internalJob.getTrigger() instanceof SimpleTriggerImpl) {
				SimpleTrigger newTrigger = (SimpleTrigger) internalJob.getTrigger();
				// new trigger must have use property elements stored as string
				// as per quartz.properties
				newTrigger.getJobDataMap().put("jobDetail", newTrigger.getJobDataMap().get("jobDetail").toString());

				TriggerKey tK = t.getKey();

				scheduler.pauseTrigger(tK);

				try {
					Date srjDate = scheduler.rescheduleJob(tK, newTrigger);
					if (srjDate == null) {
						logger.error("Problem scheduling job: "
								+ validateStringInput(internalJob.getFlowJob().getName(), LOG_FORGING));
					}
				} catch (SchedulerException e) {
					logger.error("Scheduler exception with job: " + e.getMessage());
				}
			} else if (internalJob.getTrigger() instanceof CronTriggerImpl) {
				CronTrigger newTrigger = (CronTrigger) internalJob.getTrigger();
				CronTriggerImpl trigger = (CronTriggerImpl) t;
				TriggerKey tK = trigger.getKey();

				scheduler.pauseTrigger(tK);

				try {

					trigger.setCronExpression(newTrigger.getCronExpression());
					trigger.setStartTime(new Date());

					Date srjDate = scheduler.rescheduleJob(tK, trigger);

					if (srjDate == null) {
						logger.error("Problem scheduling internal job: "
								+ validateStringInput(internalJob.getFlowJob().getName(), LOG_FORGING));
					}
				} catch (ParseException e) {
					logger.error(
							"Parse error with Cron Expression: " + validateStringInput(e.getMessage(), LOG_FORGING));
				}

			}
		} catch (SchedulerException e) {
			logger.error("Caught Exception: " + e.getMessage());
		}
	}

	/**
	 * Save the batch job in the database and update the Quartz Scheduler as a
	 * simple triggered job.
	 * 
	 * @param batchJob
	 *            The batch job from which to update database and scheduler.
	 */
	public void saveBatchScheduled(BatchJob batchJob) {

		// Save in DB!
		batchService.updateBatchJob(batchJob);

		// Set the schedule!
		setBatchScheduled(batchJob);
	}

	/**
	 * Set the scheduler with the updated batch simple triggered job.
	 * 
	 * @param batchJob
	 *            The batch job from which to update scheduler.
	 */
	public void setBatchScheduled(BatchJob batchJob) {
		try {

			JobKey jk = new JobKey(batchJob.getName(), jobGroup);

			@SuppressWarnings("unchecked")
			List<Trigger> tL = (List<Trigger>) scheduler.getTriggersOfJob(jk);

			// If trigger list is empty then we make a new trigger.
			if (tL.isEmpty()) {

				if (!scheduler.checkExists(jk)) {
					// This is an unrecoverable error. The job key should
					// always exist.
					logger.error("setBatchScheduled - job key does not exist! - exiting");
					return;
				}

				SimpleTriggerImpl trigger = new SimpleTriggerImpl();
				TriggerKey tK = new TriggerKey(buildTriggerNameForJob(batchJob), triggerGroup);

				checkJobSchedule(batchJob);

				final String delims = ":";
				String[] tokens = batchJob.getSchedule().split(delims);
				Calendar calStart = Calendar.getInstance();
				calStart.setTime(batchJob.getBatchConfig().getRangeStart());
				calStart.set(Calendar.HOUR_OF_DAY, Integer.parseInt(tokens[0]));
				calStart.set(Calendar.MINUTE, Integer.parseInt(tokens[1]));
				calStart.set(Calendar.SECOND, Integer.parseInt(tokens[2]));
				calStart.set(Calendar.MILLISECOND, 0);
				Date myStartTime = calStart.getTime();

				Calendar calEnd = Calendar.getInstance();
				calEnd.setTime(batchJob.getBatchConfig().getRangeEnd());
				calEnd.set(Calendar.HOUR_OF_DAY, Integer.parseInt(tokens[0]));
				calEnd.set(Calendar.MINUTE, Integer.parseInt(tokens[1]));
				calEnd.set(Calendar.SECOND, Integer.parseInt(tokens[2]));
				calEnd.set(Calendar.MILLISECOND, 0);
				Date myEndTime = calEnd.getTime();

				trigger.setKey(tK);
				trigger.setStartTime(myStartTime);
				trigger.setEndTime(myEndTime);
				trigger.setRepeatCount(0);
				trigger.setJobKey(jk);
				trigger.setJobName(batchJob.getName());

				trigger.setMisfireInstruction(
						SimpleTriggerImpl.MISFIRE_INSTRUCTION_RESCHEDULE_NEXT_WITH_EXISTING_COUNT);

				logger.info(trigger.toString());

				Date nextFireDate = scheduler.scheduleJob(trigger);

				if (nextFireDate == null) {
					logger.error("Problem scheduling job: " + validateStringInput(batchJob.getName(), LOG_FORGING));
				}
				return;
			}

			Trigger t = tL.get(0);
			if (t instanceof CronTriggerImpl) {
				CronTriggerImpl trigger = (CronTriggerImpl) t;
				TriggerKey tK = trigger.getKey();

				scheduler.pauseTrigger(tK);

				final String delims = ":";
				String[] tokens = batchJob.getSchedule().split(delims);
				Calendar calStart = Calendar.getInstance();
				calStart.setTime(batchJob.getBatchConfig().getRangeStart());
				calStart.set(Calendar.HOUR_OF_DAY, Integer.parseInt(tokens[0]));
				calStart.set(Calendar.MINUTE, Integer.parseInt(tokens[1]));
				calStart.set(Calendar.SECOND, Integer.parseInt(tokens[2]));
				calStart.set(Calendar.MILLISECOND, 0);

				Date myStartTime = calStart.getTime();

				Calendar calEnd = Calendar.getInstance();
				calEnd.setTime(batchJob.getBatchConfig().getRangeEnd());
				calEnd.set(Calendar.HOUR_OF_DAY, Integer.parseInt(tokens[0]));
				calEnd.set(Calendar.MINUTE, Integer.parseInt(tokens[1]));
				calEnd.set(Calendar.SECOND, Integer.parseInt(tokens[2]));
				calEnd.set(Calendar.MILLISECOND, 0);

				Date myEndTime = calEnd.getTime();

				SimpleTriggerImpl newTrigger = new SimpleTriggerImpl();
				newTrigger.setKey(tK);
				newTrigger.setStartTime(myStartTime);
				newTrigger.setEndTime(myEndTime);
				newTrigger.setRepeatCount(0);
				newTrigger.setMisfireInstruction(
						SimpleTriggerImpl.MISFIRE_INSTRUCTION_RESCHEDULE_NEXT_WITH_EXISTING_COUNT);

				Date srjDate = scheduler.rescheduleJob(tK, newTrigger);
				if (srjDate == null) {
					logger.error("Problem scheduling job: " + validateStringInput(batchJob.getName(), LOG_FORGING));
				}

			} else if (t instanceof SimpleTriggerImpl) {
				SimpleTriggerImpl trigger = (SimpleTriggerImpl) t;
				TriggerKey tK = trigger.getKey();

				scheduler.pauseTrigger(tK);

				final String delims = ":";
				String[] tokens = batchJob.getSchedule().split(delims);

				Calendar calStart = Calendar.getInstance();
				calStart.setTime(batchJob.getBatchConfig().getRangeStart());
				calStart.set(Calendar.HOUR_OF_DAY, Integer.parseInt(tokens[0]));
				calStart.set(Calendar.MINUTE, Integer.parseInt(tokens[1]));
				calStart.set(Calendar.SECOND, Integer.parseInt(tokens[2]));
				calStart.set(Calendar.MILLISECOND, 0);

				Date myStartTime = calStart.getTime();

				Calendar calEnd = Calendar.getInstance();
				calEnd.setTime(batchJob.getBatchConfig().getRangeEnd());
				calEnd.set(Calendar.HOUR_OF_DAY, Integer.parseInt(tokens[0]));
				calEnd.set(Calendar.MINUTE, Integer.parseInt(tokens[1]));
				calEnd.set(Calendar.SECOND, Integer.parseInt(tokens[2]));
				calEnd.set(Calendar.MILLISECOND, 0);

				Date myEndTime = calEnd.getTime();

				trigger.setKey(tK);
				trigger.setStartTime(myStartTime);
				trigger.setEndTime(myEndTime);
				trigger.setRepeatCount(0);

				trigger.setMisfireInstruction(
						SimpleTriggerImpl.MISFIRE_INSTRUCTION_RESCHEDULE_NEXT_WITH_EXISTING_COUNT);

				Date srjDate = scheduler.rescheduleJob(tK, trigger);
				if (srjDate == null) {
					logger.error("Problem scheduling job: " + validateStringInput(batchJob.getName(), LOG_FORGING));
				}

			} else {
				logger.error("Unknown trigger type! - Exiting");
				return;
			}

		} catch (SchedulerException e) {
			logger.error("Caught Exception: " + e.getMessage());
		}

	}

	/**
	 * Save the batch job in the database and update the Quartz Scheduler to
	 * unschedule the job trigger.
	 * 
	 * @param batchJob
	 *            The batch job from which to update database and scheduler.
	 */
	public void saveBatchManual(BatchJob batchJob) {

		// Save in DB!
		batchService.updateBatchJob(batchJob);

		// Set the schedule!
		setBatchManual(batchJob);
	}

	/**
	 * Update the scheduler to unschedule the specified batch job.
	 * 
	 * @param batchJob
	 *            The batch job from which to update scheduler.
	 */
	public void setBatchManual(BatchJob batchJob) {

		try {
			JobKey jk = new JobKey(batchJob.getName(), jobGroup);

			@SuppressWarnings("unchecked")
			List<Trigger> tL = (List<Trigger>) scheduler.getTriggersOfJob(jk);

			if (!tL.isEmpty()) {

				// Turn off trigger.
				Trigger t = tL.get(0);
				scheduler.unscheduleJob(t.getKey());
			}

		} catch (SchedulerException e) {
			logger.error("Caught Exception: " + e.getMessage());
		}
	}

	/**
	 * Check the job schedule and save a default value if invalid.
	 * 
	 * @param batchJob
	 *            The batch job to check.
	 */
	private void checkJobSchedule(BatchJob batchJob) {
		String schedule = batchJob.getSchedule();
		if ((schedule == null) || schedule.isEmpty()) {
			schedule = "23:00:00";
			batchJob.setSchedule(schedule);
			logger.warn("Job '" + batchJob.getName() + "' schedule was invalid so using default value: " + schedule);
			batchService.updateBatchJob(batchJob);
		}
	}

	/**
	 * Forcefully run the specified batch job by it's ID.
	 * 
	 * @param id
	 *            The ID of the batch job to run.
	 */
	public void runJob(int id) {

		// Look up job based on id!
		BatchJob batchJob = batchService.getBatchJobById(id);

		if (batchJob == null) {
			logger.error("Batch job " + id + " does not exist.");
		} else {

			// Forcefully run the job specified by the id.
			JobKey jk = new JobKey(batchJob.getName(), jobGroup);

			try {
				if (scheduler.checkExists(jk)) {
					scheduler.triggerJob(jk);
				} else {
					logger.error("Job key does not exist.");
				}
			} catch (SchedulerException e) {
				logger.error("runJob exception: " + e.getMessage());
			}
		}
	}

	/**
	 * Initialize the Quartz Scheduler upon deployment.
	 */
	public void initNotificationScheduler() {

		// Create jobs from the database.
		List<BatchJob> bjL = batchService.getBatchJobList();

		// Save a list of the job names to check if any old jobs should be
		// removed.
		List<String> nameL = new ArrayList<>();

		for (BatchJob bj : bjL) {
			try {

				// Reset status for job to not running if was found to have
				// running state.
				// This would indicate the job was unexpected interrupted or
				// ended in an earlier run.
				if (batchService.setNotRunningById(bj)) {
					logger.warn("Last status for batch job (" + validateStringInput(bj.getName(), LOG_FORGING)
							+ ") set to not running");
				}

				Class<?> clazz = Class.forName("gov.va.cpss.job." + bj.getName());

				if (clazz.newInstance() instanceof QuartzJobBean) {

					JobDataMap map = new JobDataMap();
					map.put("name", bj.getName());

					@SuppressWarnings("unchecked")
					JobDetail job = JobBuilder.newJob((Class<? extends Job>) clazz).withIdentity(bj.getName(), jobGroup)
							.usingJobData(map).storeDurably().build();
					initializeTrigger(bj, job);

					nameL.add(bj.getName());

				} else {
					logger.error("Invalid Job Class: " + validateStringInput(bj.getName(), LOG_FORGING));
				}

			} catch (ClassNotFoundException e) {
				logger.error("ClassNotFoundException when initializing job: "
						+ validateStringInput(bj.getName(), LOG_FORGING) + ", e: " + e.getMessage());
			} catch (InstantiationException e) {
				logger.error("InstantiationException when initializing job: "
						+ validateStringInput(bj.getName(), LOG_FORGING) + ", e: " + e.getMessage());
			} catch (IllegalAccessException e) {
				logger.error("IllegalAccessException when initializing job: "
						+ validateStringInput(bj.getName(), LOG_FORGING) + ", e: " + e.getMessage());
			}

		}

		for (InternalJob internalJob : internalGroup) {
			Class<?> clazz = internalJob.getJobDetail().getJobClass();

			JobDataMap map = new JobDataMap();
			map.put("name", internalJob.getFlowJob().getName());

			@SuppressWarnings("unchecked")
			JobDetail job = JobBuilder.newJob((Class<? extends Job>) clazz)
					.withIdentity(internalJob.getFlowJob().getName(), jobGroup).usingJobData(map).storeDurably()
					.build();
			// Run the set batch cron with some specific thing that does not
			// need BatchJob
			if (createJob(job)) {
				setInternalCron(internalJob);
			}
			nameL.add(internalJob.getFlowJob().getName());
		}

		if (!deleteNonexistantJobs(nameL)) {
			logger.warn("Problem deleting nonexistant jobs");
		}
	}

	/**
	 * Initialize Quartz Job Triggers for the specified batch job and job
	 * details.
	 * 
	 * @param bj
	 *            The batch job for which to update the scheduler.
	 * @param job
	 *            The Quartz job details for the batch job.
	 */
	private void initializeTrigger(BatchJob bj, JobDetail job) {

		logger.info("Initializing trigger for job: " + job.getKey().getName());

		// Handle trigger initialization based on type.
		switch (bj.getBatchType().getJobType()) {
		case DAILY: {
			logger.info("DAILY");
			if (createJob(job)) {
				setBatchCron(bj);
			}
			break;
		}
		case MANUAL: {
			logger.info("MANUAL");
			if (createJob(job)) {
				setBatchManual(bj);
			}
			break;
		}
		case SCHEDULED: {
			logger.info("SCHEDULED");
			if (createJob(job)) {
				setBatchScheduled(bj);
			}
			break;
		}
		case UNDEFINED: {
			logger.error("Unable to initialize trigger for UNDEFINED job type: "
					+ validateStringInput(bj.getBatchType().getName(), LOG_FORGING));
			break;
		}
		default: {
			logger.error("Unable to initialize trigger for unrecognized job type: "
					+ validateStringInput(bj.getBatchType().getName(), LOG_FORGING));
			break;
		}
		}

	}

	/**
	 * Create the Quartz Job in the scheduler if it does not already exist.
	 * 
	 * @param job
	 *            The job to create.
	 * @return Boolean flag that indicates if successfully created job or if the
	 *         job already exists.
	 */
	private boolean createJob(JobDetail job) {
		boolean success = false;
		try {
			if (!scheduler.checkExists(job.getKey())) {
				scheduler.addJob(job, false);
				logger.info("Job added to scheduler: " + job.getKey().getName());
			} else {
				logger.info("Job exists in scheduler: " + job.getKey().getName());
			}
			success = true;
		} catch (SchedulerException e) {
			logger.error("Unable to add job to scheduler: " + job.getKey().getName() + ", e: " + e.getMessage());
		}
		return success;
	}

	/**
	 * Delete any jobs for the group that is not in the cpss batch job table.
	 * 
	 * @param nameL
	 *            The list of batch jobs in the cpss batch job table.
	 */
	private boolean deleteNonexistantJobs(final List<String> nameL) {
		boolean result = false;
		GroupMatcher<JobKey> matcher = GroupMatcher.groupEquals(jobGroup);
		try {
			Set<JobKey> jobKeyS = scheduler.getJobKeys(matcher);
			for (JobKey jobKey : jobKeyS) {
				// If job not found then delete it.
				if (!nameL.contains(jobKey.getName())) {
					logger.info("Deleting old scheduled job that was not in the batch job database table: "
							+ jobKey.getName());
					scheduler.deleteJob(jobKey);
				}
			}
			result = true;
		} catch (SchedulerException e) {
			logger.error("Error deleting nonexistant jobs: " + e.getMessage());
		}
		return result;
	}

}
