package gov.va.cpss.service;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;

import java.sql.Timestamp;
import java.util.Calendar;

import org.apache.log4j.Logger;
import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.item.ExecutionContext;

import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.dao.SiteBalDAO;
import gov.va.cpss.dao.SiteBalRecDAO;
import gov.va.cpss.job.loadbill.LoadBillProcessingConstants;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.bal.SiteBalRec;
import gov.va.cpss.model.bal.SiteBalance;

public class LoadBillService {

	private static final Logger logger = Logger.getLogger(LoadBillService.class.getCanonicalName());

	private ProcessStatusDAO processStatusDAO;
	private SiteBalDAO siteBalDao;
	private SiteBalRecDAO siteBalRecDao;
	
	public LoadBillService() {
	}
	
	public ProcessStatusDAO getProcessStatusDAO() {
		return processStatusDAO;
	}

	public void setProcessStatusDAO(ProcessStatusDAO processStatusDAO) {
		this.processStatusDAO = processStatusDAO;
	}
	
	public SiteBalDAO getSiteBalDao() {
		return siteBalDao;
	}

	public void setSiteBalDao(SiteBalDAO siteBalDao) {
		this.siteBalDao = siteBalDao;
	}

	public SiteBalRecDAO getSiteBalRecDao() {
		return siteBalRecDao;
	}

	public void setSiteBalRecDao(SiteBalRecDAO siteBalRecDao) {
		this.siteBalRecDao = siteBalRecDao;
	}
	
	public SiteBalRec startProcessLoadBillDataJob(final int batchRunId, final String fileName) {

		logger.info("Starting Load Bill Data Job");

		SiteBalRec siteBalRec = null;

		// Hard-coded for simplicity
		final Integer processStatus = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.INITIAL);
		
		if (processStatus != null) {
			Timestamp receivedDate = new Timestamp(Calendar.getInstance().getTime().getTime());

			siteBalRec = new SiteBalRec(batchRunId, processStatus, receivedDate, fileName);

			long receivedId = siteBalRecDao.save(siteBalRec);
			siteBalRec.setId(receivedId);
		} else {
			logger.error("Unable to obtain status mapping for: " + ProcessStatus.Status.INITIAL);
		}

		return siteBalRec;
	}

	public boolean endProcessLoadBillDataJob(JobExecution execution, SiteBalRec siteBalRec) {

		boolean jobSuccess = false;
		if (execution == null) {
			logger.error("Job ended with null execution.");
			errorProcessLoadBillDataJob(siteBalRec);

		} else if (execution.getExitStatus() != ExitStatus.COMPLETED) {
			if (execution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {
				logger.error("Job ended with failure: " + execution.getExecutionContext().getString(JOB_FAILURE_KEY));
			} else {
				logger.error("Job ended with unknown error: " + execution.getExitStatus());
			}
			errorProcessLoadBillDataJob(siteBalRec);
		} else {
			jobSuccess = successProcessLoadBillDataJob(execution, siteBalRec);
			// If unable to successfully save then treat as an error.
			if (!jobSuccess) {
				logger.error("Job was unable to save succesfully.");
				errorProcessLoadBillDataJob(siteBalRec);
			}
		}

		logger.info("End Load Bill Data Job");

		return jobSuccess;
	}
	
	private void errorProcessLoadBillDataJob(SiteBalRec siteBalRec) {

		final Integer errorStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.ERROR);
		if (errorStatusId == null) {
			logger.error("Unable to obtain status mapping for: " + ProcessStatus.Status.ERROR);
		} else {
			siteBalRec.setProcessStatus(errorStatusId);
			siteBalRecDao.updateResults(siteBalRec);
		}
	}
	
	private boolean successProcessLoadBillDataJob(JobExecution execution, SiteBalRec siteBalRec) {

		boolean completeSuccess = true;
		ExecutionContext context = execution.getExecutionContext();
		
		if (context.containsKey(LoadBillProcessingConstants.TOTAL_PATIENT_COUNT_KEY)) {
			siteBalRec.setTotNumPatient(context.getInt(LoadBillProcessingConstants.TOTAL_PATIENT_COUNT_KEY));
		} else {
			logger.error("Unable to obtain total patient count from process results");
			completeSuccess = false;
		}
		
		if (context.containsKey(LoadBillProcessingConstants.TOTAL_FACILITY_COUNT_KEY)) {
			siteBalRec.setTotNumFacility(context.getInt(LoadBillProcessingConstants.TOTAL_FACILITY_COUNT_KEY));
		}
		else {
			logger.error("Unable to obtain total facility count from process results");
			completeSuccess = false;
		}

		if (completeSuccess) {
			final Integer successStatusId = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.SUCCESS);
			if (successStatusId == null) {
				logger.error("Unable to obtain status mapping for: " + ProcessStatus.Status.SUCCESS);
				completeSuccess = false;
			} else {
				siteBalRec.setProcessStatus(successStatusId);
				siteBalRecDao.updateResults(siteBalRec);
			}
			
			logger.info("Success in completing the process Load Bill Data Job");
		}

		return completeSuccess;
	}
	
	/**
	 * Insert the SiteBal record into the database.
	 * @param siteBalance the SiteBal record to save
	 */
	public void saveSiteBalance(SiteBalance siteBalance) {
		siteBalDao.save(siteBalance);
	}
	
	/**
	 * Return the specified SiteBalance by acctNum (icn number) and stationNum, null if the record does not exist.
	 */
	public SiteBalance getSiteBalanceByAcntNumAndStationNum(final long dfn, final String stationNum) {
		return siteBalDao.getByAcctNumAndStationNum(dfn, stationNum);
	}		
}
